/***************************************************************************
 *	BGraph.cpp	Graph plotter
 *				T.Barnaby,	BEAM Ltd,	2007-05-31
 ***************************************************************************
 */
#include <BGraph.h>
#include <BString.h>
#include <qpainter.h>
#include <math.h>

#define BDEBUG	0

#if BDEBUG
#define	dprintf(fmt, a...)	printf(fmt, ##a);
#else
#define	dprintf(fmt, a...)
#endif

WAxisLabel::WAxisLabel(QWidget* parent, Bool yaxis) : QWidget(parent){
	oyaxis = yaxis;
	omin = 0;
	omax = 0;
}

void WAxisLabel::setLabel(BString label){
	olabel = label;
}

void WAxisLabel::setRange(BFloat min, BFloat max){
	omin = min;
	omax = max;
}

void WAxisLabel::paintEvent(QPaintEvent* event){
	QPainter	p;
	int		w, h;
	BString		str;
	
	w = width();
	h = height();

#ifdef ZAP_QT3
	oxLabelBox->erase();
#endif

	if(oyaxis){
		p.begin(this);
       		p.setPen(Qt::blue);

		str.printf("%12.6g", omax);
		p.drawLine(w - 5, 2, w, 2);
		p.drawText(QRect(0, 0, w, 16), Qt::AlignLeft, str.retStr());

		str.printf("%12.6g", omin);
		p.drawLine(w - 5, h - 3, w, h - 3);
		p.drawText(QRect(0, h - 16, w, h), Qt::AlignLeft, str.retStr());

		p.drawText(QRect(0, int(h * 0.5 - 8), w, int(h * 0.5 + 8)), Qt::AlignLeft, olabel.retStr());
		p.end();
	}
	else {
		p.begin(this);
       		p.setPen(Qt::blue);

		str.printf("%-12.6g", omin);
		p.drawLine(2, 0, 2, 6);
		p.drawText(QRect(8, 0, w - 16, h), Qt::AlignLeft, str.retStr());

		str.printf("%12.6g", omax);
		p.drawLine(w - 3, 0, w - 3, 6);
		p.drawText(QRect(8, 0, w - 16, h), Qt::AlignRight, str.retStr());

		p.drawText(QRect(0, 0, w, h), Qt::AlignCenter, olabel.retStr());
		p.end();
	}
}


BGraphPlot::BGraphPlot(QWidget* parent) : QWidget(parent){
	oxStep = 0;
	oyStep = 0;
	oxRangeMin = 0.0;
	oxRangeMax = 0.0;
	oyRangeMin = 0.0;
	oyRangeMax = 0.0;
	oyMinValue = 0;
	oyMaxValue = 0;
	oxMinValue = 0;
	oxMaxValue = 0;
}

void BGraphPlot::setGrid(int xStep, int yStep){
	oxStep = xStep;
	oyStep = yStep;
}

void BGraphPlot::setXRange(double min, double max){
	oxRangeMin = min;
	oxRangeMax = max;
	plotData();
	update();
}

void BGraphPlot::setYRange(double min, double max){
	oyMinValue = oyRangeMin = min;
	oyMaxValue = oyRangeMax = max;
	plotData();
	update();
}

void BGraphPlot::setData(BArray<float>& data){
	odata = data;
	plotData();
	update();
}

void BGraphPlot::getXMinMax(double& min, double& max){
	min = oxMinValue;
	max = oxMaxValue;
}

void BGraphPlot::plotData(){
	unsigned int	x;
	unsigned int	i;
	double		xScale = 1.0;
	double		xOffset = 1.0;
	double		yScale = 1.0;
	double		yOffset = 0;
	unsigned int	xMin;
	unsigned int	xMax;
	
	dprintf("BGraphPlot::plotData\n");

	if(odata.size() == 0){
		opoints.clear();
		return;
	}

	if((oyRangeMin == 0.0) && (oyRangeMax == 0.0)){
		oyMinValue = oyMaxValue = odata[0];
		for(x = 0; x < odata.size(); x++){
			if(odata[x] > oyMaxValue)
				oyMaxValue = odata[x];
			if(odata[x] < oyMinValue)
				oyMinValue = odata[x];
		}
		
		yScale = (height() - 4) / (oyMaxValue - oyMinValue);
		yOffset = (height() - 2) + oyMinValue * yScale;
		dprintf("YMin: %f YMax: %f\n", oyMinValue, oyMaxValue);
		dprintf("Height: %d Offset: %f Scale: %f\n", height(), yOffset, yScale);
	}
	else {
		yScale =  (height() - 4) / (oyRangeMax - oyRangeMin);
		yOffset = (height() - 2) + oyRangeMin * yScale;
		dprintf("YMin: %f YMax: %f\n", oyRangeMin, oyRangeMax);
		dprintf("Height: %d Offset: %f Scale: %f\n", height(), yOffset, yScale);
	}
	
	if((oxRangeMin == 0.0) && (oxRangeMax == 0.0)){
		xOffset = 0;
		xMin = 0;
		xMax = odata.size() - 1;
		xScale = double(width()) / (xMax - xMin);
	}
	else {
		xOffset = int(oxRangeMin);
		xMin = int(oxRangeMin);
		xMax = int(oxRangeMax);
		xScale = double(width()) / (xMax - xMin);
	}
	oxMinValue = xMin;
	oxMaxValue = xMax;
	dprintf("XMin: %d XMax: %d\n", xMin, xMax);
	dprintf("Width: %d Offset: %f Scale: %f\n", width(), xOffset, xScale);
	
	opoints.resize(xMax - xMin);

	for(i = 0; i < (xMax - xMin); i++){
		x = int(i * xScale);
		if(((xOffset + i) >= 0) && ((xOffset + i) < odata.size()))
			opoints[i] = QPoint(x, int(yOffset + yScale * -odata[int(xOffset) + i]));
		else
			opoints[i] = QPoint(0, 0);
	}
}


void BGraphPlot::resizeEvent(QResizeEvent* event){
	dprintf("BGraphPlot::ResizeEvent: %d,%d\n", width(), height());
	plotData();
}

void BGraphPlot::paintEvent(QPaintEvent* event){
	QPainter	p;
	double		x;
	double		y;
	double		xs;
	double		ys;

	dprintf("BGraphPlot::paintEvent: grid: %d,%d\n", oxStep, oyStep);

	p.begin(this);
        p.setPen("black");
	p.setPen(Qt::DotLine);

	if(oxStep){
		xs = double(width()) / oxStep;
		for(x = xs; x < double(width()); x += xs){
			p.drawLine(int(x), 0, int(x), height());
		}
	}
	if(oyStep){
		ys = double(height()) / oyStep;
		for(y = ys; y < double(height()); y += ys){
			p.drawLine(0, int(y), width(), int(y));
		}
	}

        p.setPen(Qt::blue);
	p.drawPolyline(opoints.data(), opoints.size());
	
	p.end();
}


BGraph::BGraph(QWidget* parent) : BGrid(2, Qt::Horizontal, parent){
	blayoutSetSpacing(layout(), 0);
	setFrameShape(QFrame::StyledPanel);
	setFrameShadow(QFrame::Raised);

	// Display labels etc
	oyLabelBox = new WAxisLabel(this, 1);
	oyLabelBox->setFixedWidth(100);

	oplotFrame = new BHBox(this);
	blayoutSetSpacing(oplotFrame->layout(), 0);
	oplotFrame->setFrameShape(QFrame::StyledPanel);
	oplotFrame->setFrameShadow(QFrame::Raised);

	// Create main plot widget
	oplot = new BGraphPlot(oplotFrame);
	oplot->show();

	new QWidget(this);
	oxLabelBox = new WAxisLabel(this, 0);
	oxLabelBox->setFixedHeight(0);
}

BGraph::~BGraph(){
}

void BGraph::setGrid(int xStep, int yStep){
	oplot->setGrid(xStep, yStep);
}

void BGraph::setXLabel(BString label){
	oxLabelBox->setLabel(label);
	if(label != "")
		oxLabelBox->setFixedHeight(16);
	else
		oxLabelBox->setFixedHeight(0);
}

void BGraph::setYLabel(BString label){
	oyLabelBox->setLabel(label);
}

void BGraph::setXRange(double min, double max){
	oplot->setXRange(min, max);
}

void BGraph::setYRange(double min, double max){
	oplot->setYRange(min, max);
}

void BGraph::getXMinMax(double& min, double& max){
	oplot->getXMinMax(min, max);
}

void BGraph::setYLabelWidth(int width){
	oyLabelBox->setFixedWidth(width);
}

void BGraph::setData(BArray<float> data){
	odata = data;
	oplot->setData(data);
}

void BGraph::getData(BArray<float>& data){
	data = odata;
}

void BGraph::clear(){
	dprintf("Clear\n");
#ifdef ZAP_QT3
	oplot->erase();
#endif
	odata.clear();
}


#if DEBUG_MAIN

#include <stdio.h>
#include <syslog.h>
#include <unistd.h>
#include <qapp.h>
#include <qvbox.h>
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qmainwindow.h>
#include <qframe.h>
#include <qdatetimeedit.h>
#include <math.h>

void usage(){
	fprintf(stderr, "Usage: testChart\n");
}

int main(int argc, char** argv){
	int		a;
	QApplication*	app;
	QMainWindow*	mw;
	BVBox*		v;
	BGraph*		p1;
	BGraph*		p2;
	BGraph*		p3;
	BGraph*		p4;
	BArray<float>	data;
	unsigned int	x;
	
	for(a = 1; a < argc; a++){
		if(argv[a][0] == '-'){
			switch(argv[a][1]){
			case 'd':
				if(argc <= (a + 1)){
					usage();
					return 1;
				}
//				setDebug(strtol(argv[++a], 0, 0));
				break;
			// QT Options
			case 'g':
				break;
			default:	usage();	return 1;
			}
		}
	}

	// Create Gui control class
	app = new QApplication(argc, argv);
	mw = new QMainWindow();
	app->setMainWidget(mw);
	mw->show();

	v = new BVBox(mw);
	mw->setCentralWidget(v);
	v->show();
	
	data.resize(100);
	for(x = 0; x < data.size(); x++){
		data[x] = sin((4.0 * 2.0 * 3.14 * x) / data.size());
	}

#ifdef ZAP
	p1 = new BGraph(v);
	p1->show();

	p1->setYLabel("FRef");	
//	p1->setXLabel("Samples");	
	p1->setData(data);

	p2 = new BGraph(v);
	p2->show();

	p2->setYRange(-0.555555, 2);	
//	p2->setYRange(-1.0, 1.0);	
	p2->setYLabel("FRef");	
	p2->setXLabel("Samples");	
	p2->setData(data);
#endif

	p3 = new BGraph(v);
	p3->show();

	p3->setYLabelWidth(100);
	p3->setGrid(8, 4);
	p3->setXRange(25, 75);	
	p3->setYLabel("FRef");	
	p3->setXLabel("Samples");	
	p3->setData(data);

#ifdef ZAP
	p4 = new BGraph(v);
	p4->show();

	for(x = 0; x < data.size(); x++){
		data[x] = 30040000.0 + 10000.0 * sin((4.0 * 2.0 * 3.14 * x) / data.size());
	}

	p4->setYLabelWidth(100);
	p4->setGrid(4, 4);
	p4->setYLabel("FRef");	
	p4->setXLabel("Samples");	
	p4->setData(data);
#endif
	app->exec();
	
	return 0;
}
#endif
