/*******************************************************************************
 *	TimeStamp.h	TimeStamp classes
 *			T.Barnaby,	BEAM Ltd,	2005-10-20
 *******************************************************************************
 */
#include <BTimeStampMs.h>
#include <sys/time.h>

int BTimeStampMs::isLeap(int year){
	return ((year) % 4 == 0 && ((year) % 100 != 0 || (year) % 400 == 0));
}

static int mon_yday[2][13] = {
	/* Normal years.  */
	{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
	/* Leap years.  */
	{ 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }
};


BTimeStampMs::BTimeStampMs(BString str){
	if(str == "")
		clear();
	else
		setString(str);
}

BTimeStampMs::~BTimeStampMs(){
}

void BTimeStampMs::clear(){
	memset(this, 0, sizeof(*this));
}

void BTimeStampMs::setNow(){
	struct timeval	tv;
	struct tm	tm;
	
	gettimeofday(&tv, 0);
	gmtime_r(&tv.tv_sec, &tm);
	
	year = 1900 + tm.tm_year;
	yday = tm.tm_yday;
	hour = tm.tm_hour;
	minute = tm.tm_min;
	second = tm.tm_sec;
	milliSecond = tv.tv_usec / 1000;
	sampleNumber = 0;
}

BString BTimeStampMs::getString(BString separator){
	BString		s;
	int		mon = 0;
	int		day = 0;
	
	for(mon = 11; yday < mon_yday[isLeap(year)][mon]; --mon);
	day = yday - mon_yday[isLeap(year)][mon];

	s.printf("%04d-%02d-%02d%s%02d:%02d:%02d.%03d", year, mon + 1, day + 1, separator.retStr(), hour, minute, second, milliSecond);	
	return s;
}

BString BTimeStampMs::getStringNoMs(BString separator){
	BString		s;
	int		mon = 0;
	int		day = 0;
	
	for(mon = 11; yday < mon_yday[isLeap(year)][mon]; --mon);
	day = yday - mon_yday[isLeap(year)][mon];

	s.printf("%04d-%02d-%02d%s%02d:%02d:%02d", year, mon + 1, day + 1, separator.retStr(), hour, minute, second);	
	return s;
}

BString BTimeStampMs::getDurationString(BString separator){
	BString		s;
	int		mon = 0;
	int		day = 0;
	
	for(mon = 11; yday < mon_yday[isLeap(year)][mon]; --mon);
	day = yday - mon_yday[isLeap(year)][mon];

	s.printf("%04d-%02d-%02d%s%02d:%02d:%02d.%03d", year, mon, day, separator.retStr(), hour, minute, second, milliSecond);	
	return s;
}

BString BTimeStampMs::getDurationStringNoMs(BString separator){
	BString		s;
	int		mon = 0;
	int		day = 0;
	
	for(mon = 11; yday < mon_yday[isLeap(year)][mon]; --mon);
	day = yday - mon_yday[isLeap(year)][mon];

	s.printf("%04d-%02d-%02d%s%02d:%02d:%02d", year, mon, day, separator.retStr(), hour, minute, second);
	return s;
}

BError BTimeStampMs::setString(BString dateTime){
	BError		err;
	int		y = 0;
	int		mon = 0;
	int		day = 0;
	int		h = 0;
	int		m = 0;
	int		s = 0;
	int		ms = 0;
	int		n;

	clear();
	n = sscanf(dateTime, "%04u-%02u-%02u%*[ T]%02u:%02u:%02u.%03u", &y, &mon, &day, &h, &m, &s, &ms);
	if((n != 3) && (n != 6) && (n != 7)){
		err.set(ErrorMisc, "Time format error\n");
	}
	else {
		year = y;
		yday = mon_yday[isLeap(year)][mon-1] + day - 1;
		hour = h;
		minute = m;
		second = s;
		milliSecond = ms;
	}
	return err;
}

BError BTimeStampMs::setDurationString(BString dateTime){
	BError		err;
	int		y = 0;
	int		mon = 0;
	int		day = 0;
	int		h = 0;
	int		m = 0;
	int		s = 0;
	int		ms = 0;
	int		n;

	clear();
	n = sscanf(dateTime, "%04u-%02u-%02u%*[ T]%02u:%02u:%02u.%03u", &y, &mon, &day, &h, &m, &s, &ms);
	if((n != 3) && (n != 6) && (n != 7)){
		err.set(ErrorMisc, "Time format error\n");
	}
	else {
		year = y;
		yday = mon_yday[isLeap(year)][mon] + day;
		hour = h;
		minute = m;
		second = s;
		milliSecond = ms;
	}
	return err;
}

BString BTimeStampMs::getStringRaw(){
	BString	s;
	
	s.printf("%d %d %d:%d:%d.%d %d", year, yday, hour, minute, second, milliSecond, sampleNumber);
	return s;
}

void BTimeStampMs::getDate(int& year, int& mon, int& day){
	year = this->year;
	for(mon = 11; yday < mon_yday[isLeap(year)][mon]; --mon);
	day = yday - mon_yday[isLeap(year)][mon];
}

BTimeStampMs& BTimeStampMs::addMilliSeconds(int milliSeconds){
	addSeconds(milliSeconds / 1000);
	milliSecond += (milliSeconds % 1000);
	if(milliSecond >= 1000){
		if(milliSeconds >= 0){
			milliSecond -= 1000;
			addSeconds(1);
		}
		else {
			milliSecond += 1000;
			addSeconds(-1);
		}
	}
	return *this;
}

BTimeStampMs& BTimeStampMs::subMilliSeconds(int milliSeconds){
	return addMilliSeconds(-milliSeconds);
}

BTimeStampMs& BTimeStampMs::addSeconds(int seconds){
	int	nyday = isLeap(year) ? 366 : 365;
	
	if(seconds >= 0){
		second += seconds % 60;
		if(second >= 60){
			minute++;
			second -= 60;
		}

		minute += (seconds / 60) % 60;
		if(minute >= 60){
			hour++;
			minute -= 60;
		}

		hour += (seconds / (60 * 60)) % 24;
		if(hour >= 24){
			yday++;
			hour -= 24;
		}

		yday += (seconds / (24 * 60 * 60)) % nyday;
		if(yday >= nyday){
			year++;
			yday -= nyday;
		}

		if(second >= 60){
			printf("BTimeStampMs::addSeconds: Error: Seconds: %d\n", seconds);
		}
	}
	else {
		seconds = -seconds;
		second -= seconds % 60;
		if(second >= 60){
			minute--;
			second += 60;
		}

		minute -= (seconds / 60) % 60;
		if(minute >= 60){
			hour--;
			minute += 60;
		}

		hour -= (seconds / (60 * 60)) % 24;
		if(hour >= 24){
			yday--;
			hour += 24;
		}

		yday -= (seconds / (24 * 60 * 60)) % nyday;
		if(yday >= nyday){
			year--;
			yday += isLeap(year) ? 366 : 365;
		}
		if(second >= 60){
			printf("BTimeStampMs::subSeconds: Error: Seconds: %d\n", seconds);
		}
	}
	return *this;
}

BTimeStampMs& BTimeStampMs::subSeconds(int seconds){
	return addSeconds(-seconds);
}

uint32_t BTimeStampMs::getYearSeconds(){
	uint32_t	s = 0;
	
	s += second;
	s += (60 * minute);
	s += (60 * 60 * hour);
	s += (60 * 60 * 24 * yday);

	return s;
}

uint64_t BTimeStampMs::getYearMilliSeconds(){
	uint64_t	ms = 0;
	
	ms += milliSecond;
	ms += (1000LLU * second);
	ms += (1000LLU * 60 * minute);
	ms += (1000LLU * 60 * 60 * hour);
	ms += (1000LLU * 60 * 60 * 24 * yday);

	return ms;	
}

int BTimeStampMs::compare(const BTimeStampMs& timeStamp){
	if(year > timeStamp.year)
		return 1;
	else if(year < timeStamp.year)
		return -1;

	else if(yday > timeStamp.yday)
		return 1;
	else if(yday < timeStamp.yday)
		return -1;

	else if(hour > timeStamp.hour)
		return 1;
	else if(hour < timeStamp.hour)
		return -1;

	else if(minute > timeStamp.minute)
		return 1;
	else if(minute < timeStamp.minute)
		return -1;

	else if(second > timeStamp.second)
		return 1;
	else if(second < timeStamp.second)
		return -1;

	else if(milliSecond > timeStamp.milliSecond)
		return 1;
	else if(milliSecond < timeStamp.milliSecond)
		return -1;

	else
		return 0;
}

BUInt64 BTimeStampMs::difference(BTimeStampMs t2, BTimeStampMs t1){
	int	y;
	BUInt64	t = 0;

	for(y = t1.year; y < t2.year; y++){
		if(isLeap(y))
			t = t + (366 * (24 * 3600) * 1000ULL);
		else
			t = t + (365 * (24 * 3600) * 1000ULL);
	}
	
	t = t + t2.getYearMilliSeconds() - t1.getYearMilliSeconds();

	return t;
}

