/*******************************************************************************
 *	BTime.cpp	BTime class
 *	T.Barnaby,	Beam Ltd,	2012-11-12
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 */
#include <BTime.h>

static inline bool yearIsLeap(BUInt16 year){
	return ((year % 4) == 0 && ((year % 100) != 0 || (year % 400) == 0));
}

static inline BUInt16 yearDays(BUInt16 year){
	return (365 + yearIsLeap(year));
}

static BUInt16 monDays[2][13] = {
	{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },	// Normal years.
	{ 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }	// Leap years
};


BTime::BTime(BUInt32 t){
	otime = t;
}

void BTime::set(BUInt32 seconds){
	otime = seconds;
}

void BTime::set(BUInt year, BUInt month, BUInt day, BUInt hour, BUInt minute, BUInt second){
	BUInt32	t = 0;

	t += ((((year - 1970) * 365) + ((year - 1969) / 4) - ((year - 1901) / 100) + ((year - 1601) / 400)) * 86400);
	t += monDays[yearIsLeap(year)][month - 1] * 86400;
	t += (day-1) * 86400;
	t += hour * 3600;
	t += minute * 60;
	t += second;
	otime = t;
}

void BTime::setYearDay(BUInt year, BUInt yearDay, BUInt hour, BUInt minute, BUInt second){
	BUInt32	t = 0;

	t += ((((year - 1970) * 365) + ((year - 1969) / 4) - ((year - 1901) / 100) + ((year - 1601) / 400)) * 86400);
	t += yearDay * 86400;
	t += hour * 3600;
	t += minute * 60;
	t += second;
	otime = t;
}

void BTime::getDate(BUInt& year, BUInt& month, BUInt& day) const{
	BUInt32	d = otime / 86400;

	year = 1970;
	while(d >= yearDays(year)){
		d -= yearDays(year);
		year++;
	}
	
	month = 1;
	while(d >= (monDays[yearIsLeap(year)][month])){
		month++;
	}
	d -= monDays[yearIsLeap(year)][month-1];
	
	day = d + 1;
}

void BTime::getTime(BUInt& hour, BUInt& minute, BUInt& second) const{
	BUInt32	t = otime;

	second = t % 60;
	t /= 60;

	minute = t % 60;
	t /= 60;

	hour = t % 24;
}

BUInt32 BTime::getSeconds() const{
	return otime;
}

void BTime::addSeconds(int seconds){
	otime += seconds;
}

int BTime::isLeapYear(){
	BUInt	y, m, d;
	
	getDate(y, m, d);
	return yearIsLeap(y);
}

BString BTime::getString(BString format) const {
	BString		str;
	BUInt		year, mon, day;
	BUInt		h, m, s;

	getDate(year, mon, day);
	getTime(h, m, s);

	if(format == "isoSpace")
		str.printf("%04d-%02d-%02d %02d:%02d:%02d", year, mon, day, h, m, s);
	else if(format == "isoUnderscore")
		str.printf("%04d-%02d-%02d_%02d:%02d:%02d", year, mon, day, h, m, s);
	else if(format == "isoDate")
		str.printf("%04d-%02d-%02d", year, mon, day);
	else if(format == "isoTime")
		str.printf("%02d:%02d:%02d", h, m, s);
	else
		str.printf("%04d-%02d-%02dT%02d:%02d:%02d", year, mon, day, h, m, s);

	return str;
}

BError BTime::setString(const BString dateTime){
	BError		err;
	BUInt		year = 0;
	BUInt		mon = 0;
	BUInt		day = 0;
	BUInt		h = 0;
	BUInt		m = 0;
	BUInt		s = 0;
	int		n;
	int		yearDayFormat = 0;

	if(dateTime == "clear")
		year = 0;
	if(dateTime == "first")
		year = 1;
	else {
		n = sscanf(dateTime, "%04u-%02u-%02u%*[ T]%02u:%02u:%02u", &year, &mon, &day, &h, &m, &s);
		if((n != 3) && (n != 6)){
			n = sscanf(dateTime, "%02u/%02u/%04u%*[ T]%02u:%02u:%02u", &day, &mon, &year, &h, &m, &s);
			if((n != 3) && (n != 6)){
				n = sscanf(dateTime, "%02u:%02u:%02u", &h, &m, &s);
				if(n != 3){
					n = sscanf(dateTime, "%04u%3u%*[ T]%02u:%02u:%02u", &year, &day, &h, &m, &s);
					if((n != 5)){
						err.set(ErrorMisc, "Time format error\n");
					}
					else{
						yearDayFormat = 1;
					}
				}
			}
		}

		if(!err){
			if(!yearDayFormat){
				if((mon < 1) || (mon > 12))
					return err.set(ErrorMisc, "Month value out of range");
				if((day < 1) || (day > 31))
					return err.set(ErrorMisc, BString("Day value out of range: ") + day);
			}
			if((h < 0) || (h > 23))
				return err.set(ErrorMisc, "Hour value out of range");
			if((m < 0) || (m > 59))
				return err.set(ErrorMisc, "Minute value out of range");
			if((s < 0) || (s > 59))
				return err.set(ErrorMisc, "Second value out of range");

			if(yearDayFormat){
				setYearDay(year, day - 1, h, m, s);
			}
			else {
				set(year, mon, day, h, m, s);
			}
		}
	}
	return err;
}
