/*******************************************************************************
 *	BFifoCirc.h		FIFO Buffer Class
 *			T.Barnaby,	BEAM Ltd,	2006-02-22
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 */
#ifndef BFifoCirc_H
#define BFifoCirc_H	1

#include <stdint.h>
#include <BError.h>
#include <BCondInt.h>
#include <BMutex.h>

/// This class implements a pointer into the Fifo's circular buffer
class BFifoCircPos {
public:
			BFifoCircPos(uint32_t size);
	void		setSize(uint32_t size);

	void		set(uint32_t pos);			///< Sets the position
	uint32_t	pos();					///< The current position
	
	void		increment(uint32_t numFifoSamples);	///< Increment the pointer by the given value
	uint32_t	difference(const BFifoCircPos& pos);	///< Return the difference between the two pointers

			operator int();
	void		operator+=(uint32_t numFifoSamples);
	int		operator==(const BFifoCircPos& pos);
	int		operator!=(const BFifoCircPos& pos);
private:
	uint32_t	osize;
	uint32_t	opos;
};

/// This class implements a thread safe FIFO buffer
template <class Type> class BFifoCirc {
public:
	enum		{ defaultSize = 1024 };

			BFifoCirc(uint32_t size = defaultSize);
			~BFifoCirc();

	uint32_t	size();						///< Return the buffers actual size
	void		clear();					///< Clear all of the data in the buffer

			// Data write functions
	uint32_t	writeAvailable();				///< Returns the space available to write
	BError		writeWaitAvailable(uint32_t numFifoSamples);		///< Wait for the given number of samples
	
	BError		write(const Type* data, uint32_t numFifoSamples);	///< Writes the data to the buffer. Blocks until complete

	Type*		writeData();					///< Return a pointer to the current start of the buffer
	void		writeDone(uint32_t numFifoSamples);			///< Update the write pointer

			// Data read functions
	uint32_t	readAvailable();				///< Returns the number of bytes of data available
	BError		readWaitAvailable(uint32_t numFifoSamples);		///< Wait for given number of samples
	
	BError		read(Type* data, uint32_t numFifoSamples);
	
	Type*		readData();					///< Pointer to raw data
	BError		readDone(uint32_t numFifoSamples);			///< Updates read pointer

	Type&		operator[](int pos);				///< Direct access to read samples in buffer

protected:
	// Functions for internal use
	BError			mapCircularBuffer(uint32_t size);
	void			unmapCircularBuffer();

	BMutex			olock;
	uint32_t		ovmSize;
	uint32_t		osize;
	Type*			odata;
	BFifoCircPos		owritePos;				///< Current write position
	BCondValue		owriteNumFifoSamples;			///< The number of samples in the FIFO
	BFifoCircPos		oreadPos;				///< Current read position
};

#include <BFifoCirc.inc>

#endif
