/*******************************************************************************
 *	BFifo.inc	BFifo class
 *	T.Barnaby,	Beam Ltd,	2013-05-03
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 *
 * Implementation
 */
#include <BFifo.h>
#include <stdlib.h>

// The BFifo functions
template <class Type> BFifo<Type>::BFifo(BUInt size){
	osize = size;
	odata = new Type [osize];
	owritePos = 0;
	oreadPos = 0;
}

template <class Type> BFifo<Type>::~BFifo(){
	delete [] odata;
	odata = 0;
	osize = 0;
	owritePos = 0;
	oreadPos = 0;
}

template <class Type> void BFifo<Type>::clear(){
	BMutexLock	lock(olock, 1);

	owritePos = 0;
	oreadPos = 0;
}

template <class Type> BUInt BFifo<Type>::size(){
	return osize;
}

template <class Type> BError BFifo<Type>::resize(BUInt size){
	BMutexLock	lock(olock, 1);
	BError		err;
	
	delete [] odata;
	osize = size;
	odata = new Type [osize];
	owritePos = 0;
	oreadPos = 0;
	
	return err;
}


template <class Type> BUInt BFifo<Type>::writeAvailable(){
	BUInt		readPos = oreadPos;

	if(readPos <= owritePos)
		return osize - owritePos + readPos - 1;
	else
		return (readPos - owritePos - 1);
}

template <class Type> BUInt BFifo<Type>::writeAvailableChunk(){
	BUInt		readPos = oreadPos;

	if(readPos == 0)
		return osize - owritePos - 1;
	else if(readPos <= owritePos)
		return osize - owritePos;
	else
		return (readPos - owritePos - 1);
}

template <class Type> BError BFifo<Type>::write(const Type v){
	BMutexLock	lock(olock, 1);
	BError		err;
	
	odata[owritePos] = v;
	if(owritePos < (osize-1))
		owritePos++;
	else
		owritePos = 0;
	
	return err;
}

template <class Type> BError BFifo<Type>::write(const Type* data, BUInt num){
	BError	err;
	
	while(num--)
		write(*data++);
		
	return err;
}

template <class Type> Type* BFifo<Type>::writeData(){
	return &odata[owritePos];
}

template <class Type> Type* BFifo<Type>::writeData(BUInt& num){
	num = writeAvailableChunk();
	return writeData();
}

template <class Type> void BFifo<Type>::writeDone(BUInt num){
	BMutexLock	lock(olock, 1);

	if((owritePos + num) >= osize)
		owritePos = owritePos + num - osize;
	else
		owritePos += num;
}

template <class Type> void BFifo<Type>::writeBackup(BUInt num){
	BMutexLock	lock(olock, 1);
	BUInt		n = num;
	
	if(readAvailable() < n)
		n = readAvailable();
	
	if(owritePos < n)
		owritePos = owritePos + osize - n;
	else
		owritePos -= n;
}


template <class Type> BUInt BFifo<Type>::readAvailable(){
	BUInt		writePos = owritePos;
	
	if(oreadPos <= writePos)
		return writePos - oreadPos;
	else
		return osize - oreadPos + writePos;
}

template <class Type> BUInt BFifo<Type>::readAvailableChunk(){
	BUInt		writePos = owritePos;
	
	if(oreadPos <= writePos)
		return writePos - oreadPos;
	else
		return osize - oreadPos;
}

template <class Type> Type BFifo<Type>::read(){
	BMutexLock	lock(olock, 1);
	BFloat		v;

	v = odata[oreadPos];
	if(oreadPos < (osize-1))
		oreadPos++;
	else
		oreadPos = 0;

	return v;
}

template <class Type> Type BFifo<Type>::readPos(BUInt pos){
	BUInt	p = oreadPos + pos;

	if(p >= osize)
		return odata[p - osize];
	else
		return odata[p];
}

template <class Type> BError BFifo<Type>::read(Type* data, BUInt num){
	BError	err;
	
	while(num--)
		*data++ = read();
	
	return err;
}

template <class Type> Type* BFifo<Type>::readData(){
	return &odata[oreadPos];
}

template <class Type> Type* BFifo<Type>::readData(BUInt& num){
	num = readAvailableChunk();
	return readData();
}

template <class Type> void BFifo<Type>::readDone(BUInt num){
	BMutexLock	lock(olock, 1);

	if((oreadPos + num) >= osize)
		oreadPos = oreadPos + num - osize;
	else
		oreadPos += num;
}

template <class Type> Type& BFifo<Type>::operator[](int pos){
	BUInt	p = oreadPos + pos;

	if(p >= osize)
		return odata[p - osize];
	else
		return odata[p];
}
