/*
** File: model_adpexrc4fx.c
** Project: ADMXRC2 module driver
** Purpose: Code specific to the ADPE-XRC-4FX, including the function to
**          bootstrap the device context.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include <core/adb1.h>
#include <core/adpexrc4fx.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdpexrc4fxRev1* pVpd = (VpdAdpexrc4fxRev1*)pVpdBuffer;
  unsigned int order;

  if (bankIndex >= 5) {
    return ADMXRC2_INVALID_INDEX;
  }

  if (bankIndex < 4) {
    order = pVpd->rev0.sdram[bankIndex].order;
    pIoctl->out.width = 64U;
    pIoctl->out.typeMask = ADMXRC2_RAM_SDRAM_DDR2;
    pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  } else {
    order = pVpd->ssram.order;
    pIoctl->out.width = 64U;
    pIoctl->out.typeMask = ADMXRC2_RAM_SRAM_DDR2;
    pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  }
  pIoctl->out.bFitted = order ? TRUE : FALSE;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdpexrc4fxRev1* pVpd = (VpdAdpexrc4fxRev1*)pVpdBuffer;
  unsigned int i;

  pIoctl->out.cardId = pVpd->rev0.cardId;
  pIoctl->out.fpgaType = pVpd->rev0.fpgaType;
  pIoctl->out.serialNumber = pVpd->rev0.serialNumber;
  pIoctl->out.numMemoryBank = 5;
  pIoctl->out.memoryBankFitted = 0;
  for (i = 0; i < 4; i++) {
    if (pVpd->rev0.sdram[i].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << i);
    }
  }
  if (pVpd->ssram.order) {
    /* Bank is fitted */
    pIoctl->out.memoryBankFitted |= (0x1U << 4);
  }
  pIoctl->out.logicRevision = pVpd->rev0.cpldRev;
  pIoctl->out.boardRevision = pVpd->rev0.pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdpexrc4fxRev1* pVpd = (VpdAdpexrc4fxRev1*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1517");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->rev0.fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  pIoctl->out.scd = pVpd->rev0.scd;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdpexrc4fx(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdpexrc4fx: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 4U;
  pDevCtx->configureDmaAddress = 0x800018U; /* TODO - should make this dynamic, i.e. based on current value of LS1B register */
  pDevCtx->configureDmaMode =
    ADB1_DMA_CFG_32BIT | ADB1_DMA_CFG_USE_READY | ADB1_DMA_CFG_USE_BTERM |
    ADB1_DMA_CFG_ENABLE_BURST | ADB1_DMA_CFG_FIXED_LOCAL;
}
