/*
** File: model_admxrc2.c
** Project: ADMXRC2 module driver
** Purpose: ADM-XRC-II specific code, including the function to bootstrap
**          the device context.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include <core/pci9xxx.h>
#include <core/admxrc2.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdmxrc2Rev0* pVpd = (VpdAdmxrc2Rev0*)pVpdBuffer;
  uint32_t zbt, bank;
  unsigned int shift, order;

  if (bankIndex >= 6) {
    return ADMXRC2_INVALID_INDEX;
  }
  zbt = pVpd->zbt01 | ((uint32_t)pVpd->zbt23 << 8) | ((uint32_t)pVpd->zbt45 << 16);
  shift = bankIndex * 4;
  bank = (zbt >> shift) & 0xfU;
  order = (bank >> 2) + 18;
  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.width = (pVpd->fpgaType == ADMXRC2_FPGA_2V3000) ? 32 : 36;
  pIoctl->out.typeMask = bank & 0x3U;
  pIoctl->out.bFitted = (bank & 0x3U) ? TRUE : FALSE;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdmxrc2Rev0* pVpd = (VpdAdmxrc2Rev0*)pVpdBuffer;
  uint32_t zbtAll;
  unsigned int i;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType;
  pIoctl->out.serialNumber = pVpd->serialNumber;
  pIoctl->out.numMemoryBank = 6;
  pIoctl->out.memoryBankFitted = 0;
  zbtAll = pVpd->zbt01 | ((uint32_t)pVpd->zbt23 << 8) | ((uint32_t)pVpd->zbt45 << 16);
  for (i = 0; i < 6; i++) {
    uint8_t bank = (uint8_t)((zbtAll >> (i * 4)) & 0xfU);

    if (0 == (bank & 0x3)) {
      /* Bank is not fitted */
    } else {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << i);
    }
  }
  pIoctl->out.logicRevision = pVpd->cpldRev;
  pIoctl->out.boardRevision = pVpd->pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdmxrc2Rev0* pVpd = (VpdAdmxrc2Rev0*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1152");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  pIoctl->out.tempGrade = (uint32_t)-1;
  pIoctl->out.scd = (uint32_t)-1;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdmxrc2(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdmxrc2: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 2U;
  pDevCtx->configureDmaAddress = 0x800008U; /* TODO - should make this dynamic, i.e. based on current value of LAS1BA */
  pDevCtx->configureDmaMode =
    PCI9656_DMAMODE_8BIT | PCI9656_DMAMODE_USE_READY | PCI9656_DMAMODE_USE_BTERM |
    PCI9656_DMAMODE_ENABLE_BURST | PCI9656_DMAMODE_FIXED_LOCAL;
}
