/*
** File: model_adcbbp.c
** Project: ADMXRC2 module driver
** Purpose: Code specific to ADC-BBP, including the function to bootstrap the
**          device context.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include <core/adb1.h>
#include <core/adcbbp.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdcbbpRev0* pVpd = (VpdAdcbbpRev0*)pVpdBuffer;
  uint8_t order, type;

  if (bankIndex < DF_ARRAY_LENGTH(pVpd->zbt0)) {
    order = pVpd->zbt0[bankIndex].order;
    type = pVpd->zbt0[bankIndex].type;
  } else {
    bankIndex -= (unsigned int)DF_ARRAY_LENGTH(pVpd->zbt0);
    if (bankIndex < DF_ARRAY_LENGTH(pVpd->zbt1)) {
      order = pVpd->zbt1[bankIndex].order;
      type = pVpd->zbt1[bankIndex].type;
    } else {
      return ADMXRC2_INVALID_INDEX;
    }
  }

  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.width = 32U;
  pIoctl->out.typeMask = 0;
  if (type & 0x1U) {
    pIoctl->out.typeMask |= ADMXRC2_RAM_ZBTFT;
  }
  if (type & 0x2U) {
    pIoctl->out.typeMask |= ADMXRC2_RAM_ZBTP;
  }
  pIoctl->out.bFitted = order ? TRUE : FALSE;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdcbbpRev0* pVpd = (VpdAdcbbpRev0*)pVpdBuffer;
  unsigned int i, j;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType[0];
  pIoctl->out.serialNumber = pVpd->serialNumber;
  pIoctl->out.numMemoryBank = DF_ARRAY_LENGTH(pVpd->zbt0) + DF_ARRAY_LENGTH(pVpd->zbt1);
  pIoctl->out.memoryBankFitted = 0;
  j = 0;
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->zbt0); i++, j++) {
    if (pVpd->zbt0[i].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << j);
    }
  }
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->zbt1); i++, j++) {
    if (pVpd->zbt1[i].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << j);
    }
  }
  pIoctl->out.logicRevision = pVpd->logicRev;
  pIoctl->out.boardRevision = pVpd->pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdcbbpRev0* pVpd = (VpdAdcbbpRev0*)pVpdBuffer;

  if (targetIndex >= 2) {
    return ADMXRC2_INVALID_INDEX;
  }

  if (targetIndex == 0) {
    dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1136");
    pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType[0];
    pIoctl->out.scd = pVpd->scd[0];
  } else {
    dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1738");
    pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType[1];
    pIoctl->out.scd = pVpd->scd[1];
  }
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  switch (pVpd->tempGrade) {
  default:
    pIoctl->out.tempGrade = (uint32_t)-1;
    break;
  }

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdcbbp(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdcbbp: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 4U;
  pDevCtx->configureDmaAddress = 0x800020U; /* TODO - should make this dynamic, i.e. based on current value of LS1B register */
  pDevCtx->configureDmaMode =
    ADB1_DMA_CFG_32BIT | ADB1_DMA_CFG_USE_READY | ADB1_DMA_CFG_USE_BTERM |
    ADB1_DMA_CFG_ENABLE_BURST | ADB1_DMA_CFG_FIXED_LOCAL;
}
