/*
** File: fpga.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for configuring target FPGA(s).
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "fpga.h"
#include <admxrc2/types.h>

#define DEBUGLEVEL_ERROR  (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_CANCEL (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_NORMAL (6) /* Level of debug messages relating to normal operation */

static void
reverseBytes(
  void* pBuf,
  size_t n)
{
  const size_t mask32 = 0x3U;
  uint8_t* pBuf8;
  uint8_t a8, b8;
  uint32_t a32, b32;
  uint32_t* pBuf32;
  size_t n8, n32;

  /* Ensure pointer aligned to 4-byte boundary by doing first few bytes */
  pBuf8 = (uint8_t*)pBuf;
  n8 = (size_t)(uintptr_t)pBuf & mask32;
  n -= n8;
  while (n8) {
    a8 = *pBuf8;
    /* Reverse bits in byte */
    b8 = (uint8_t)(((a8 & 0x01U) << 7) | ((a8 & 0x02U) << 5) | ((a8 & 0x04U) << 3) | ((a8 & 0x08U) << 1) |
                   ((a8 & 0x10U) >> 1) | ((a8 & 0x20U) >> 3) | ((a8 & 0x40U) >> 5) | ((a8 & 0x80U) >> 7));
    *pBuf8++ = b8;
  }

  /* Do bulk of bit-reversal as 32-bit words */
  pBuf32 = (uint32_t*)pBuf8;
  n32 = n & ~mask32;
  n -= n32;
  while (n32) {
    a32 = *pBuf32;
    /* Reverse bits in each byte of 32-bit word */
    b32 = ((a32 & 0x01010101U) << 7) | ((a32 & 0x02020202U) << 5) | ((a32 & 0x04040404U) << 3) | ((a32 & 0x08080808U) << 1) |
          ((a32 & 0x10101010U) >> 1) | ((a32 & 0x20202020U) >> 3) | ((a32 & 0x40404040U) >> 5) | ((a32 & 0x80808080U) >> 7);
    *pBuf32++ = b32;
    n32 -= 4;
  }

  /* Do last few bytes */
  pBuf8 = (uint8_t*)pBuf32;
  while (n--) {
    a8 = *pBuf8;
    /* Reverse bits in byte */
    b8 = (uint8_t)(((a8 & 0x01U) << 7) | ((a8 & 0x02U) << 5) | ((a8 & 0x04U) << 3) | ((a8 & 0x08U) << 1) |
                   ((a8 & 0x10U) >> 1) | ((a8 & 0x20U) >> 3) | ((a8 & 0x40U) >> 5) | ((a8 & 0x80U) >> 7));
    *pBuf8++ = b8;
  }
}

static DfIoStatus
mapDmaStatus(
  CoreDmaStatus dmaStatus)
{
  switch (dmaStatus) {
  case CoreDmaSuccess:
    return DfIoStatusSuccess;

  case CoreDmaHardwareError:
    return DfIoStatusError(ADMXRC2_FAILED);

  case CoreDmaInvalidChannel:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreDmaInvalidRegion:
    return DfIoStatusError(ADMXRC2_INVALID_REGION);

  case CoreDmaDirectionNotSupported:
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);

  case CoreDmaCancelled:
    return DfIoStatusError(ADMXRC2_CANCELLED);

  case CoreDmaUnexpectedError:
  case CoreDmaInvalidTicket:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapDescriptionResult(
  DfDescriptionResult result)
{
  switch (result) {
  case DfDescriptionSuccess:
    return DfIoStatusSuccess;

  case DfDescriptionNullPointer:
    return DfIoStatusError(ADMXRC2_NULL_POINTER);

  case DfDescriptionInvalidBuffer:
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */

  case DfDescriptionNoMemory:
    return DfIoStatusError(ADMXRC2_NO_MEMORY);

  case DfDescriptionUnexpectedError:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapAcquireStatus(
  CoreAcquireStatus acqStatus)
{
  switch (acqStatus) {
  case CoreAcquireSuccess:
    return DfIoStatusSuccess;

  case CoreAcquireMustWait:
    return DfIoStatusError(ADMXRC2_DEVICE_BUSY);

  case CoreAcquireCancelled:
    return DfIoStatusError(ADMXRC2_CANCELLED);

  case CoreAcquireRemoved:
  case CoreAcquireFailed:
    return DfIoStatusError(ADMXRC2_FAILED);

  case CoreAcquireInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreAcquireInvalid:
  case CoreAcquireUnexpectedError:
  case CoreAcquireWaiting:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapFpgaControlStatus(
  CoreFpgaControlStatus status)
{
  switch (status) {
  case CoreFpgaControlSuccess:
    return DfIoStatusSuccess;

  case CoreFpgaControlInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreFpgaControlNotSupported:
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);

  case CoreFpgaControlInvalidOp:
  case CoreFpgaControlInvalidTicket:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapFpgaSelectMapStatus(
  CoreFpgaSelectMapStatus status)
{
  switch (status) {
  case CoreFpgaSelectMapSuccess:
    return DfIoStatusSuccess;

  case CoreFpgaSelectMapInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreFpgaSelectMapNotSupported:
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);

  case CoreFpgaSelectMapInvalidTicket:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
unconfigureWork(
	Admxrc2DeviceContext* pDevCtx,
  CoreTicket* pTicket,
  unsigned int targetIndex)
{
  DfIoStatus status = DfIoStatusSuccess;
  CoreFpgaControlStatus ctlStatus;
  DfTime start, expires;
  boolean_t bInterrupt, bValue;

  /* The first phase of full configuration must complete within 1s */
  start = dfTimeGet();
  expires = start + dfMicrosecondsToTime(1000000);

  /* Assert PROG */
  bValue = TRUE;
  ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpSetProg, &bValue);
  if (CoreFpgaControlSuccess != ctlStatus) {
    status = mapFpgaControlStatus(ctlStatus);
    goto done;
  }

  /* Wait for DONE deasserted */
  while (1) {
    ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpGetDone, &bValue);
    if (CoreFpgaControlSuccess != ctlStatus) {
      status = mapFpgaControlStatus(ctlStatus);
      goto done;
    }
    if (!bValue) {
      break;
    }

    bInterrupt = dfDelayThreadFor(dfMicrosecondsToTime(10000));
    if (bInterrupt) {
      /* Deassert PROG */
      bValue = FALSE;
      ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpSetProg, &bValue);
      /* Abort early */
      status = DfIoStatusError(ADMXRC2_CANCELLED);
      goto done;
    }
    if (dfTimeGet() >= expires) {
      /* Timeout waiting for DONE deasserted */
      bValue = FALSE;
      ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpSetProg, &bValue);
      status = DfIoStatusError(ADMXRC2_FAILED);
      goto done;
    }
  }

  /* Wait for INIT# asserted */
  while (1) {
    ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpGetInit, &bValue);
    if (CoreFpgaControlSuccess != ctlStatus) {
      status = mapFpgaControlStatus(ctlStatus);
      goto done;
    }
    if (bValue) {
      break;
    }

    bInterrupt = dfDelayThreadFor(dfMicrosecondsToTime(10000));
    if (bInterrupt) {
      /* Deassert PROG */
      bValue = FALSE;
      ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpSetProg, &bValue);
      /* Abort early */
      status = DfIoStatusError(ADMXRC2_CANCELLED);
      goto done;
    }
    if (dfTimeGet() >= expires) {
      /* Timeout waiting for INIT# asserted */
      bValue = FALSE;
      ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpSetProg, &bValue);
      status = DfIoStatusError(ADMXRC2_FAILED);
      goto done;
    }
  }

  /* Deassert PROG */
  bValue = FALSE;
  ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpSetProg, &bValue);
  if (CoreFpgaControlSuccess != ctlStatus) {
    status = mapFpgaControlStatus(ctlStatus);
    goto done;
  }

  /* Wait for INIT# deasserted */
  while (1) {
    ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpGetInit, &bValue);
    if (CoreFpgaControlSuccess != ctlStatus) {
      status = mapFpgaControlStatus(ctlStatus);
      goto done;
    }
    if (!bValue) {
      break;
    }

    bInterrupt = dfDelayThreadFor(dfMicrosecondsToTime(10000));
    if (bInterrupt) {
      /* Abort early */
      status = DfIoStatusError(ADMXRC2_CANCELLED);
      goto done;
    }
    if (dfTimeGet() >= expires) {
      /* Timeout waiting for INIT# deasserted */
      status = DfIoStatusError(ADMXRC2_FAILED);
      goto done;
    }
  }

done:
  return status;
}
static DfIoStatus
checkConfiguredWork(
	Admxrc2DeviceContext* pDevCtx,
  CoreTicket* pTicket,
  unsigned int targetIndex,
  uint32_t flags)
{
  DfIoStatus status = DfIoStatusSuccess;
  CoreFpgaControlStatus ctlStatus;
  DfTime start, expires;
  boolean_t bInterrupt, bValue;

  /* The second phase of full configuration must complete within 1s */
  start = dfTimeGet();
  expires = start + dfMicrosecondsToTime(1000000);

  /* Check if INIT# is asserted */
  ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpGetInit, &bValue);
  if (CoreFpgaControlSuccess != ctlStatus) {
    status = mapFpgaControlStatus(ctlStatus);
    goto done;
  }
  if (bValue) {
    status = DfIoStatusError(ADMXRC2_FAILED);
    goto done;
  }

  /* Wait for DONE asserted */
  while (1) {
    ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpGetDone, &bValue);
    if (CoreFpgaControlSuccess != ctlStatus) {
      status = mapFpgaControlStatus(ctlStatus);
      goto done;
    }
    if (bValue) {
      break;
    }

    bInterrupt = dfDelayThreadFor(dfMicrosecondsToTime(10000));
    if (bInterrupt) {
      /* Abort early */
      status = DfIoStatusError(ADMXRC2_CANCELLED);
      goto done;
    }
    if (dfTimeGet() >= expires) {
      /* Timeout waiting for DONE asserted */
      status = DfIoStatusError(ADMXRC2_FAILED);
      goto done;
    }
  }

  if (!(flags & ADMXRC2_CONFIGURE_NOCHECK)) {
    /* Wait for bridge <-> target link OK indication */
    while (1) {
      ctlStatus = pDevCtx->coreInterface.pFpgaControl(pDevCtx->pCoreContext, pTicket, targetIndex, CoreFpgaControlOpGetLinkStatus, &bValue);
      if (CoreFpgaControlSuccess != ctlStatus) {
        status = mapFpgaControlStatus(ctlStatus);
        goto done;
      }
      if (bValue) {
        break;
      }

      bInterrupt = dfDelayThreadFor(dfMicrosecondsToTime(10000));
      if (bInterrupt) {
        /* Abort early */
        status = DfIoStatusError(ADMXRC2_CANCELLED);
        goto done;
      }
      if (dfTimeGet() >= expires) {
        /* Timeout waiting for DONE asserted */
        status = DfIoStatusError(ADMXRC2_FAILED);
        goto done;
      }
    }
  }

done:
  return status;
}

static void
configureDmaCallback(
  CoreTicket* pCoreTicket,
  void* pCallbackContext, /* ignored */
  CoreDmaStatus dmaStatus)
{
  ConfigureDmaTicket* pTicket = DF_CONTAINER_OF(pCoreTicket, ConfigureDmaTicket, ticket);

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("configureDmaCallback: entered, pTicket=%p dmaStatus=%u\n",
    pTicket, dmaStatus));

  pTicket->status = mapDmaStatus(dmaStatus);
  dfEventSignal(&pTicket->event);
}

/*
** -----------------------------------------------------------------------------
** IOCTL handlers
** -----------------------------------------------------------------------------
*/

#define THRESHOLD_ALLOC (0x1000U)   /* If request larger than this, allocate thread-accessible memory; otherwise use stack */
#define MAX_CHUNK_STACK (128)       /* Size of buffer if using stack */
#define MAX_CHUNK_ALLOC (0x10000U)  /* Max. size of buffer if allocating memory */

DfIoStatus
ioctlConfigure(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  const uint32_t legalFlagsMask = ADMXRC2_CONFIGURE_PARTIAL | ADMXRC2_CONFIGURE_NOCHECK | ADMXRC2_CONFIGURE_SHARE;
#endif
  CoreTicket ticket;
  CoreTicket* pTicket = &ticket;
  uint8_t tmp[MAX_CHUNK_STACK];         /* Stack buffer */
#if defined(ADMXRC2_CONFIGURE_ALLOC)
  uint8_t* pAllocBuffer = NULL;         /* Allocated buffer */
#endif
  size_t n, maxChunk = MAX_CHUNK_STACK; /* Assume using stack buffer, may modify later */
  uint8_t* pTmp = tmp;                  /* Assume using stack buffer, may modify later */
  IOCTLS_ADMXRC2_CONFIGURE* pIoctl = (IOCTLS_ADMXRC2_CONFIGURE*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  DfSpinLockFlags f;
  CoreAcquireStatus acqStatus;
  CoreFpgaSelectMapStatus smapStatus;
  unsigned int targetIndex;
  uint32_t flags;
  size_t length, remaining, chunk;
  uint8_t* pUserBuffer;
  boolean_t bAcquired = FALSE;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  targetIndex = pIoctl->in.targetIndex;
  flags = pIoctl->in.flags;
  pUserBuffer = (uint8_t*)pIoctl->in.pBuffer;
  length = pIoctl->in.length;
  
  dfDebugPrint(6, ("ioctlConfigure: targetIndex=%lu flags=0x%lx pUserBuffer=%p length=" DF_FMT_U64 "(0x%08lx_%08lx)\n",
    (unsigned long)targetIndex, (unsigned long)flags, pUserBuffer, (unsigned long long)length, dfSplitUint64((uint64_t)length)));

  if (targetIndex >= pDevCtx->info.numTargetFpga) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }
#if defined(ADMXRC2_CHECK_FLAGS)
  if (flags & ~legalFlagsMask) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

#if defined(ADMXRC2_CONFIGURE_ALLOC)
  /* If the request is large enough, try to allocate thread-accessible memory for the buffer */
  if (length >= THRESHOLD_ALLOC) {
    n = (length > MAX_CHUNK_ALLOC) ? MAX_CHUNK_ALLOC : length;
    pAllocBuffer = (uint8_t*)dfMallocThread(n);
    if (NULL != pAllocBuffer) {
      /* Successfully allocated memory, so use it */
      pTmp = pAllocBuffer;
      maxChunk = n;
    }
  }
#endif

  f = dfSpinLockGet(&pDevCtx->fpga[targetIndex].owner.lock);
  if (pDevCtx->fpga[targetIndex].owner.pClCtx == NULL) {
    /* Target FPGA is currently not owned by any client */
    if (!(flags & ADMXRC2_CONFIGURE_SHARE)) {
      /* Set owner of target FPGA to this client, and proceed. */
      pDevCtx->fpga[targetIndex].owner.pClCtx = pClCtx;
    }
  } else if (pClCtx == pDevCtx->fpga[targetIndex].owner.pClCtx) {
    /* Target FPGA is currently owned by this client; OK to proceed. */
  } else {
    /* Target FPGA is currently owned by a different client; don't proceed. */
    dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);
    return DfIoStatusError(ADMXRC2_NOT_OWNER);
  }
  dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);

  pTicket->count = 1;
  pTicket->resources[0].code = CORE_RESOURCE_TARGETFPGA;
  pTicket->resources[0].index = (uint8_t)targetIndex;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, pTicket);
  if (CoreAcquireSuccess != acqStatus) {
    status = mapAcquireStatus(acqStatus);
    goto done;
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, pTicket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    status = mapAcquireStatus(acqStatus);
    goto done;
  }
  bAcquired = TRUE;

  if (!(flags & ADMXRC2_CONFIGURE_PARTIAL)) {
    status = unconfigureWork(pDevCtx, pTicket, targetIndex);
    if (DfIoStatusIsError(status)) {
      goto done;
    }
  }

  DF_USER_SPACE_BEGIN(TAG1)

  dfUserBufferValidate(pUserBuffer, length, FALSE, TAG1);
  remaining = length;
  while (remaining) {
    chunk = maxChunk;
    if (chunk > remaining) {
      chunk = remaining;
    }
    dfCopyUK(pTmp, pUserBuffer, chunk, TAG1);
    if (pDevCtx->bReverseSelectMapData) {
  		reverseBytes(pTmp, chunk);
    }
    if (remaining == length) {
      n = (chunk > 0x100U) ? 0x100U : chunk;
      dfDebugPrint(8, ("ioctlConfigure: First %lu bytes of SelectMap data:\n", (unsigned long)n));
      dfDebugDumpMemory(8, 0, 16, 2, TRUE, pTmp, n);
    }
    smapStatus = pDevCtx->coreInterface.pFpgaSelectMap(pDevCtx->pCoreContext, pTicket, targetIndex, TRUE, chunk, pTmp);
    if (CoreFpgaSelectMapSuccess != smapStatus) {
      status = mapFpgaSelectMapStatus(smapStatus);
      break;
    }
    remaining -= chunk;
    pUserBuffer += chunk;
  }

#if defined(ADMXRC2_CONFIGURE_ALLOC)
  /* Free the buffer as soon as it is no longer required */
  if (NULL != pAllocBuffer) {
    dfFreeThread(pAllocBuffer);
    pAllocBuffer = NULL;
  }
#endif

  DF_USER_SPACE_EXCEPT(TAG1)

  status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */

  DF_USER_SPACE_END(TAG1)

  if (DfIoStatusIsError(status)) {
    goto done;
  }

  if (!(flags & ADMXRC2_CONFIGURE_PARTIAL)) {
    status = checkConfiguredWork(pDevCtx, pTicket, targetIndex, flags);
    if (DfIoStatusIsError(status)) {
      goto done;
    }
  }

done:
#if defined(ADMXRC2_CONFIGURE_ALLOC)
  if (NULL != pAllocBuffer) {
    dfFreeThread(pAllocBuffer);
    pAllocBuffer = NULL;
  }
#endif

  if (bAcquired) {
    pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, pTicket);
    bAcquired = FALSE;
  }

  if (DfIoStatusIsError(status) && !(flags & ADMXRC2_CONFIGURE_SHARE)) {
    /* Failed to configure target FPGA, so mark it as not owned by any client. */
    f = dfSpinLockGet(&pDevCtx->fpga[targetIndex].owner.lock);
    pDevCtx->fpga[targetIndex].owner.pClCtx = NULL;
    dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);
  }

  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlConfigureThunk(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_CONFIGURE* pIoctl32 = (IOCTLS32_ADMXRC2_CONFIGURE*)pBuffer;
  IOCTLS_ADMXRC2_CONFIGURE ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.targetIndex = pIoctl32->in.targetIndex;
  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.pBuffer = dfThunkConstPtr(pIoctl32->in.pBuffer);
  ioctl.in.length = (size_t)pIoctl32->in.length;
  return ioctlConfigure(pDevCtx, pClCtx, &ioctl, sizeof(ioctl.in));
}
#endif

DfIoStatus
ioctlUnconfigure(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  const uint32_t legalFlagsMask = ADMXRC2_CONFIGURE_PARTIAL | ADMXRC2_CONFIGURE_SHARE;
#endif
  CoreTicket ticket;
  CoreTicket* pTicket = &ticket;
  IOCTLS_ADMXRC2_UNCONFIGURE* pIoctl = (IOCTLS_ADMXRC2_UNCONFIGURE*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  DfSpinLockFlags f;
  CoreAcquireStatus acqStatus;
  unsigned int targetIndex;
  uint32_t flags;
  boolean_t bAcquired = FALSE;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  targetIndex = pIoctl->in.targetIndex;
  flags = pIoctl->in.flags;
  
  dfDebugPrint(6, ("ioctlUnconfigure: targetIndex=%lu flags=0x%lx\n",
    (unsigned long)targetIndex, (unsigned long)flags));

  if (targetIndex >= pDevCtx->info.numTargetFpga) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }
#if defined(ADMXRC2_CHECK_FLAGS)
  if (flags & ~legalFlagsMask) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  f = dfSpinLockGet(&pDevCtx->fpga[targetIndex].owner.lock);
  if (pDevCtx->fpga[targetIndex].owner.pClCtx != NULL && pClCtx != pDevCtx->fpga[targetIndex].owner.pClCtx) {
    /* Target FPGA is currently not owned by this client; don't proceed. */
    dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);
    return DfIoStatusError(ADMXRC2_NOT_OWNER);
  }
  dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);

  if (!(flags & ADMXRC2_CONFIGURE_PARTIAL)) {
    pTicket->count = 1;
    pTicket->resources[0].code = CORE_RESOURCE_TARGETFPGA;
    pTicket->resources[0].index = (uint8_t)targetIndex;
    acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, pTicket);
    if (CoreAcquireSuccess != acqStatus) {
      status = mapAcquireStatus(acqStatus);
      goto done;
    }
    acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, pTicket, 0);
    if (CoreAcquireSuccess != acqStatus) {
      status = mapAcquireStatus(acqStatus);
      goto done;
    }
    bAcquired = TRUE;

    status = unconfigureWork(pDevCtx, pTicket, targetIndex);
    if (DfIoStatusIsError(status)) {
      goto done;
    }
  }

done:
  if (bAcquired) {
    pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, pTicket);
    bAcquired = FALSE;
  }

  if (!(flags & ADMXRC2_CONFIGURE_SHARE)) {
    /* Regardless of any error, mark this target FPGA as not owned by any client. */
    f = dfSpinLockGet(&pDevCtx->fpga[targetIndex].owner.lock);
    pDevCtx->fpga[targetIndex].owner.pClCtx = NULL;
    dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);
  }

  return status;
}

DfIoStatus
ioctlConfigureDma(
  Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  DfClientRequest* pRequest,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  const uint32_t legalFlagsMask = ADMXRC2_CONFIGURE_PARTIAL | ADMXRC2_CONFIGURE_NOCHECK | ADMXRC2_CONFIGURE_SHARE;
#endif
  IOCTLS_ADMXRC2_CONFIGUREDMA* pIoctl = (IOCTLS_ADMXRC2_CONFIGUREDMA*)pBuffer;
  ConfigureDmaTicket ticket;
  ConfigureDmaTicket* pTicket = &ticket;
  CoreAcquireStatus acqStatus;
  CoreDmaStatus dmaStatus;
  boolean_t bAcquired = FALSE, bInterrupted;
  DfBufferDescription* pDescription;
  DfDescriptionResult descResult;
  DfIoStatus status;
  DfSpinLockFlags f;
  unsigned int targetIndex, dmaChannel;
  uint32_t flags;
  size_t length;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }
  if (pDevCtx->bReverseSelectMapData) {
    /* Bit-reversal, if required, should have already been done in user-mode */
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);
  }

  targetIndex = pIoctl->in.targetIndex;
  if (targetIndex >= pDevCtx->info.numTargetFpga) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }
  dmaChannel = pIoctl->in.dmaChannel;
  if (dmaChannel != ADMXRC2_DMACHAN_ANY && dmaChannel >= pDevCtx->coreInterface.pGetNumDmaChannel(pDevCtx->pCoreContext)) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }
  length = pIoctl->in.length;
  if (0 == length) {
    dfRequestPending(pRequest);
    dfRequestComplete(pRequest, DfIoStatusSuccess);
    return DfIoStatusDeferred;
  }
  flags = pIoctl->in.flags;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif
  descResult = dfBufferDescriptionCreateUser(pIoctl->in.pBuffer, length, &pDescription);
  if (DfDescriptionSuccess != descResult) {
    return mapDescriptionResult(descResult);
  }
  if (!dfBufferDescriptionLock(pDescription)) {
    dfBufferDescriptionDelete(pDescription);
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }

  f = dfSpinLockGet(&pDevCtx->fpga[targetIndex].owner.lock);
  if (pDevCtx->fpga[targetIndex].owner.pClCtx == NULL) {
    /* Target FPGA is currently not owned by any client */
    if (!(flags & ADMXRC2_CONFIGURE_SHARE)) {
      /* Set owner of target FPGA to this client, and proceed. */
      pDevCtx->fpga[targetIndex].owner.pClCtx = pClCtx;
    }
  } else if (pClCtx == pDevCtx->fpga[targetIndex].owner.pClCtx) {
    /* Target FPGA is currently owned by this client; OK to proceed. */
  } else {
    /* Target FPGA is currently owned by a different client; don't proceed. */
    dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);
    dfBufferDescriptionUnlock(pDescription);
    dfBufferDescriptionDelete(pDescription);
    return DfIoStatusError(ADMXRC2_NOT_OWNER);
  }
  dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);

  pTicket->ticket.count = 2;
  pTicket->ticket.resources[0].code = CORE_RESOURCE_DMAENGINE;
  pTicket->ticket.resources[0].index = (uint8_t)((dmaChannel == ADMXRC2_DMACHAN_ANY) ? CORE_RESOURCE_INDEX_ANY : (uint8_t)dmaChannel);
  pTicket->ticket.resources[1].code = CORE_RESOURCE_TARGETFPGA;
  pTicket->ticket.resources[1].index = (uint8_t)targetIndex;
  dfEventInit(&pTicket->event);
  pTicket->status = DfIoStatusSuccess;
  pTicket->dmaParameters.pBufferDescription = pDescription;
  pTicket->dmaParameters.bWriteToLocal = TRUE;
  pTicket->dmaParameters.bFixedLocalAddress = TRUE;
  pTicket->dmaParameters.region.offset = 0;
  pTicket->dmaParameters.region.length = length;
  pTicket->dmaParameters.region.localAddress = pDevCtx->configureDmaAddress;
  pTicket->dmaParameters.modeFlags = pDevCtx->configureDmaMode;
  pTicket->variant.pDescription = pDescription;

  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, (CoreTicket*)&pTicket->ticket);
  if (acqStatus != CoreAcquireSuccess) {
    status = mapAcquireStatus(acqStatus);
    goto done;
  }

  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, (CoreTicket*)&pTicket->ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    status = mapAcquireStatus(acqStatus);
    goto done;
  }
  bAcquired = TRUE;
  dmaChannel = pTicket->ticket.resources[0].actualIndex;

  if (!(flags & ADMXRC2_CONFIGURE_PARTIAL)) {
    status = unconfigureWork(pDevCtx, (CoreTicket*)pTicket, targetIndex);
    if (DfIoStatusIsError(status)) {
      goto done;
    }
  }

  dmaStatus = pDevCtx->coreInterface.pDmaInitialize(pDevCtx->pCoreContext, (CoreTicket*)&pTicket->ticket, dmaChannel, &pTicket->dmaParameters, configureDmaCallback, NULL);
  if (CoreDmaSuccess != dmaStatus) {
    status = mapDmaStatus(dmaStatus);
    goto done;
  }

  pDevCtx->coreInterface.pDmaStart(pDevCtx->pCoreContext, (CoreTicket*)&pTicket->ticket, dmaChannel);
  bInterrupted = dfEventWaitInterruptible(&pTicket->event);
  if (bInterrupted) {
    pDevCtx->coreInterface.pDmaCancel(pDevCtx->pCoreContext, (CoreTicket*)&pTicket->ticket, dmaChannel);
    dfEventWait(&pTicket->event);
    status = DfIoStatusError(ADMXRC2_CANCELLED);
  } else {
    status = pTicket->status;
  }

  if (DfIoStatusIsError(status)) {
    goto done;
  }

  if (!(flags & ADMXRC2_CONFIGURE_PARTIAL)) {
    status = checkConfiguredWork(pDevCtx, (CoreTicket*)pTicket, targetIndex, flags);
    if (DfIoStatusIsError(status)) {
      goto done;
    }
  }

done:
  dfEventUninit(&pTicket->event);
  if (bAcquired) {
    pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, (CoreTicket*)&pTicket->ticket);
  }
  if (DfIoStatusIsError(status) && !(flags & ADMXRC2_CONFIGURE_SHARE)) {
    /* Failed to configure target FPGA, so mark it as not owned by any client. */
    f = dfSpinLockGet(&pDevCtx->fpga[targetIndex].owner.lock);
    pDevCtx->fpga[targetIndex].owner.pClCtx = NULL;
    dfSpinLockPut(&pDevCtx->fpga[targetIndex].owner.lock, f);
  }
  dfBufferDescriptionUnlock(pDescription);
  dfBufferDescriptionDelete(pDescription);
  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlConfigureDmaThunk(
  Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  DfClientRequest* pRequest,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_CONFIGUREDMA* pIoctl32 = (IOCTLS32_ADMXRC2_CONFIGUREDMA*)pBuffer;
  IOCTLS_ADMXRC2_CONFIGUREDMA ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.targetIndex = pIoctl32->in.targetIndex;
  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.pBuffer = dfThunkConstPtr(pIoctl32->in.pBuffer);
  ioctl.in.length = (size_t)pIoctl32->in.length;
  ioctl.in.dmaChannel = pIoctl32->in.dmaChannel;
  return ioctlConfigureDma(pDevCtx, pClCtx, pRequest, &ioctl, sizeof(ioctl.in));
}
#endif
