/*
** File: mmap.c  
** Project: Linux Driver Framework
** Purpose: Mechanism by which driver can register regions of device memory that are
**          eligible to be mapped into a user-mode process' address space.
**
** (C) Copyright Alpha Data 2010
*/

#include "df.h"
#include "dflinux.h"

DfMmappableRegion*
dfMmappableRegionFind(
  DfDeviceObject* pDevObj,
  uint32_t regionTag)
{
  DfMmappableRegion** ppRegion;
  DfMmappableRegion* pRegion;

  dfDebugPrint(3, ("dfMmappableRegionFind: entered, pDevObj=%p regionTag=%lu(0x%lx)\n", pDevObj, (unsigned long)regionTag, (unsigned long)regionTag));

  ppRegion = &pDevObj->system.pMmappableRegions;
  pRegion = pDevObj->system.pMmappableRegions;
  while (1) {
    pRegion = *ppRegion;
    if (NULL == pRegion) {
      break;
    }
    if (regionTag == pRegion->system.regionTag) {
        return pRegion;
    }
    ppRegion = &pRegion->system.pNext;
  }

  return NULL;
}

DfMmappableStatus
dfMmappableRegionRegister(
  DfDeviceObject* pDevObj,
  uint32_t regionTag,
  boolean_t bIsMemory,
  void* pKernelBase,
  size_t kernelSize,
  uint64_t translatedBase,
  uint64_t translatedSize)
{
  const unsigned int pageOffsetMask = ~PAGE_MASK;
  DfMmappableRegion* pRegion;
  unsigned int pageOffset;
  uint64_t limit;

  dfDebugPrint(3, ("dfMmappableRegionRegister: entered, pDevObj=%p regionTag=%lu(0x%lx) pKernelBase=%p kernelSize=0x%08lx_%08lx translatedBase=0x%08lx_%08lx translatedSize=0x%08lx_%08lx\n",
    pDevObj, (unsigned long)regionTag, (unsigned long)regionTag, pKernelBase, dfSplitUint64((uint64_t)kernelSize), dfSplitUint64(translatedBase), dfSplitUint64(translatedSize)));

  /* Sanity check */
  pRegion = dfMmappableRegionFind(pDevObj, regionTag);
  dfAssert(NULL == pRegion);

  dfAssert(!bIsMemory || translatedBase == 0);
  dfAssert(!bIsMemory || translatedSize == 0);

  pRegion = dfMalloc(sizeof(*pRegion));
  if (NULL == pRegion) {
    return DfMmappableNoMemory;
  }
  pRegion->system.regionTag = regionTag;
  pRegion->system.bIsMemory = bIsMemory;
  if (bIsMemory) {
    translatedBase = virt_to_phys(pKernelBase);
    translatedSize = kernelSize;
  }
  pageOffset = ((unsigned int)translatedBase) & pageOffsetMask;
  dfAssert(translatedSize != 0);
  limit = translatedBase + translatedSize - 1;
  pRegion->system.pfnStart = (unsigned long)(translatedBase >> PAGE_SHIFT);
  pRegion->system.pfnEnd = (unsigned long)(limit >> PAGE_SHIFT);
  pRegion->system.pageOffset = pageOffset;
  dfAssert(sizeof(unsigned long) >= sizeof(size_t));
  pRegion->system.kernelSize = kernelSize;
  pRegion->system.pNext = pDevObj->system.pMmappableRegions;
  pDevObj->system.pMmappableRegions = pRegion;
  return DfMmappableSuccess;
}

DfMmappableStatus
dfMmappableRegionUnregister(
	DfDeviceObject* pDevObj,
  uint32_t regionTag)
{
  DfMmappableRegion** ppRegion;
  DfMmappableRegion* pRegion;

  dfDebugPrint(3, ("dfMmappableRegionUnregister: entered, pDevObj=%p regionTag=%lu(0x%lx)\n", pDevObj, (unsigned long)regionTag, (unsigned long)regionTag));

  ppRegion = &pDevObj->system.pMmappableRegions;
  pRegion = pDevObj->system.pMmappableRegions;
  while (1) {
    pRegion = *ppRegion;
    if (NULL == pRegion) {
      break;
    }
    if (regionTag == pRegion->system.regionTag) {
      *ppRegion = pRegion->system.pNext;
      dfFree(pRegion);
      return DfMmappableSuccess;
    } else {
      ppRegion = &pRegion->system.pNext;
    }
  }

  return DfMmappableInvalidRegion;
}
