/*
** File: dma.c  
** Project: Linux Driver Framework
** Purpose: Implementation of DMA mapper functions
**
** (C) Copyright Alpha Data 2010, 2015
*/

#include "df.h"
#include "dflinux.h"

/* Error injection for testing robustness to IOMMU / bounce buffer exhaustion/scarcity */
/* Enable at most ONE of the below types of error injection */
#if 0
/* Inject an error on the first page of a mapping operation */
/* The number represents the Nth call to dfDmaMapperSetup() that should fail with a DMA mapping error on the first page */
# define INJECT_MAPPING_ERROR_ON_0TH_ENTRY (25)
static atomic_t g_dmaTransferCount = ATOMIC_INIT(INJECT_MAPPING_ERROR_ON_0TH_ENTRY);
#endif
#if 0
/* Inject an error after N successful dma_map_page() calls */
/* The number represents the Nth call to dma_map_page() that should be considered to fail */
# define INJECT_MAPPING_ERROR_AFTER_PAGES (1000)
static atomic_t g_dmaPageMapCount = ATOMIC_INIT(INJECT_MAPPING_ERROR_AFTER_PAGES);
#endif

DfDmaBuffer*
dfDmaBufferAllocate(
  DfDeviceObject* pDevObj,
  size_t size,
  size_t alignment)
{
  DfDmaBuffer* pDmaBuffer = NULL;
  uint8_t* pKernelAddress;
  uint8_t* pAlignedAddress;
  dma_addr_t busAddress;
  size_t extraSize, offset;

  dfAssert(pDevObj->hardwareId.busType == DfBusPci);
  extraSize = size + alignment - 1;
  dfAssert(extraSize >= size);
#if 0
  if (size > g_dfDmaBufferMaxLength) {
    dfDebugPrint(0, ("*** dfDmaBufferAllocate: attempt to allocate more than %lu(0x%lx) bytes, pDevObj=%p size=%lu(0x%lx) extraSize=%lu(0x%lx)\n",
      (unsigned long)g_dfDmaBufferMaxLength, (unsigned long)g_dfDmaBufferMaxLength, (void*)pDevObj,
      (unsigned long)size, (unsigned long)size, (unsigned long)extraSize, (unsigned long)extraSize));
    return NULL;
  }
#endif
  pDmaBuffer = (DfDmaBuffer*)dfMalloc(sizeof(*pDmaBuffer));
  if (NULL == pDmaBuffer) {
    dfDebugPrint(0, ("*** dfDmaBufferAllocate: failed to allocate DfDmaBuffer object, pDevObj=%p\n", pDevObj));
    return NULL;
  }
  pKernelAddress = dma_alloc_coherent(&pDevObj->system.device.pPci->dev, extraSize, &busAddress, GFP_KERNEL);
  if (NULL == pKernelAddress) {
    dfFree(pDmaBuffer);
    dfDebugPrint(0, ("*** dfDmaBufferAllocate: failed to allocate DMA memory, pDevObj=%p\n", (void*)pDevObj));
    return NULL;
  }
  pAlignedAddress = DF_ALIGN_POINTER(pKernelAddress, alignment);
  offset = pAlignedAddress - pKernelAddress;
  pDmaBuffer->size = size;
  pDmaBuffer->busAddress = (uint64_t)busAddress + offset;
  pDmaBuffer->pKernelAddress = pKernelAddress + offset;
  pDmaBuffer->system.pDevObj = pDevObj;
  pDmaBuffer->system.unaligned.size = extraSize;
  pDmaBuffer->system.unaligned.busAddress = (uint64_t)busAddress;
  pDmaBuffer->system.unaligned.pKernelAddress = pKernelAddress;
  return pDmaBuffer;
}

void
dfDmaBufferFree(
  DfDmaBuffer* pDmaBuffer)
{
  DfDeviceObject* pDevObj;

  pDevObj = pDmaBuffer->system.pDevObj;
  dfAssert(pDevObj->hardwareId.busType == DfBusPci);
  dma_free_coherent(&pDevObj->system.device.pPci->dev, pDmaBuffer->system.unaligned.size, pDmaBuffer->system.unaligned.pKernelAddress, pDmaBuffer->system.unaligned.busAddress);
  dfFree(pDmaBuffer);
}

boolean_t
dfDmaMapperAllocate(
  DfDeviceObject* pDevObj,
  unsigned int numberOfMaps,
  unsigned int mapLength,          /* In blocks */
  boolean_t b64BitAddressing,
  size_t maxMapSize,
  unsigned int* pActualMapLength)  /* In blocks */
{
  const uint64_t dmaMaskAll64Bits  = DMA_BIT_MASK(64);
  const uint64_t dmaMaskOnly32Bits = DMA_BIT_MASK(32);
  unsigned int i;
  DfLinuxDmaMapperContext* pMapArray = NULL;

  dfAssert(pDevObj->hardwareId.busType == DfBusPci);

  if (pDevObj->system.dma.pMapArray != NULL) {
    dfDebugPrint(0, ("*** dfDmaMapperAllocate: attempt to allocate mapper twice, pDevObj=%p\n", pDevObj));
    return FALSE;
  }

  pMapArray = (DfLinuxDmaMapperContext*)dfMalloc(numberOfMaps * sizeof(DfLinuxDmaMapperContext));
  if (NULL == pMapArray) {
    dfDebugPrint(0, ("*** dfDmaMapperAllocate: failed to allocate mapper context array, pDevObj=%p\n", pDevObj));
    return FALSE;
  }
  pDevObj->system.dma.pMapArray = pMapArray;
  pDevObj->system.dma.numMap = numberOfMaps;
  pDevObj->system.dma.maxMapRegister = mapLength;
  pDevObj->system.dma.maxMapSize = maxMapSize;
  for (i = 0; i < numberOfMaps; i++) {
    pMapArray[i].pDeviceObject = pDevObj;
    pMapArray[i].index = i;
    pMapArray[i].pTable = NULL;
#if DF_DBG_BUILD
    pMapArray[i].curr.bIdle = TRUE;
#endif
  }

  for (i = 0; i < numberOfMaps; i++) {
    pMapArray[i].pTable = (DfDmaMapperNode*)dfMalloc(mapLength * sizeof(DfDmaMapperNode));
    if (NULL == pMapArray[i].pTable) {
      dfDebugPrint(0, ("*** dfDmaMapperAllocate: failed to allocate table for map %lu, pDevObj=%p mapLength=%lu",
		       (unsigned long)i, pDevObj, (unsigned long)mapLength));
      goto failed;
    }
  }

  if (b64BitAddressing) {
    if (!pci_set_dma_mask(pDevObj->system.device.pPci, dmaMaskAll64Bits)) {
      dfDebugPrint(1, ("dfDmaMapperAllocate: pDevObj=%p 64-bit DMA mask supported by system\n", pDevObj));
    } else {
      dfDebugPrint(1, ("+++ dfDmaMapperAllocate: pDevObj=%p 64-bit DMA mask not supported by system\n", pDevObj));
      if (pci_set_dma_mask(pDevObj->system.device.pPci, dmaMaskOnly32Bits)) {
        dfDebugPrint(0, ("*** dfDmaMapperAllocate: pDevObj=%p 32-bit DMA mask not supported by system\n", pDevObj));
        goto failed;
      }
    }
    if (!pci_set_consistent_dma_mask(pDevObj->system.device.pPci, dmaMaskAll64Bits)) {
      dfDebugPrint(1, ("dfDmaMapperAllocate: pDevObj=%p 64-bit consistent DMA mask supported by system\n", pDevObj));
    } else {
      dfDebugPrint(1, ("+++ dfDmaMapperAllocate: pDevObj=%p 64-bit consistent DMA mask not supported by system\n", pDevObj));
      if (pci_set_consistent_dma_mask(pDevObj->system.device.pPci, dmaMaskOnly32Bits)) {
        dfDebugPrint(0, ("*** dfDmaMapperAllocate: pDevObj=%p 32-bit consistent DMA mask not supported by system\n", pDevObj));
        goto failed;
      }
    }
  } else {
    if (pci_set_dma_mask(pDevObj->system.device.pPci, dmaMaskOnly32Bits)) {
      dfDebugPrint(0, ("*** dfDmaMapperAllocate: pDevObj=%p 32-bit DMA mask not supported by system\n", pDevObj));
      goto failed;
    }
    if (pci_set_consistent_dma_mask(pDevObj->system.device.pPci, dmaMaskOnly32Bits)) {
      dfDebugPrint(0, ("*** dfDmaMapperAllocate: pDevObj=%p 32-bit consistent DMA mask not supported by system\n", pDevObj));
      goto failed;
    }
  }
  *pActualMapLength = mapLength;
  dfDebugPrint(3, ("dfDmaMapperAllocate: pDevObj=%p numberOfMaps=%lu mapLength=%lu\n",
		   pDevObj, (unsigned long)numberOfMaps, (unsigned long)mapLength));
  return TRUE;

 failed:
  if (NULL != pMapArray) {
    for (i = 0; i < numberOfMaps; i++) {
      if (NULL != pMapArray[i].pTable) {
        dfFree(pMapArray[i].pTable);
        pMapArray[i].pTable = NULL;
      }
    }
    dfFree(pMapArray);
    pDevObj->system.dma.pMapArray = NULL;
  }
  return FALSE;
}

extern void
dfDmaMapperFree(
  DfDeviceObject* pDevObj)
{
  DfLinuxDmaMapperContext* pMapArray;
  unsigned int i, numberOfMaps;

  dfAssert(pDevObj->hardwareId.busType == DfBusPci);

  pMapArray = pDevObj->system.dma.pMapArray;
  numberOfMaps = pDevObj->system.dma.numMap;
  if (NULL != pMapArray) {
    for (i = 0; i < numberOfMaps; i++) {
      if (NULL != pMapArray[i].pTable) {
        dfFree(pMapArray[i].pTable);
        pMapArray[i].pTable = NULL;
      }
    }
    dfFree(pMapArray);
    pDevObj->system.dma.pMapArray = NULL;
  } else {
    dfDebugPrint(0, ("*** dfDmaMapperFree: attempting to free mapper when mapper not allocated, pDevObj=%p\n", pDevObj));
  }
}

extern boolean_t
dfDmaMapperSetup(
  DfDeviceObject* pDevObj,
  unsigned int mapIndex,
  DfBufferDescription* pBufferDescription,
  DfDmaMapperCallback* pCallback,
  void* pCallbackContext,
  boolean_t bWriteToDevice,
  boolean_t bFixedLocalAddress,
  DfDmaMapperSlice* pSlices,
  size_t numSlice,
  DfDmaMapperPosition* pStartPosition)
{
  const unsigned int pageOffsetMask = ~PAGE_MASK;
  struct device* pDev;
  DfDmaMapperSlice* pSlice;
  DfDmaMapperNode* pNode;
  DfLinuxDmaMapperContext* pMapperContext;
  DfDmaMapperPosition newPosition;
  struct page** pPages;
  struct page* p;
  size_t sliceIndex, slicePosition;
  size_t sliceOffset, sliceLength, sliceRemaining = 0;
  uint64_t sliceLocalAddress;
  size_t pageIndex, regionIndex;
  unsigned int pageOffset, pageRemaining;
  size_t chunk, maxMapSize;
  unsigned int bufferPageOffset;
  unsigned int numEntry, maxEntry;
  dma_addr_t busAddress;
#if defined(INJECT_MAPPING_ERROR_AFTER_PAGES) || defined(INJECT_MAPPING_ERROR_ON_0TH_ENTRY)
  boolean_t bInjectError;
#else
  const boolean_t bInjectError = FALSE;
#endif

  dfAssert(pDevObj->hardwareId.busType == DfBusPci);
  dfAssert(numSlice > 0);
  dfAssert(mapIndex < pDevObj->system.dma.numMap);

  maxMapSize = pDevObj->system.dma.maxMapSize;

  pDev = &pDevObj->system.device.pPci->dev;

  bufferPageOffset = (unsigned int)(uintptr_t)pBufferDescription->pAddress & pageOffsetMask;

  sliceIndex = pStartPosition->sliceIndex;
  dfAssert(sliceIndex < numSlice);
  slicePosition = pStartPosition->slicePosition;

  pMapperContext = &pDevObj->system.dma.pMapArray[mapIndex];
  maxEntry = pDevObj->system.dma.maxMapRegister;
  pNode = pMapperContext->pTable;

  numEntry = 0;
  while (sliceIndex < numSlice && numEntry < maxEntry) {
    /* Get current offset into current slice, and remaining bytes in current slice */
    pSlice = pSlices + sliceIndex;
    sliceOffset = pSlice->offset + slicePosition + bufferPageOffset;
    sliceLength = pSlice->length;
    dfAssert(slicePosition < sliceLength);
    sliceRemaining = sliceLength - slicePosition;
    sliceLocalAddress = pSlice->localAddress;
    if (!bFixedLocalAddress) {
      sliceLocalAddress += slicePosition;
    }
    /* Find corresponding page index in region and region index */
    pageOffset = (unsigned int)sliceOffset & pageOffsetMask;
    pageRemaining = PAGE_SIZE - pageOffset;
    pageIndex = sliceOffset >> PAGE_SHIFT;
    regionIndex = pageIndex / DF_BUFFER_DESCRIPTION_REGION_SIZE;
    pageIndex &= (DF_BUFFER_DESCRIPTION_REGION_SIZE - 1);
    /* Get corresponding page */
    pPages = pBufferDescription->system.region[regionIndex];
    p = pPages[pageIndex];
    /* Calculate how many bytes we can map this time around */
    chunk = maxMapSize;
    if (chunk > sliceRemaining) {
      chunk = sliceRemaining;
    }
    if (chunk > pageRemaining) {
      chunk = pageRemaining;
    }
    /* Map the page chunk */
#if defined(INJECT_MAPPING_ERROR_AFTER_PAGES)
    bInjectError = atomic_dec_and_test(&g_dmaPageMapCount) ? TRUE : FALSE;
#endif
#if defined(INJECT_MAPPING_ERROR_ON_0TH_ENTRY)
    if (numEntry == 0) {
      bInjectError = atomic_dec_and_test(&g_dmaTransferCount) ? TRUE : FALSE;
    } else {
      bInjectError = FALSE;
    }
#endif
    busAddress = dma_map_page(pDev, p, pageOffset, chunk, bWriteToDevice ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
    if (bInjectError || dma_mapping_error(pDev, busAddress)) {
      if (numEntry) {
        dfDebugPrint(1, ("dfDmaMapperSetup: dma_map_page failed on entry %u\n", numEntry));
      } else {
        dfDebugPrint(0, ("*** dfDmaMapperSetup: dma_map_page failed on zero'th entry\n", numEntry));
      }
      break;
    }
    pNode->busAddress = busAddress;
    pNode->length = chunk;
    pNode->localAddress = sliceLocalAddress;
    pNode++;
    /* Increment pointers etc. */
    slicePosition += chunk;
    if (slicePosition == sliceLength) {
      /* Reached end of current slice */
      sliceIndex++;
      slicePosition = 0;
    }
    numEntry++;
  }

  pMapperContext->curr.bWriteToDevice = bWriteToDevice;
  pMapperContext->curr.numEntry = numEntry;
  newPosition.sliceIndex = sliceIndex;
  newPosition.slicePosition = slicePosition;
  
  /* Finished, so call the client's callback function */
  pCallback(
    pDevObj,
    pMapperContext->index,
    pMapperContext->pTable,
    numEntry,
    &newPosition,
    pCallbackContext);

  return (numEntry > 0) ? TRUE : FALSE;
}

extern void
dfDmaMapperTeardown(
  DfDeviceObject* pDevObj,
  unsigned int mapIndex)
{
  struct device* pDev;
  DfDmaMapperNode* pNode;
  DfLinuxDmaMapperContext* pMapperContext;
  unsigned int i, numEntry;
  boolean_t bWriteToDevice;

  dfAssert(pDevObj->hardwareId.busType == DfBusPci);

  pDev = &pDevObj->system.device.pPci->dev;

  pMapperContext = &pDevObj->system.dma.pMapArray[mapIndex];
  pNode = pMapperContext->pTable;
  numEntry = pMapperContext->curr.numEntry;
  bWriteToDevice = pMapperContext->curr.bWriteToDevice;

  for (i = 0; i < numEntry; i++) {
    dma_unmap_page(pDev, pNode->busAddress, pNode->length, bWriteToDevice ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
    pNode++;
  }
}
