/*
** File: directif.c  
** Project: Linux Driver Framework
** Purpose: Mechanism by which OS-independent driver code can obtain a direct-call
**          interface to another driver.
**
** (C) Copyright Alpha Data 2010
*/

#include "df.h"
#include "dflinux.h"

static DfExportedInterface*
findExportedInterface(
  DfDeviceObject* pDevObj,
  const char* pInterfaceName)
{
  DfExportedInterface* pExportedInterface;

  pExportedInterface = pDevObj->system.pExportedInterfaces;
  while (NULL != pExportedInterface) {
    if (strcmp(pInterfaceName, pExportedInterface->name) == 0) {
      break;
    }
    pExportedInterface = pExportedInterface->pNext;
  }

  return pExportedInterface;
}

DF_ATTR_PASSIVE_CONTEXT
boolean_t
dfImportInterface(
  DfDeviceObject* pDevObj,
  const char* pInterfaceName,
  unsigned int size,
  void* pInterfaceStruct,
  void** pInterfaceContext)
{
  DfImportedInterface* pInterface = NULL;
  DfExportedInterface* pExportedInterface = NULL;

  dfAssertPassiveContext();

  if (size > DF_MAX_INTERFACE_SIZE) {
    return FALSE;
  }

  /* Attempt to find interface locally */
  pExportedInterface = findExportedInterface(pDevObj, pInterfaceName);
  if (NULL != pExportedInterface) {
    /* Found the interface locally */
    dfDebugPrint(1, ("dfImportInterface: found interface '%s' locally.\n", pInterfaceName));
    if (size != pExportedInterface->size) {
      dfDebugPrint(0, ("*** dfImportInterface: interface size mismatch: 0x%lx vs. 0x%lx\n",
        (unsigned long)size, (unsigned long)pExportedInterface->size));
      return FALSE;
    }
    pInterface = dfMalloc(sizeof(*pInterface));
    if (NULL == pInterface) {
      dfDebugPrint(0, ("*** dfImportInterface: failed to allocate imported interface.\n"));
      return FALSE;
    }
    dfStrCpy(pInterface->name, DF_ARRAY_LENGTH(pInterface->name), pInterfaceName);
    pInterface->pNext = pDevObj->system.pImportedInterfaces;
    pDevObj->system.pImportedInterfaces = pInterface;
    *pInterfaceContext = pExportedInterface->pInterfaceContext;
    memcpy(pInterfaceStruct, pExportedInterface->pInterfaceStruct, size);
    return TRUE;
  } else {
    return FALSE;
  }
}

DF_ATTR_PASSIVE_CONTEXT
boolean_t
dfReleaseInterface(
  DfDeviceObject* pDevObj,
  const char* pInterfaceName)
{
  DfImportedInterface** ppInterface;
  DfImportedInterface* pInterface;

  dfAssertPassiveContext();

  dfDebugPrint(1, ("dfReleaseInterface: pDevObj=%p pInterfaceName=%s\n", (void*)pDevObj, pInterfaceName));

  ppInterface = &pDevObj->system.pImportedInterfaces;
  pInterface = *ppInterface;
  while (NULL != pInterface) {
    if (strcmp(pInterface->name, pInterfaceName) == 0) {
      *ppInterface = pInterface->pNext;
      break;
    }
    ppInterface = &pInterface->pNext;
    pInterface = *ppInterface;
  }
  
  if (NULL != pInterface) {
    dfFree(pInterface);
    return TRUE;
  } else {
    dfDebugPrint(0, ("*** dfReleaseInterface: failed to find interface '%s'\n", pInterfaceName));
    return FALSE;
  }
}

DF_ATTR_PASSIVE_CONTEXT
boolean_t
dfExportInterface(
  DfDeviceObject* pDevObj,
  const char* pInterfaceName,
  unsigned int size,
  void* pInterfaceStruct,
  void* pInterfaceContext)
{
  DfExportedInterface* pInterface = NULL;

  dfAssertPassiveContext();

  if (size > DF_MAX_INTERFACE_SIZE) {
    return FALSE;
  }

  /* First attempt to unregister the interface, to prevent duplicates */
  dfUnexportInterface(pDevObj, pInterfaceName);

  pInterface = (DfExportedInterface*)dfMalloc(sizeof(*pInterface));
  if (NULL == pInterface) {
    goto failed;
  }
  dfStrCpy(pInterface->name, DF_ARRAY_LENGTH(pInterface->name), pInterfaceName);
  pInterface->pDevObj = pDevObj;
  pInterface->size = size;
  pInterface->pInterfaceStruct = pInterfaceStruct;
  pInterface->pNext = pDevObj->system.pExportedInterfaces;
  pInterface->pInterfaceContext = pInterfaceContext;
  pDevObj->system.pExportedInterfaces = pInterface;

  dfDebugPrint(2, ("dfExportInterface: registered interface '%s', pInterface=%p\n",
    pInterface->name, (void*)pInterface));

  return TRUE;

failed:
  if (NULL != pInterface) {
    dfFree(pInterface);
  }
  return FALSE;
}

DF_ATTR_PASSIVE_CONTEXT
boolean_t
dfUnexportInterface(
  DfDeviceObject* pDevObj,
  const char* pInterfaceName)
{
  DfExportedInterface** ppInterface = &pDevObj->system.pExportedInterfaces;
  DfExportedInterface* pInterface;

  dfAssertPassiveContext();

  pInterface = *ppInterface;
  while (NULL != pInterface) {
    if (strcmp(pInterface->name, pInterfaceName) == 0) {
      break;
    } else {
      ppInterface = &pInterface->pNext;
      pInterface = *ppInterface;
    }
  }

  if (NULL != pInterface) {
    *ppInterface = pInterface->pNext;
    dfFree(pInterface);
    return TRUE;
  } else {
    return FALSE;
  }
}
