/*
** File: v5clksynth.c  
** Project: ADB3 core driver
** Purpose: Common functions for Virtex-5 DCM clock synthesizer
**
** (C) Copyright Alpha Data 2011-2013
**
** NOTES
**
** 1. Clock word layout is:
**
**   d0[15:0]   M multiplier value - 1
**   d0[31:16]  D divider value - 1
**   d1         (unused)
**   d2         (unused)
*/

#include <df.h>

#include "v5clksynth.h"
#include "v5clksynth_common.h"

/* Min. & max. CLKFX output frequency for Virtex-5 -1 speed grade */
static const uint32_t g_minFrequency = 31999000U, g_maxFrequency = 140001000U;

V5ClkSynthClockWordStatus
v5ClkSynthClockWord(
  uint32_t flags,
	uint32_t refFrequency,
	uint64_t frequencyReq,
	uint64_t frequencyMin,
	uint64_t frequencyMax,
	CoreClockWord* pClockWord)
{
	uint32_t m, bestM = 0, d, bestD = 0; /* init. to avoid compiler whine */
	uint32_t f, bestFrequency = 0; /* init. to avoid compiler whine */
	uint32_t error, bestError = (uint32_t)-1;

	dfDebugPrint(5, ("v5ClkSynthClockWord: refFrequency=%lu reqFrequency=" DF_FMT_U64 "\n",
    (unsigned long)refFrequency, (unsigned long long)frequencyReq));

  if (refFrequency == 0 || refFrequency > 268000000) {
		return V5ClkSynthClockWordStatusGeneralFailure;
	}
	if (frequencyReq < g_minFrequency || frequencyReq > g_maxFrequency) {
		return V5ClkSynthClockWordStatusOutOfRange;
	}

	/* Won't overflow assuming (refFrequency <= 268e6) */
	for (d = 1; d <= 32; d++) {
		for (m = 2; m <= 32; m++) {
			f = ((refFrequency >> 1) * m / d) << 1;
      if ((flags & CoreClockMinimumFrequency) && f < frequencyMin) {
        continue;
      }
      if ((flags & CoreClockMaximumFrequency) && f > frequencyMax) {
        continue;
      }
			if (f > (uint32_t)frequencyReq) {
				error = f - (uint32_t)frequencyReq;
			} else {
				error = (uint32_t)frequencyReq - f;
			}
			if (error < bestError) {
				bestError = error;
				bestFrequency = f;
				bestM = m;
				bestD = d;
			}
		}
	}

	if (bestError == (uint32_t)-1) {
		/* Failed to find a suitable value */
		return V5ClkSynthClockWordStatusGeneralFailure;
	}

	pClockWord->frequency = bestFrequency;
	pClockWord->d0 = ((bestD - 1) << 16) | (bestM - 1);

	dfDebugPrint(5, ("v5ClkSynthClockWord: bestFrequency=%lu clockWord=0x%08lx\n", (unsigned long)bestFrequency, (unsigned long)pClockWord->d0));

	return V5ClkSynthClockWordStatusSuccess;
}

boolean_t
v5ClkSynthInit(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx,
  unsigned int lclkRef,
  uint32_t* pFOut )
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32, lclkStat3;
  uint16_t lclkD, lclkM;
  uint64_t frequency;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfTimerInit(&pV5ClkSynthCtx->timer, pDevCtx);
  pV5ClkSynthCtx->save.bLastValid = FALSE;

  if (lclkRef == 0 || lclkRef > 268000000) {
    dfDebugPrint(0, ("*** v5ClkSynthInit: reference clock frequency %u Hz is out of range\n", lclkRef));
	}

  /* Figure out current LCLK frequency */
  lclkStat3 = dfPciMemRead32(hModel, &pModelRegs->model.v5ClkSynth.lclkStat3);
  if (lclkStat3 & 0x4U) {
    val32 = dfPciMemRead32(hModel, &pModelRegs->model.v5ClkSynth.lclkStat2);
  } else {
    val32 = dfPciMemRead32(hModel, &pModelRegs->model.v5ClkSynth.lclkStat1);
  }
  lclkM = (uint16_t)(val32 + 1U);
  if (lclkM < 2 || lclkM > 32) {
    dfDebugPrint(0, ("*** v5ClkSynthInit: M multipler value %u is out of range\n", (unsigned int)lclkM));
    return FALSE;
  }
  lclkD = (uint16_t)((val32 >> 16) + 1U);
  if (lclkD < 1 || lclkD > 32) {
    dfDebugPrint(0, ("*** v5ClkSynthInit: D divider value %u is out of range\n", (unsigned int)lclkD));
    return FALSE;
  }
	/* Won't overflow assuming (lclkRef <= 268e6) */
  frequency = ((lclkRef >> 1) * lclkM / lclkD) << 1;
  if (frequency < g_minFrequency || frequency > g_maxFrequency) {
    dfDebugPrint(0, ("*** v5ClkSynthInit: initial LCLK frequency is " DF_FMT_U64 " Hz, out of range\n", (unsigned long long)frequency));
  } else {
    dfDebugPrint(1, ("v5ClkSynthInit: initial LCLK frequency is " DF_FMT_U64 " Hz\n", (unsigned long long)frequency));
  }
  *pFOut = (uint32_t)frequency; /* cast is safe due to above range checks */

  return TRUE;
}

void
v5ClkSynthUninit(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx)
{
  dfTimerUninit(&pV5ClkSynthCtx->timer);
}

static void
onCheckV5ClkSynth(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  V5ClkSynthProgramRequest* pRequest = (V5ClkSynthProgramRequest*)pCallbackContext;
  V5ClkSynthDeviceContext* pV5ClkSynthCtx = pRequest->pV5ClkSynthCtx;
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
	uint32_t val32, mask;
  V5ClkSynthProgramStatus status = V5ClkSynthProgramStatusSuccess;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;
  val32 = dfPciMemRead32(hModel, &pModelRegs->model.v5ClkSynth.lclkStat3);

  dfDebugPrint(5, ("onCheckV5ClkSynth: LCLKSTAT3=0x%lx\n", (unsigned long)val32));	
	mask = (val32 & 0x4U) ? 0x2U : 0x1U;
	if (!(val32 & mask)) {
		/* DCM not locked */
		status = V5ClkSynthProgramStatusHardwareError;
  }
  if (val32 & 0x28U) {
    /* Control logic busy or write of DCM parameters timed out */
    status = V5ClkSynthProgramStatusHardwareError;
  }
  if (val32 & 0x10U) {
    /* Error */
    status = V5ClkSynthProgramStatusHardwareError;
  }
  if (V5ClkSynthProgramStatusSuccess == status) {
    pV5ClkSynthCtx->save.bLastValid = TRUE;
    pV5ClkSynthCtx->save.last = pRequest->clockWord;
  }
  pRequest->pCallback(pDevCtx, pV5ClkSynthCtx, pRequest, status);
}

V5ClkSynthProgramStatus
v5ClkSynthProgram(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx,
  V5ClkSynthProgramRequest* pRequest,
  const CoreClockWord* pClockWord,
  V5ClkSynthProgramCallback* pCallback)
{
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  pRequest->pV5ClkSynthCtx = pV5ClkSynthCtx;
  pRequest->pCallback = pCallback;
  pRequest->clockWord = *pClockWord;

  dfPciMemWrite32(hModel, &pModelRegs->model.v5ClkSynth.lclkCtl, pClockWord->d0 | (0x1U << 31));
  dfPciMemRead32(hModel, &pModelRegs->model.v5ClkSynth.lclkStat3);
  dfTimerSchedule(&pV5ClkSynthCtx->timer, dfTimeGet() + dfMicrosecondsToTime(50000), onCheckV5ClkSynth, (void*)pRequest);

  return V5ClkSynthProgramStatusSuccess;
}

static void
onProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx,
  V5ClkSynthProgramRequest* pRequest,
  V5ClkSynthProgramStatus status)
{
  V5ClkSynthProgramRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, V5ClkSynthProgramRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

V5ClkSynthProgramStatus
v5ClkSynthProgramSync(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx,
  const CoreClockWord* pClockWord)
{
  V5ClkSynthProgramRequestSync req;
  V5ClkSynthProgramStatus status;

  dfEventInit(&req.ev);
  status = v5ClkSynthProgram(pDevCtx, pV5ClkSynthCtx, &req.request, pClockWord, onProgramDone);
  if (V5ClkSynthProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

extern void
v5ClkSynthSaveState(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx)
{
  /* Nothing to do yet */
}

V5ClkSynthProgramStatus
v5ClkSynthRestoreState(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx)
{
  V5ClkSynthProgramStatus status = V5ClkSynthProgramStatusSuccess;

  /* If the clock synthesizer had been programmed before power-down, reprogram it with the same thing */
  if (pV5ClkSynthCtx->save.bLastValid) {
    status = v5ClkSynthProgramSync(pDevCtx, pV5ClkSynthCtx, &pV5ClkSynthCtx->save.last);
  }

  return status;
}

ClockProgramStatus
v5ClkSynthMapProgramStatus(
  V5ClkSynthProgramStatus status)
{
  switch (status) {
  case V5ClkSynthProgramStatusSuccess:
    return ClockProgramSuccess;

  case V5ClkSynthProgramStatusHardwareError:
    return ClockProgramHardwareError;

  case V5ClkSynthProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
v5ClkSynthMapClockWordStatus(
  V5ClkSynthClockWordStatus status)
{
  switch (status) {
  case V5ClkSynthClockWordStatusSuccess:
    return ClockWordSuccess;

  case V5ClkSynthClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case V5ClkSynthClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
