/*
** File: si5338.c  
** Project: ADB3 core driver
** Purpose: Functions for programming the SI5338 clock generator.
**
** (C) Copyright Alpha Data 2011, 2015
**
** SI5338 frequency formulae are as follows:
**
**   Fpfd = Fref / P1
**   Fvco = Fpfd * MSn
**   Foutx = Fvco / (MSx * Rx) = Fin * MSn / (P * MSx * Rx)
**
** where x is the multisynth index in the range 0..3.
**
** si5338FullConfiguration
** -----------------------
** The function si5338FullConfiguration can be called when the driver wants
** to perform a complete reconfiguration of the SI5338 (i.e. including the
** PLL).
**
** si5338Interrogate
** -----------------
** The driver finds the VCO frequency from the reference clock frequency and
** by reading the SI5338 registers. The calculated VCO frequency is then used
** as the basis for subsequent multisynth-related calculations when changing
** the frequencies of the four clock outputs via si5338ClockProgram.
**
** Also calls internal function initFrequencyStepping to set up the glitch-
** free frequency-stepping feature of the SI5338. This allows
** si5338ClockProgramStepping to be used instead of si5338ClockProgram for
** certain outputs of the SI5338.
**
** si5338ClockProgram
** ------------------
** The driver does not change the PLL, input stage or output stage registers.
** The output stage registers are expected to be configured for divide by 1.
** In order to program a new frequency on one of the four clock outputs, the
** driver keeps the VCO frequency constant and changes only the multisynth
** registers for the clock output in question.
** NOTE: if this method of programming is used for a particular multisynth,
** si5338ClockProgramStepping should not be used for the same multisynth.
**
** si5338ClockProgramStepping
** --------------------------
** The driver repeatedly steps the specified multisynth up or down using the
** glitch-free stepping feature of the SI5338 until the desired output
** frequency is reached.
** NOTE: if this method of programming is used for a particular multisynth,
** si5338ClockProgram should not be used for the same multisynth.
**
** si5338SaveState
** ---------------
** Currently does nothing other than invalidate the shadow value of the
** SI5338's page register, in order to force it to be reestablished when
** si5338RestoreState is called.
**
** si5338RestoreState
** ------------------
** Calls internal function initFrequencyStepping and then restores SI5338
** outputs to their last programmed values. Uses the appropriate method
** (si5338ClockProgram vs si5338ClockProgramStepping) for setting the output
** frequency.
*/

#include <df.h>

#include "coreif.h"
#include "i2c_common.h"
#include "si5338_common.h"

/* From SI5338 data sheet */
static const uint32_t g_fVcoMin = 2200000000U;        /* Minimum permitted VCO frequency */    
static const uint32_t g_fVcoMax = 2840000000U;        /* Maximum permitted VCO frequency */

/* Frequency stepping register addresses in SI5338, handily indexed by Multisynth index */
static const uint16_t g_fidp1Addr[4] = {
  SI5338_REGIDX_MS0_FIDP1(0),
  SI5338_REGIDX_MS1_FIDP1(0),
  SI5338_REGIDX_MS2_FIDP1(0),
  SI5338_REGIDX_MS3_FIDP1(0),
};
static const uint16_t g_fidp2Addr[4] = {
  SI5338_REGIDX_MS0_FIDP2(0),
  SI5338_REGIDX_MS1_FIDP2(0),
  SI5338_REGIDX_MS2_FIDP2(0),
  SI5338_REGIDX_MS3_FIDP2(0),
};
static const uint16_t g_fidp3Addr[4] = {
  SI5338_REGIDX_MS0_FIDP3(0),
  SI5338_REGIDX_MS1_FIDP3(0),
  SI5338_REGIDX_MS2_FIDP3(0),
  SI5338_REGIDX_MS3_FIDP3(0),
};

/*
** Register write masks for all SI5338 registers that apply ONLY when writing
** the ClockBuilder configuration to the SI5338. These masks should not be used
** in general when writing the SI5338.
**
** A 0 bit means that a bit's value must be preserved when its register is written.
*/
static const uint8_t g_registerMasks[SI5338_NUM_REG] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1D, 0x00,  /* 0x000 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x008 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x010 */
  0x00, 0x00, 0x00, 0x80, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x018 */
  0xFF, 0xFF, 0xFF, 0xFF, 0x1F, 0x1F, 0x1F, 0x1F,  /* 0x020 */
  0xFF, 0x7F, 0x3F, 0x00, 0x00, 0xFF, 0xFF, 0x3F,  /* 0x028 */
  0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF,  /* 0x030 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0x7F,  /* 0x038 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x040 */
  0xFF, 0x3F, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x048 */
  0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0x7F, 0xFF, 0xFF,  /* 0x050 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F,  /* 0x058 */
  0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x060 */
  0xFF, 0xFF, 0xBF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF,  /* 0x068 */
  0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x070 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x078 */
  0xFF, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x080 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x088 */
  0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x090 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F,  /* 0x098 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x0A0 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x0A8 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0xFF, 0xFF,  /* 0x0B0 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x0B8 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x0C0 */
  0xFF, 0xFF, 0xFF, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x0C8 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x0D0 */
  0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x0D8 */
  0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0F, 0x00,  /* 0x0E0 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x0E8 */
  0x00, 0x0F, 0x02, 0x00, 0x00, 0x00, 0x0F, 0x00,  /* 0x0F0 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,  /* 0x0F8 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x100 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x108 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0x110 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,  /* 0x118 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x120 */
  0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0x00, 0xFF,  /* 0x128 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x130 */
  0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0x00, 0xFF,  /* 0x138 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x140 */
  0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0x00, 0xFF,  /* 0x148 */
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,  /* 0x150 */
  0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0x00         /* 0x158 */
};

static Si5338Status
mapI2cStatus(
  I2cStatus status)
{
  switch (status) {
  case I2cStatusSuccess:
    return Si5338StatusSuccess;

  case I2cStatusTimeout:
    return Si5338StatusI2CTimeout;

  case I2cStatusHardwareError:
    return Si5338StatusI2CError;

  case I2cStatusInvalidBus:
  case I2cStatusInvalidDevice:
  case I2cStatusInvalidAddress:
  case I2cStatusGeneralFailure:
  default:
    return Si5338StatusGeneralFailure;
  }
}

Si5338ClockWordStatus
si5338ClockWord(
	uint32_t frequencyVCO,
	uint64_t frequencyReq,
	uint64_t frequencyMin,
	uint64_t frequencyMax,
  boolean_t bAllowIntDivRange,
  CoreClockWord* pClockWord)
{
  const uint32_t fOutMin = frequencyVCO / 568;   /* Minimum possible output frequency = fVCO / 568 */
  const uint32_t fOutMaxFrac = frequencyVCO / 8; /* Maximum possible output frequency = fVCO / 8 with fractional divider */
  const uint32_t fOutMaxInt = frequencyVCO / 4;  /* Maximum possible output frequency = fVCO / 4 with integer divider */
  uint32_t msxaValue, msxbValue, msxcValue;      /* MSx value as (a + b/c) */
  uint32_t fOut, fError, fErrorBest, msxaValueTmp;
  uint64_t tmp;

	dfDebugPrint(3, ("si5338ClockWord: bAllowIntDivRange=%u frequencyVCO=%lu reqFrequency=" DF_FMT_U64 "\n",
    (unsigned int)bAllowIntDivRange, (unsigned long)frequencyVCO, (unsigned long long)frequencyReq));

  if (frequencyVCO < g_fVcoMin || frequencyVCO > g_fVcoMax) {
    /* VCO frequency should be in the range 2.2 to 2.84 GHz */
		return Si5338ClockWordStatusGeneralFailure;
  }

  if (0 == frequencyReq || frequencyReq < (fOutMin - 1000)) {
	  return Si5338ClockWordStatusOutOfRange;
  }

  if (bAllowIntDivRange) {
    if (frequencyReq > (fOutMaxInt + 1000)) {
		  return Si5338ClockWordStatusOutOfRange;
    }
  } else {
    if (frequencyReq > (fOutMaxFrac + 1000)) {
		  return Si5338ClockWordStatusOutOfRange;
    }
  }

  /* Find MSx value, as (a + b/c) */
  msxaValue = frequencyVCO / (uint32_t)frequencyReq;
  if (msxaValue > 568U) {
    /* This means that the requested frequency is below the minumum possible */
		return Si5338ClockWordStatusOutOfRange;
  }
  if (msxaValue < 8U) {
    /* This means that the requested frequency is in the integer-only divider range */

    if (!(bAllowIntDivRange && msxaValue >= 4U)) {
      /* This means that the requested frequency is above the maximum possible */
  		return Si5338ClockWordStatusOutOfRange;
    }

    /* Integer-only divider value: b = 0, c = 1 */
    msxbValue = 0U;
    msxcValue = 1U;

    /* Look for closest frequency with integer divider */
    fErrorBest = 0xFFFFFFFFU;
    for (msxaValueTmp = 4U; msxaValueTmp <= 8U; msxaValueTmp++) {
      fOut = frequencyVCO / msxaValueTmp;
      fError = (fOut > (uint32_t)frequencyReq) ? fOut - (uint32_t)frequencyReq : (uint32_t)frequencyReq - fOut;
      if (fError < fErrorBest) {
        pClockWord->frequency = fOut;
        fErrorBest = fError;
        msxaValue = msxaValueTmp;
      }
    }
  } else {
    /* Fractional divider value */
    msxbValue = frequencyVCO % (uint32_t)frequencyReq;
    msxcValue = (uint32_t)frequencyReq;

    /* Actual frequency will always be very close to requested frequency */
    pClockWord->frequency = frequencyReq;
  }

  /*
  ** Get MSx_P1, MSx_P2 and MSx_P3 from a, b and c values:
  **
  **   MSx_P1 = (128 * (a * c + b)) / c - 512
  **   MSx_P2 = (128 * (a * c + b)) % c
  **   MSx_P3 = c
  **
  ** where / is a division operator that rounds down to nearest integer
  **       % is a modulus operator
  */
  tmp = (uint64_t)128U * ((uint64_t)msxaValue * (uint64_t)msxcValue + (uint64_t)msxbValue);
  pClockWord->d0 /* MSx_P1 */ = (uint32_t)(dfDivide64By32Rem(tmp, msxcValue, &pClockWord->d1 /* MSx_P2 */)) - 512U;
  pClockWord->d2 /* MSx_P3 */ = msxcValue;                           

	dfDebugPrint(3, ("si5338ClockWord: bestFrequency=%lu p1=0x%08lx p2=0x%08lx p3=0x%08lx\n",
    (unsigned long)frequencyReq, (unsigned long)pClockWord->d0, (unsigned long)pClockWord->d1, (unsigned long)pClockWord->d2));

	return Si5338ClockWordStatusSuccess;
}

Si5338ClockWordStatus
si5338ClockWordStepping(
  Si5338DeviceContext* pSi5338Ctx,
  uint32_t flags,
  unsigned int msIndex,
	uint64_t frequencyReq,
	uint64_t frequencyMin,
	uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  const uint32_t fVco = pSi5338Ctx->vco.frequency;
  const uint32_t fOutMin = fVco / 568; /* Minimum possible output frequency = fVCO / 568 */
  const uint32_t fOutMax = fVco / 8;   /* Maximum possible output frequency = fVCO / 8 */
  int32_t fOut, fStep, fDeviation, fOutPredicted, fError;
  int32_t currentStep, targetStep;

	dfDebugPrint(3, ("si5338ClockWordStepping: msIndex=%u frequencyReq=%lu\n", msIndex, (unsigned long)frequencyReq));

  if (fVco < g_fVcoMin || fVco > g_fVcoMax) {
    /* VCO frequency should be in the range 2.2 to 2.84 GHz */
		return Si5338ClockWordStatusGeneralFailure;
  }

  if (0 == frequencyReq || frequencyReq < fOutMin || frequencyReq > fOutMax) {
		return Si5338ClockWordStatusOutOfRange;
  }

  if (msIndex > 3) {
    return Si5338ClockWordStatusGeneralFailure;
  }

  fOut = (int32_t)pSi5338Ctx->multisynth[msIndex].frequency;

  fStep = (int32_t)pSi5338Ctx->multisynth[msIndex].frequencyStep;

  currentStep = pSi5338Ctx->multisynth[msIndex].currentStep;

  /* Find required frequency deviation from multisynth frequency */
  fDeviation = (int32_t)frequencyReq - (int32_t)fOut;

  /* Find the number of steps required to get closest to the desired frequency */
  if (fDeviation > 0) {
    /* Find step that will be closest to desired output frequency, when desired > current */
    targetStep = (fDeviation + (fStep >> 1) - 1) / fStep;
  } else  {
    /* Find step that will be closest to desired output frequency, when desired < current */
    targetStep = (fDeviation - (fStep >> 1) + 1) / fStep;
  }

  /* Find predicted output frequency */
  fOutPredicted = fOut + targetStep * fStep;

  /*
  ** It is possible that the above calculations result in (fVco/568 <= fOutPredicted <= fVco/8) not being satisfied.
  ** If this happens, we need to do one less step; otherwise, bad things may happen inside the SI5338B such as all
  ** output clocks stopping.
  */
  if (fOutPredicted < (int32_t)fOutMin) {
    /* Go back up a step. */
    targetStep++;
    /* Adjust predicted output frequency up by 1 step */
    fOutPredicted += fStep;
  } else if (fOutPredicted > (int32_t)fOutMax) {
    /* Go back down a step. */
    targetStep--;
    /* Adjust predicted output frequency down by 1 step */
    fOutPredicted -= fStep;
  }

  if ((flags & CoreClockMinimumFrequency) && fOut < (int32_t)fOutMin) {
    /* Reject this set of parameters due to frequency being below minimum requested */
    return Si5338ClockWordStatusOutOfRange;
  }
  if ((flags & CoreClockMaximumFrequency) && fOut > (int32_t)fOutMax) {
    /* Reject this set of parameters due to frequency being above maximum requested */
    return Si5338ClockWordStatusOutOfRange;
  }

  /* Find the error (difference) between predicted and desired frequency */
  fError = fOutPredicted - (int32_t)frequencyReq;

	pClockWord->frequency = fOutPredicted;
  pClockWord->d0 = (uint32_t)targetStep;

  dfDebugPrint(2, ("si5338ClockWordStepping: fRequested=%lu fOut=%lu fDeviation=%ld fOutPredicted=%lu fError=%ld currentStep=%ld targetStep=%ld\n",
    (unsigned long)frequencyReq, (long)fOut, (long)fDeviation, (long)fOutPredicted, (long)fError, (long)currentStep, (long)targetStep));

	return Si5338ClockWordStatusSuccess;
}

static Si5338Status
si5338ReadByteSync(
  Si5338DeviceContext* pSi5338Ctx,
  uint8_t address,
  uint8_t* pVal8)
{
  I2cStatus status;

  status = i2cReadSync(pSi5338Ctx->pI2cCtx, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, address, pVal8);
  dfDebugPrint(7, ("si5338ReadByteSync: busIndex=0x%x slot=0x%02x address=0x%02x data=0x%02x\n", (unsigned int)pSi5338Ctx->busIndex, (unsigned int)pSi5338Ctx->deviceIndex, (unsigned int)address, (unsigned int)*pVal8));
  return mapI2cStatus(status);
}

static Si5338Status
si5338WriteByteSync(
  Si5338DeviceContext* pSi5338Ctx,
  uint8_t address,
  uint8_t val8)
{
  I2cStatus status;

  dfDebugPrint(7, ("si5338WriteByteSync: busIndex=0x%x slot=0x%02x address=0x%02x data=0x%02x\n", (unsigned int)pSi5338Ctx->busIndex, (unsigned int)pSi5338Ctx->deviceIndex, (unsigned int)address, (unsigned int)val8));
  status = i2cWriteSync(pSi5338Ctx->pI2cCtx, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, address, val8);
  return mapI2cStatus(status);
}

/* Must be called while holding page bit serializer */
static Si5338Status
si5338SetPageBit(
  Si5338DeviceContext* pSi5338Ctx,
  uint8_t page)
{
  Si5338Status status;
  uint8_t tmp8;

  status = si5338ReadByteSync(pSi5338Ctx, SI5338_REGIDX_PAGE, &tmp8);
  if (Si5338StatusSuccess != status) {
    return status;
  }
  tmp8 = (uint8_t)(tmp8 & ~0x1U);
  tmp8 = (uint8_t)(tmp8 | (page & 0x1U));
  return si5338WriteByteSync(pSi5338Ctx, SI5338_REGIDX_PAGE, tmp8);
}

Si5338Status
si5338ReadSync(
  Si5338DeviceContext* pSi5338Ctx,
  uint16_t regIndex,
  uint8_t* pVal8)
{
  SerializerRequestSync request;
  uint8_t page;
  Si5338Status status;

  if (regIndex >= DF_ARRAY_LENGTH(g_registerMasks)) {
    return Si5338StatusInvalidRegister;
  }
  page = (uint8_t)((regIndex >> 8) & 0x1U);
  serializerGetSync(&pSi5338Ctx->pageBitSerializer, &request);
  if (pSi5338Ctx->pageBit != page) {
    pSi5338Ctx->pageBit = page;
    status = si5338WriteByteSync(pSi5338Ctx, SI5338_REGIDX_PAGE, page);
    if (Si5338StatusSuccess != status) {
      /* Force page bit to be changed on next access */
      pSi5338Ctx->pageBit = 0xffU;
      serializerPut(&pSi5338Ctx->pageBitSerializer);
      return status;
    }
  }
  status = si5338ReadByteSync(pSi5338Ctx, (uint8_t)regIndex, pVal8);
  serializerPut(&pSi5338Ctx->pageBitSerializer);
  return status;
}

Si5338Status
si5338WriteMaskSync(
  Si5338DeviceContext* pSi5338Ctx,
  uint16_t regIndex,
  uint8_t mask8,
  uint8_t val8)
{
  SerializerRequestSync request;
  uint8_t page, tmp8;
  Si5338Status status;

  if (regIndex > DF_ARRAY_LENGTH(g_registerMasks)) {
    return Si5338StatusInvalidRegister;
  }
  page = (uint8_t)((regIndex >> 8) & 0x1U);
  serializerGetSync(&pSi5338Ctx->pageBitSerializer, &request);
  if (regIndex != SI5338_REGIDX_PAGE && pSi5338Ctx->pageBit != page) {
    status = si5338SetPageBit(pSi5338Ctx, page);
    if (Si5338StatusSuccess != status) {
      /* Force page bit to be changed on next access */
      pSi5338Ctx->pageBit = 0xffU;
      serializerPut(&pSi5338Ctx->pageBitSerializer);
      return status;
    } else {
      pSi5338Ctx->pageBit = page;
    }
  }
  if (mask8 != 0xFFU) {
    status = si5338ReadByteSync(pSi5338Ctx, (uint8_t)regIndex, &tmp8);
    if (Si5338StatusSuccess != status) {
      serializerPut(&pSi5338Ctx->pageBitSerializer);
      return status;
    }
    val8 = (uint8_t)(val8 & mask8);
    val8 = (uint8_t)(val8 | (tmp8 & ~mask8));
  }
  status = si5338WriteByteSync(pSi5338Ctx, (uint8_t)regIndex, val8);
  serializerPut(&pSi5338Ctx->pageBitSerializer);
  return status;
}

/* I2CCallback invoked when SI5338 register is written */
static void
onRegisterWritten(
  I2cContext* pI2cCtx,
  I2cRequest* pRequest,
  uint8_t data,
  I2cStatus status,
  void* pContextIgnored)
{
  Si5338Request* pSi5338Request = DF_CONTAINER_OF(pRequest, Si5338Request, i2cRequest);
  Si5338DeviceContext* pSi5338Ctx = pSi5338Request->pSi5338Ctx;

  serializerPut(&pSi5338Ctx->pageBitSerializer);
  if (status == I2cStatusSuccess) {
    /* Success */
    pSi5338Request->pCallback(pSi5338Ctx, pSi5338Request, pSi5338Request->data, Si5338StatusSuccess, pSi5338Request->pContext);
  } else {
    /* An I2C error occurred */
    pSi5338Request->pCallback(pSi5338Ctx, pSi5338Request, 0, mapI2cStatus(status), pSi5338Request->pContext);
  }
}

/* I2CCallback invoked when SI5338 register is read */
static void
onRegisterRead(
  I2cContext* pI2cCtx,
  I2cRequest* pRequest,
  uint8_t data,
  I2cStatus status,
  void* pContextIgnored)
{
  Si5338Request* pSi5338Request = DF_CONTAINER_OF(pRequest, Si5338Request, i2cRequest);
  Si5338DeviceContext* pSi5338Ctx = pSi5338Request->pSi5338Ctx;

  serializerPut(&pSi5338Ctx->pageBitSerializer);
  if (status == I2cStatusSuccess) {
    /* Success */
    pSi5338Request->pCallback(pSi5338Ctx, pSi5338Request, data, Si5338StatusSuccess, pSi5338Request->pContext);
  } else {
    /* An I2C error occurred */
    pSi5338Request->pCallback(pSi5338Ctx, pSi5338Request, 0, mapI2cStatus(status), pSi5338Request->pContext);
  }
}

/* I2CCallback invoked when SI5338 register is read as part of read-modify-write cycle */
static void
onRegisterReadForWrite(
  I2cContext* pI2cCtx,
  I2cRequest* pRequest,
  uint8_t data,
  I2cStatus status,
  void* pContextIgnored)
{
  Si5338Request* pSi5338Request = DF_CONTAINER_OF(pRequest, Si5338Request, i2cRequest);
  Si5338DeviceContext* pSi5338Ctx = pSi5338Request->pSi5338Ctx;
  uint16_t regIndex = pSi5338Request->regIndex;
  uint8_t val8;

  if (status == I2cStatusSuccess) {
    /* Perform the write part of the read-modify-write cycle */
    val8 = (uint8_t)(pSi5338Request->data & pSi5338Request->mask);
    val8 = (uint8_t)(val8 | (data & ~pSi5338Request->mask));
    i2cWrite(pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, (uint8_t)regIndex, val8, onRegisterWritten, NULL);
  } else {
    /* An I2C error occurred */
    serializerPut(&pSi5338Ctx->pageBitSerializer);
    pSi5338Request->pCallback(pSi5338Ctx, pSi5338Request, 0, mapI2cStatus(status), pSi5338Request->pContext);
  }
}

/* I2CCallback invoked when SI5338 page register is written */
static void
onPageBitWritten(
  I2cContext* pI2cCtx,
  I2cRequest* pRequest,
  uint8_t data,
  I2cStatus status,
  void* pContextIgnored)
{
  Si5338Request* pSi5338Request = DF_CONTAINER_OF(pRequest, Si5338Request, i2cRequest);
  Si5338DeviceContext* pSi5338Ctx = pSi5338Request->pSi5338Ctx;
  uint16_t regIndex = pSi5338Request->regIndex;
  uint8_t mask;

  if (status == I2cStatusSuccess) {
    uint8_t page = (uint8_t)((regIndex >> 8) & 0x1U);
    pSi5338Ctx->pageBit = page;
    if (pSi5338Request->bWrite) {
      mask = pSi5338Request->mask;
      if (mask != 0xFFU) {
        /* Need to first read the register in question, to preserve some bits */
        i2cRead(pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, (uint8_t)regIndex, onRegisterReadForWrite, NULL);
      } else {
        /* Can write to the register in question without first reading it */
        i2cWrite(pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, (uint8_t)regIndex, pSi5338Request->data, onRegisterWritten, NULL);
      }
    } else {
      /* Read the register in question */
      i2cRead(pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, (uint8_t)regIndex, onRegisterRead, NULL);
    }
  } else {
    /* An I2C error occurred */
    serializerPut(&pSi5338Ctx->pageBitSerializer);
    pSi5338Request->pCallback(pSi5338Ctx, pSi5338Request, 0, mapI2cStatus(status), pSi5338Request->pContext);
  }
}

/* SerializerCallback invoked when SI5338 page bit serializer is acquired */
static void
onPageBitAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContextIgnored)
{
  Si5338Request* pSi5338Request = DF_CONTAINER_OF(pRequest, Si5338Request, pageBitRequest);
  Si5338DeviceContext* pSi5338Ctx = pSi5338Request->pSi5338Ctx;
  uint16_t regIndex = pSi5338Request->regIndex;
  uint8_t page, mask;

  page = (uint8_t)((regIndex >> 8) & 0x1U);
  if (regIndex != SI5338_REGIDX_PAGE && page != pSi5338Ctx->pageBit) {
    /* Need to change the page bit in the SI5338 */
    i2cWrite(pSi5338Ctx->pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, SI5338_REGIDX_PAGE, page, onPageBitWritten, NULL);
  } else {
    if (pSi5338Request->bWrite) {
      mask = pSi5338Request->mask;
      if (mask != 0xFFU) {
        /* Need to first read the register in question, to preserve some bits */
        i2cRead(pSi5338Ctx->pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, (uint8_t)regIndex, onRegisterReadForWrite, NULL);
      } else {
        /* Can write to the register in question without first reading it */
        i2cWrite(pSi5338Ctx->pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, (uint8_t)regIndex, pSi5338Request->data, onRegisterWritten, NULL);
      }
    } else {
      /* Read the register in question */
      i2cRead(pSi5338Ctx->pI2cCtx, &pSi5338Request->i2cRequest, pSi5338Ctx->busIndex, pSi5338Ctx->deviceIndex, (uint8_t)regIndex, onRegisterRead, NULL);
    }
  }
}

void
si5338Read(
  Si5338DeviceContext* pSi5338Ctx,
  Si5338Request* pRequest,
  uint16_t regIndex,
  Si5338Callback* pCallback,
  void* pContext)
{
  dfDebugPrint(5, ("si5338Read: pSi5338Ctx=%p pRequest=%p regIndex=0x%x pCallback=%p\n", pSi5338Ctx, pRequest, regIndex, pCallback));

  if (regIndex > DF_ARRAY_LENGTH(g_registerMasks)) {
    pCallback(pSi5338Ctx, pRequest, 0, Si5338StatusInvalidRegister, pContext);
  }
  pRequest->pSi5338Ctx = pSi5338Ctx;
  pRequest->pCallback = pCallback;
  pRequest->regIndex = regIndex;
  pRequest->bWrite = FALSE;
  pRequest->pContext = pContext;
  serializerGet(&pSi5338Ctx->pageBitSerializer, &pRequest->pageBitRequest, onPageBitAcquired, NULL);
}

void
si5338WriteMask(
  Si5338DeviceContext* pSi5338Ctx,
  Si5338Request* pRequest,
  uint16_t regIndex,
  uint8_t mask8,
  uint8_t val8,
  Si5338Callback* pCallback,
  void* pContext)
{
  dfDebugPrint(5, ("si5338WriteMask: pSi5338Ctx=%p pRequest=%p regIndex=0x%x mask8=0x%02x val8=0x%02x pCallback=%p\n",
    pSi5338Ctx, pRequest, regIndex, mask8, val8, pCallback));

  if (regIndex > DF_ARRAY_LENGTH(g_registerMasks)) {
    pCallback(pSi5338Ctx, pRequest, 0, Si5338StatusInvalidRegister, pContext);
  }
  pRequest->pSi5338Ctx = pSi5338Ctx;
  pRequest->pCallback = pCallback;
  pRequest->regIndex = regIndex;
  pRequest->mask = mask8;
  pRequest->data = val8;
  pRequest->bWrite = TRUE;
  pRequest->pContext = pContext;
  serializerGet(&pSi5338Ctx->pageBitSerializer, &pRequest->pageBitRequest, onPageBitAcquired, NULL);
}

void
si5338Init(
	I2cContext* pI2cCtx,
  Si5338DeviceContext* pSi5338Ctx,
  uint8_t busIndex,
  uint8_t deviceIndex)
{
  uint8_t val8;
  unsigned int i;

  pSi5338Ctx->pI2cCtx = pI2cCtx;
  pSi5338Ctx->busIndex = busIndex;
  pSi5338Ctx->deviceIndex = deviceIndex;
  serializerQueueInit(&pSi5338Ctx->pageBitSerializer);
  /* Establish the page register */
  si5338ReadByteSync(pSi5338Ctx, SI5338_REGIDX_PAGE, &val8);
  pSi5338Ctx->pageBit = (uint8_t)(val8 & 0x1U);
  for (i = 0; i < 4; i++) {
    pSi5338Ctx->save[i].bLastValid = FALSE;
  }
}

void
si5338DumpRegisters(
  Si5338DeviceContext* pSi5338Ctx)
{
  uint8_t buf[DF_ARRAY_LENGTH(g_registerMasks)];
  size_t i;

  for (i = 0; i < sizeof(buf); i++) {
    si5338ReadSync(pSi5338Ctx, (uint16_t)i, &buf[i]); /* Cast is safe unless more than 65535 registers in SI5338! */
  }
  dfDebugDumpMemory(0, 0, 16, 1, FALSE, buf, sizeof(buf));
}

static boolean_t
initFrequencyStepping(
  Si5338DeviceContext* pSi5338Ctx)
{
  uint32_t p1, p2, p3;
  uint64_t fidp1, fidp3, fidp2;
  uint32_t remainder32, fOut, freqStep;
  uint64_t quotient, dividend, remainder;
  uint16_t i;
  uint8_t val8;
  boolean_t bOk;

  for (i = 0; i < 4; i++) {
    if (0U != pSi5338Ctx->multisynth[i].frequencyStep) {
      bOk = TRUE;
      /*
      ** Calculate and program MSn_FIDP1 & MSn_FIDP3, for glitch-free frequency changes.
      ** From SI5338 AN411:
      **
      **   MSn_FIDP1 = 10^6 * MSn_P3
      **   MSn_FIDP2 = 10^6 * MSn_P3 * (fStep / fOut)
      **   MSn_FIDP3 = 10^6/128 * (MSn_P3 * (MSn_P1 + 512) + MSn_P2)
      **
      ** where fOut is multisynth output frequency. We use output divider of 1 for all outputs,
      ** so no need to taking account of output divider.
      **
      ** MSn_FIDP1 and MSn_FIDP3 are fixed because we never change fOut, p1, p2 or p3, so we
      ** program them here. MSn_FIDP2, on the other hand, must be calculated each time a
      ** frequency change is required.
      */
      p1 = pSi5338Ctx->multisynth[i].p1;
      p2 = pSi5338Ctx->multisynth[i].p2;
      p3 = pSi5338Ctx->multisynth[i].p3;
      fOut = pSi5338Ctx->multisynth[i].frequency;
      freqStep = pSi5338Ctx->multisynth[i].frequencyStep;
      fidp1 = (uint64_t)p3 * 1000000U;
      quotient = dfDivide64By32Rem(fidp1, (uint32_t)fOut, &remainder32);
      fidp2 = quotient * (uint64_t)freqStep + dfDivide64By32((uint64_t)remainder32 * (uint64_t)freqStep, (uint32_t)fOut);
      dividend = (uint64_t)p3 * (uint64_t)(p1 + 512U) + (uint64_t)p2;
      quotient = dfDivide64By64Rem(dividend, 128U, &remainder);
      fidp3 = quotient * 1000000U + (remainder * 1000000U) / 128U;
      dfDebugPrint(2, ("si5338Interrogate: multisynth %u: frequencyStep=%lu fidp1=" DF_FMT_U64 " fidp2=" DF_FMT_U64 " fidp3=" DF_FMT_U64 "\n",
        i, (unsigned long)freqStep, (unsigned long long)fidp1, (unsigned long long)fidp2, (unsigned long long)fidp3));
      pSi5338Ctx->multisynth[i].fidp1 = fidp1;
      pSi5338Ctx->multisynth[i].fidp2 = fidp2;
      pSi5338Ctx->multisynth[i].fidp3 = fidp3;
      pSi5338Ctx->multisynth[i].currentStep = 0;

      /* Reset frequency increment/decrement for MSn by disabling it */
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_MSx_FIDCTL(i), 0x10U, 0x10U));

      /* Program MSn_FIDP1 */
      val8 = (uint8_t)((fidp1 >> 0) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp1Addr[i] + 0), 0xFF, val8));
      val8 = (uint8_t)((fidp1 >> 8) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp1Addr[i] + 1), 0xFF, val8));
      val8 = (uint8_t)((fidp1 >> 16) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp1Addr[i] + 2), 0xFF, val8));
      val8 = (uint8_t)((fidp1 >> 24) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp1Addr[i] + 3), 0xFF, val8));
      val8 = (uint8_t)((fidp1 >> 32) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp1Addr[i] + 4), 0xFF, val8));
      val8 = (uint8_t)((fidp1 >> 40) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp1Addr[i] + 5), 0xFF, val8));
      val8 = (uint8_t)((fidp1 >> 48) & 0x0FU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp1Addr[i] + 6), 0x0F, val8));

      /* Program MSn_FIDP2 */
      val8 = (uint8_t)((fidp2 >> 0) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp2Addr[i] - 0), 0xFF, val8));
      val8 = (uint8_t)((fidp2 >> 8) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp2Addr[i] - 1), 0xFF, val8));
      val8 = (uint8_t)((fidp2 >> 16) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp2Addr[i] - 2), 0xFF, val8));
      val8 = (uint8_t)((fidp2 >> 24) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp2Addr[i] - 3), 0xFF, val8));
      val8 = (uint8_t)((fidp2 >> 32) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp2Addr[i] - 4), 0xFF, val8));
      val8 = (uint8_t)((fidp2 >> 40) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp2Addr[i] - 5), 0xFF, val8));
      val8 = (uint8_t)((fidp2 >> 48) & 0x0FU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp2Addr[i] - 6), 0x0F, val8));

      /* Program MSn_FIDP3 */
      val8 = (uint8_t)((fidp3 >> 0) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 0), 0xFF, val8));
      val8 = (uint8_t)((fidp3 >> 8) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 1), 0xFF, val8));
      val8 = (uint8_t)((fidp3 >> 16) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 2), 0xFF, val8));
      val8 = (uint8_t)((fidp3 >> 24) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 3), 0xFF, val8));
      val8 = (uint8_t)((fidp3 >> 32) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 4), 0xFF, val8));
      val8 = (uint8_t)((fidp3 >> 40) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 5), 0xFF, val8));
      val8 = (uint8_t)((fidp3 >> 48) & 0xFFU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 6), 0xFF, val8));
      val8 = (uint8_t)((fidp3 >> 56) & 0x7FU);
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, (uint16_t)(g_fidp3Addr[i] + 7), 0x7F, val8));

      /* Reenable frequency increment/decrement for MSn */
      bOk &= (Si5338StatusSuccess == si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_MSx_FIDCTL(i), 0x10U, 0x00U));

      if (!bOk) {
        dfDebugPrint(0, ("*** si5338Interrogate: failed to read one or more SI5338 registers (2:%u)\n", i));
        return FALSE;
      }
    }
  }

  return TRUE;
}

boolean_t
si5338Interrogate(
  Si5338DeviceContext* pSi5338Ctx,
  uint32_t fRef,
  const uint32_t freqStep[4])
{
  uint32_t fOutMin, fOutMax, fOutMaxFrac, fOutMaxInt;
  boolean_t bOk = TRUE;
  uint8_t basePart;
  char devGrade, revision;
  uint32_t nvmCode;
  uint8_t val8, i2cAddr, expI2cAddr, pfdInSel, pfdInDiv;
  uint32_t p1, p2, p3;
  uint64_t dividend, divisor, quotient, remainder, fVco, fOut;
  uint32_t fRefDivFactor;
  uint16_t i;

  /* Will correct later */
  pSi5338Ctx->vco.frequency = 0U;
  for (i = 0; i < 4; i++) {
    pSi5338Ctx->multisynth[i].frequency = 0U;
    pSi5338Ctx->multisynth[i].frequencyStep = freqStep[i];
  }

  /* Figure out part number */
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_REV_ID, &val8));
  revision = (char)('A' + (char)(val8 & 0x7U));
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_DEV_CONFIG2, &val8));
  basePart = (uint8_t)(val8 & 0x3FU);
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_DEV_CONFIG3, &val8));
  devGrade = (char)('A' - 1 + (char)((val8 & 0xF8U) >> 3));
  pSi5338Ctx->bIsSi5338a = (devGrade == 'A') ? TRUE : FALSE;
  nvmCode = (uint32_t)((uint32_t)(val8 & 0x1) << 16);
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_DEV_CONFIG4, &val8));
  nvmCode |= (uint32_t)((uint32_t)val8 << 0);
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_DEV_CONFIG5, &val8));
  nvmCode |= (uint32_t)((uint32_t)val8 << 8);
  if (devGrade < 'A' || devGrade > 'B') {
  	dfDebugPrint(0, ("*** si5338Interrogate: unsupported part SI53%u%c-%c%lu\n", (unsigned int)basePart, devGrade, (unsigned int)revision, (unsigned long)nvmCode));
  } else {
  	dfDebugPrint(1, ("si5338Interrogate: part is SI53%u%c-%c%lu\n", (unsigned int)basePart, devGrade, (unsigned int)revision, (unsigned long)nvmCode));
  }

  /* Read I2C address register */
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_I2C_ADDR, &i2cAddr));

  /* Get PFD input select register */
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_PFD_IN_SEL, &val8));
  pfdInSel = (uint8_t)((val8 >> 5) & 0x7U);
  pfdInDiv = (uint8_t)(val8 & 0x3U);

  /* Get MSn_P1, MSn_P2 & MSn_P3 */
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 0, &val8));
  p1 = val8;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 1, &val8));
  p1 |= (uint32_t)val8 << 8;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 2, &val8));
  p1 |= (uint32_t)(val8 & 0x3U) << 16;
  p2 = val8 >> 6;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 3, &val8));
  p2 |= (uint32_t)val8 << 6;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 4, &val8));
  p2 |= (uint32_t)val8 << 14;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 5, &val8));
  p2 |= (uint32_t)val8 << 22;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 6, &val8));
  p3 = (uint32_t)val8 << 0;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 7, &val8));
  p3 |= (uint32_t)val8 << 8;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 8, &val8));
  p3 |= (uint32_t)val8 << 16;
  bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_MSN_P1 + 9, &val8));
  p3 |= (uint32_t)(val8 & 0x3FU) << 24;

  if (!bOk) {
    dfDebugPrint(0, ("*** si5338Interrogate: failed to read one or more SI5338 registers (1)\n"));
    return FALSE;
  }
	dfDebugPrint(1, ("si5338Interrogate: p1=%lu p2=%lu p3=%lu\n", (unsigned long)p1, (unsigned long)p2, (unsigned long)p3));

  /* Can't reliably check bit 0 of I2C address register due to the way SI5338 I2C_LSB pin works. */
  i2cAddr = (uint8_t)(i2cAddr & 0x7EU);
  expI2cAddr = (uint8_t)(pSi5338Ctx->deviceIndex & 0x7EU);
  if (i2cAddr != expI2cAddr) {
    dfDebugPrint(0, ("*** si5338Interrogate: SI5338 I2C_ADDR=0x%02x, expected 0x%02x\n",
      i2cAddr, expI2cAddr));
    return FALSE;
  }

  switch (pfdInSel) {
  case 0U: /* REFCLK */
  case 4U: /* XOCLK */
    fRefDivFactor = 1U;
    break;

  case 2U: /* DIVREFCLK */
    if (pfdInDiv > 5U) {
      dfDebugPrint(0, ("*** si5338Interrogate: invalid SI5338 PFD reference clock divide select value (%u)\n"));
      return FALSE;
    }
    fRefDivFactor = 1U << pfdInDiv;
    break;

  case 1U: /* FBCLK */
  case 3U: /* DIVFBCLK */
  case 5U: /* NOCLK */
  default:
    dfDebugPrint(0, ("*** si5338Interrogate: invalid SI5338 PFD reference clock select (%u)\n", pfdInSel));
    return FALSE;
  }

  /*
  ** Find fVco from fRef, MSn_P1, MSn_P2 and MSn_P3 by reversing the formulae
  ** for MSn_P1, MSn_P2 and MSn_P3. We have formulae from SI5338 data sheet:
  **
  **    fVco = (fRef / d) * (a + b/c)
  **    p1 = [128(ac + b) div c] - 512
  **    p2 = 128(ac + b) mod c
  **    p3 = c
  **
  ** where p1, p2 and p3 are obtained from SI5338 registers, and d is reference
  ** clock division factor from SI5338 registers.
  **
  ** To begin with, examine p1 and p2 and note that
  **
  **    128(ac + b) = p2 + (p1 + 512) * p3
  **
  ** So
  **
  **    a + b/c = [p2 + (p1 + 512) * p3] / (128 * p3)
  ** => fVco = (fRef / d) * [p2 + (p1 + 512) * p3] / (128 * p3)
  ** => fVco = fRef * [p2 + (p1 + 512) * p3] / (128 * p3 * d)
  **
  ** This must be coded carefully in order to avoid arithmetic overflow as
  ** well as loss of precision.
  */
  if (0 == p3) {
  	dfDebugPrint(0, ("*** si5338Interrogate: P3 value is zero\n"));
    return FALSE;
  }
  dividend = (uint64_t)p3 * (uint64_t)(p1 + 512U) + (uint64_t)p2;
  divisor = (uint64_t)p3 * 128U * fRefDivFactor;
  quotient = dfDivide64By64Rem(dividend, divisor, &remainder);
  fVco = fRef * quotient + dfDivide64By64(remainder * fRef, divisor);
  if (fVco < g_fVcoMin || fVco > g_fVcoMax) {
  	dfDebugPrint(0, ("+++ si5338Interrogate: calculated Fvco is out of range (" DF_FMT_U64 " Hz)\n", (unsigned long long)fVco));
  } else {
  	dfDebugPrint(1, ("si5338Interrogate: fRef=%lu fVco=" DF_FMT_U64 " Hz\n", (unsigned long)fRef, (unsigned long long)fVco));
  }
  pSi5338Ctx->vco.frequency = (uint32_t)fVco;

  /* Find clock output frequencies - cast is safe due to above range check on fVco */
  fOutMin = (uint32_t)fVco / 568; /* Minimum possible output frequency = fVco / 568 */
  fOutMaxFrac = (uint32_t)fVco / 8;  /* Maximum possible output frequency = fVco / 8 with fractional divider */
  fOutMaxInt = (uint32_t)fVco / 4;  /* Maximum possible output frequency = fVco / 8 with integer divider */
  fOutMax = pSi5338Ctx->bIsSi5338a ? fOutMaxInt : fOutMaxFrac;
  for (i = 0; i < 4; i++) {
    /* Get MSn_P1, MSn_P2 & MSn_P3 */
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 0), &val8));
    p1 = val8;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 1), &val8));
    p1 |= (uint32_t)val8 << 8;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 2), &val8));
    p1 |= (uint32_t)(val8 & 0x3U) << 16;
    p2 = (val8 >> 2) & 0x3FU;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 3), &val8));
    p2 |= (uint32_t)val8 << 6;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 4), &val8));
    p2 |= (uint32_t)val8 << 14;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 5), &val8));
    p2 |= (uint32_t)val8 << 22;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 6), &val8));
    p3 = (uint32_t)val8 << 0;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 7), &val8));
    p3 |= (uint32_t)val8 << 8;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 8), &val8));
    p3 |= (uint32_t)val8 << 16;
    bOk &= (Si5338StatusSuccess == si5338ReadSync(pSi5338Ctx, (uint16_t)(SI5338_REGIDX_MSx_P1(i) + 9), &val8));
    p3 |= (uint32_t)(val8 & 0x3FU) << 24;
    dfDebugPrint(2, ("si5338Interrogate: multisynth %u: p1=%u p2=%u p3=%u\n", i, p1, p2, p3));
    dividend = fVco * (uint64_t)p3;
    divisor = (uint64_t)p3 * (uint64_t)(p1 + 512U) + (uint64_t)p2;
    quotient = dfDivide64By64Rem(dividend, divisor, &remainder);
    fOut = quotient * 128U + dfDivide64By64(remainder * 128U, divisor);
    if (fOut < fOutMin || fOut > fOutMax) {
	    dfDebugPrint(0, ("+++ si5338Interrogate: clock output %u frequency is out of range (" DF_FMT_U64 " Hz)\n", i, (unsigned long long)fOut));
    } else {
	    dfDebugPrint(1, ("si5338Interrogate: clock output %u frequency is %lu Hz\n", i, (unsigned long)fOut));
    }
    pSi5338Ctx->multisynth[i].p1 = p1;
    pSi5338Ctx->multisynth[i].p2 = p2;
    pSi5338Ctx->multisynth[i].p3 = p3;
    pSi5338Ctx->multisynth[i].frequency = (uint32_t)fOut; /* Cast is safe due to above range check on fOut */
  }

  if (!initFrequencyStepping(pSi5338Ctx)) {
    return FALSE;
  }

  return TRUE;
}

boolean_t
si5338ConfigureFull(
  Adb3CoreDeviceContext* pDevCtx,
  Si5338DeviceContext* pSi5338Ctx,
  uint32_t start,  /* VPD space start address */
  uint32_t length) /* VPD space length (bytes) */
{
  const unsigned int pllPollIntervalUs = 25000; /* 25 ms poll interval waiting for PLL lock */
  const unsigned int maxCount = 10;             /* Timeout is 10 periods of poll interval waiting for PLL lock */
  Si5338ConfigurationRecord record;
  unsigned int numRecord = 0;
  CoreVpdStatus vpdStatus;
  Si5338Status si5338Status;
  uint16_t addr, tmp16;
  uint8_t mask8, val8, fcal0, fcal1, fcal2;
  unsigned int count;

  vpdStatus = pDevCtx->methods.pReadWriteVpd(pDevCtx, FALSE, start, sizeof(tmp16), &tmp16);
  if (CoreVpdSuccess != vpdStatus) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to probe for valid SI5338 configuration in VPD\n"));
    return FALSE;
  }
  if (tmp16 == 0xFFFFU) {
    dfDebugPrint(1, ("+++ si5338ConfigureFull: valid SI5338 configuration not found in VPD\n"));
    return FALSE;
  }

  /* Disable all outputs */
  val8 = mask8 = 0x10U;
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_DRIVER_DISABLE, mask8, val8);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, val8, SI5338_REGIDX_DRIVER_DISABLE));
    return FALSE;
  }

  /* Pause loss-of-lock detection */
  val8 = 0xE5U;
  mask8 = 0xFFU;
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_LOL_PAUSE, mask8, val8);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, val8, SI5338_REGIDX_LOL_PAUSE));
    return FALSE;
  }

  /* Process SI5338 configuration records that are stored in VPD */
  while (length >= sizeof(record)) {
    /* Read the record header */
    vpdStatus = pDevCtx->methods.pReadWriteVpd(pDevCtx, FALSE, start, sizeof(record), &record);
    if (CoreVpdSuccess != vpdStatus) {
      dfDebugPrint(0, ("*** si5338ConfigureFull: failed to read record of %lu byte(s) from VPD address 0x%lx\n",
        (unsigned long)sizeof(record), (unsigned long)start));
      return FALSE;
    }
    start += (uint32_t)sizeof(record);
    addr = (uint16_t)((uint16_t)record.addrLo | ((uint16_t)record.addrHi << 8));
    if (addr == 0xFFFFU) {
      break;
    }
    si5338Status = si5338WriteMaskSync(pSi5338Ctx, addr, (uint8_t)(record.mask & g_registerMasks[addr]), record.data);
    if (Si5338StatusSuccess != si5338Status) {
      dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
        record.data, record.mask, addr));
      return FALSE;
    }
    numRecord++;
  }
  dfDebugPrint(1, ("si5338ConfigureFull: processed %u record(s)\n", numRecord));

  /* Validate input clock status */
  for (count = 0;; count++) {
    /* Check for input clock valid */
    si5338Status = si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_LOL_STATUS, &val8);
    if (Si5338StatusSuccess != si5338Status) {
      dfDebugPrint(0, ("*** si5338ConfigureFull: failed to read SI5338 register 0x%x\n", SI5338_REGIDX_LOL_STATUS));
      return FALSE;
    }
    if ((val8 & 0x04U) == 0U) {
      /* Input clock is valid */
      break;
    }
    if (count > maxCount) {
      break;
    }
    dfDelayThreadFor(dfMicrosecondsToTime(pllPollIntervalUs));
  }
  if (count > maxCount) {
    /* Timeout waiting for input clock valid */
    dfDebugPrint(0, ("*** si5338ConfigureFull: timeout waiting for input clock valid\n"));
    return FALSE;
  }

  /* Configure PLL for locking */
  val8 = 0x00U;
  mask8 = 0x80U;
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_FCAL_OVRD_EN, mask8, val8);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, val8, SI5338_REGIDX_FCAL_OVRD_EN));
    return FALSE;
  }

  /* Initiate locking of PLL */
  val8 = 0x02U;
  mask8 = 0xFFU; /* SI5338 AN411 says don't use read-modify-write for this register */
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_PLL_SOFT_RESET, mask8, val8);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, val8, SI5338_REGIDX_PLL_SOFT_RESET));
    return FALSE;
  }

  /* Resume loss-of-lock detection */
  val8 = 0x65U;
  mask8 = 0xFFU;
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_LOL_PAUSE, mask8, val8);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, val8, SI5338_REGIDX_LOL_PAUSE));
    return FALSE;
  }

  for (count = 0; count < maxCount; count++) {
    /* SI5338 datasheet says wait 25 ms */
    dfDelayThreadFor(dfMicrosecondsToTime(pllPollIntervalUs));

    /* Check for PLL lock */
    si5338Status = si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_LOL_STATUS, &val8);
    if (Si5338StatusSuccess != si5338Status) {
      dfDebugPrint(0, ("*** si5338ConfigureFull: failed to read SI5338 register 0x%x\n", SI5338_REGIDX_LOL_STATUS));
      return FALSE;
    }
    if ((val8 & 0x15U) == 0U) {
      /* PLL locked, input clock valid, device callibration not in progress */
      break;
    }
    dfDebugPrint(2, ("si5338ConfigureFull: LOL_STATUS=0x%02x\n", val8));
  }
  if (count == maxCount) {
    /* Timeout waiting for PLL lock */
    dfDebugPrint(0, ("*** si5338ConfigureFull: timeout waiting for PLL lock, LOL_STATUS=0x%02x\n", val8));
    return FALSE;
  }

  /* Copy FCAL registers to FCAL_OVRD registers */
  si5338Status = si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_FCAL2, &fcal2);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to read SI5338 register 0x%x\n", SI5338_REGIDX_FCAL2));
    return FALSE;
  }
  si5338Status = si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_FCAL1, &fcal1);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to read SI5338 register 0x%x\n", SI5338_REGIDX_FCAL1));
    return FALSE;
  }
  si5338Status = si5338ReadSync(pSi5338Ctx, SI5338_REGIDX_FCAL0, &fcal0);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to read SI5338 register 0x%x\n", SI5338_REGIDX_FCAL0));
    return FALSE;
  }
  /* Datasheet says set bits 7:2 to 000101 */
  fcal2 = (uint8_t)(fcal2 & 0x03U);
  fcal2 |= 0x14U;
  mask8 = 0xFFU;
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_FCAL_OVRD2, mask8, fcal2);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, fcal2, SI5338_REGIDX_FCAL_OVRD2));
    return FALSE;
  }
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_FCAL_OVRD1, mask8, fcal1);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, fcal1, SI5338_REGIDX_FCAL_OVRD1));
    return FALSE;
  }
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_FCAL_OVRD0, mask8, fcal0);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, fcal0, SI5338_REGIDX_FCAL_OVRD0));
    return FALSE;
  }

  /* Set FCAL_OVRD_EN enable bit */
  val8 = mask8 = 0x80U;
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_FCAL_OVRD_EN, mask8, val8);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, val8, SI5338_REGIDX_FCAL_OVRD_EN));
    return FALSE;
  }

  /* Reenable all outputs */
  mask8 = 0x10U;
  val8 = 0x00U;
  si5338Status = si5338WriteMaskSync(pSi5338Ctx, SI5338_REGIDX_DRIVER_DISABLE, mask8, val8);
  if (Si5338StatusSuccess != si5338Status) {
    dfDebugPrint(0, ("*** si5338ConfigureFull: failed to write value 0x%02x (mask 0x%02x) to SI5338 register 0x%x\n",
      mask8, val8, SI5338_REGIDX_DRIVER_DISABLE));
    return FALSE;
  }

  dfDebugPrint(1, ("si5338ConfigureFull: successful\n"));

  return TRUE;
}

static Si5338ProgramStatus
mapSi5338Status(
  Si5338Status status)
{
  switch (status) {
  case Si5338StatusSuccess:
    return Si5338ProgramStatusSuccess;
  
  case Si5338StatusI2CTimeout:
  case Si5338StatusI2CError:
    return Si5338ProgramStatusI2CError;

  case Si5338StatusInvalidRegister:
  case Si5338StatusGeneralFailure:
  default:
    return Si5338ProgramStatusGeneralFailure;
  }
}

/* SI5338Callback invoked when an SI5338 register access is completed */
static void
onSI5338Written(
  Si5338DeviceContext* pSi5338Ctx,
  Si5338Request* pRequest,
  uint8_t data,
  Si5338Status status,
  void* pContextIgnored)
{
  Si5338ProgramRequest* pProgRequest = DF_CONTAINER_OF(pRequest, Si5338ProgramRequest, request);
  static const uint16_t msAddrs[4] = {
    SI5338_REGIDX_MS0_P1,
    SI5338_REGIDX_MS1_P1,
    SI5338_REGIDX_MS2_P1,
    SI5338_REGIDX_MS3_P1
  };
  unsigned int stage = pProgRequest->stage;
  unsigned int msIndex = pProgRequest->msIndex;
  CoreClockWord* pClockWord = &pProgRequest->clockWord;
  uint16_t msAddr;
  uint8_t enableBit, val8;

  dfDebugPrint(4, ("onSI5338Written: pRequest=%p status=%u\n", pRequest, status));

  if (Si5338StatusSuccess != status) {
    /* An error occurred while writing to a SI5338 register */
    pProgRequest->pCallback(pSi5338Ctx, pProgRequest, pProgRequest->msIndex, mapSi5338Status(status), pProgRequest->pContext);
    return;
  }

  /* Look up address of selected MultiSynth in SI5338 */
  stage = pProgRequest->stage;
  msIndex = pProgRequest->msIndex;
  pClockWord = &pProgRequest->clockWord;
  msAddr = msAddrs[msIndex];
  pProgRequest->stage = stage + 1;

  dfDebugPrint(4, ("onSI5338Written: stage=%u msIndex=%u msAddr=0x%x\n", stage, msIndex, msAddr));

  switch (stage) {
  case 0: /* Set MSx_P1[7:0] */    
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 0), 0xFFU, (uint8_t)(pClockWord->d0 >> 0), onSI5338Written, NULL);
    break;

  case 1: /* Set MSx_P1[15:8] */    
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 1), 0xFFU, (uint8_t)(pClockWord->d0 >> 8), onSI5338Written, NULL);
    break;

  case 2: /* Set MSx_P1[17:16], MSx_P2[5:0] */
    val8 = (uint8_t)((pClockWord->d0 >> 16) & 0x3U);
    val8 = (uint8_t)(val8 | ((pClockWord->d1 & 0x3FU) << 2));
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 2), 0xFFU, val8, onSI5338Written, NULL);
    break;

  case 3: /* Set MSx_P2[13:16] */    
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 3), 0xFFU, (uint8_t)(pClockWord->d1 >> 6), onSI5338Written, NULL);
    break;

  case 4: /* Set MSx_P2[21:14] */    
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 4), 0xFFU, (uint8_t)(pClockWord->d1 >> 14), onSI5338Written, NULL);
    break;

  case 5: /* Set MSx_P2[29:22] */    
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 5), 0xFFU, (uint8_t)(pClockWord->d1 >> 22), onSI5338Written, NULL);
    break;

  case 6: /* Set MSx_P3[7:0] */
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 6), 0xFFU, (uint8_t)(pClockWord->d2 >> 0), onSI5338Written, NULL);
    break;

  case 7: /* Set MSx_P3[15:8] */
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 7), 0xFFU, (uint8_t)(pClockWord->d2 >> 8), onSI5338Written, NULL);
    break;

  case 8: /* Set MSx_P3[23:16] */
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 8), 0xFFU, (uint8_t)(pClockWord->d2 >> 16), onSI5338Written, NULL);
    break;

  case 9: /* Set MSx_P3[29:24] */
    si5338WriteMask(pSi5338Ctx, pRequest, (uint16_t)(msAddr + 9), 0x3FU, (uint8_t)(pClockWord->d2 >> 24), onSI5338Written, NULL);
    break;

  case 10: /* Reenable the selected clock output */
    enableBit = (uint8_t)(0x1U << msIndex);
    si5338WriteMask(pSi5338Ctx, pRequest, SI5338_REGIDX_DRIVER_DISABLE, enableBit, 0, onSI5338Written, NULL);
    break;

  case 11: /* Successfully wrote all necessary SI5338 registers */    
    /* Save last programming value for restoring state on power-up */
    pSi5338Ctx->save[pProgRequest->msIndex].bLastValid = TRUE;
    pSi5338Ctx->save[pProgRequest->msIndex].last = pProgRequest->clockWord;
    pProgRequest->pCallback(pSi5338Ctx, pProgRequest, pProgRequest->msIndex, Si5338ProgramStatusSuccess, pProgRequest->pContext);
    break;

  default:
    /* Should never get here! */
    pProgRequest->pCallback(pSi5338Ctx, pProgRequest, pProgRequest->msIndex, Si5338ProgramStatusGeneralFailure, pProgRequest->pContext);
    break;
  }
}

Si5338ProgramStatus
si5338ClockProgram(
  Si5338DeviceContext* pSi5338Ctx,
  Si5338ProgramRequest* pRequest,
  unsigned int msIndex,
  const CoreClockWord* pClockWord,
  Si5338ProgramCallback* pCallback,
  void* pContext)
{
  uint8_t enableBit;

  dfDebugPrint(4, ("si5338ClockProgram: pRequest=%p msIndex=%u pCallback=%p\n", pRequest, msIndex, pCallback));

  if (msIndex > 3) {
    return Si5338ProgramStatusInvalidIndex;
  }

  /* Start programming the SI5338 */
  pRequest->msIndex = msIndex;
  pRequest->stage = 0;
  pRequest->clockWord = *pClockWord;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;

  /* First, disable the selected clock output */
  enableBit = (uint8_t)(0x1U << msIndex);
  si5338WriteMask(pSi5338Ctx, &pRequest->request, SI5338_REGIDX_DRIVER_DISABLE, enableBit, enableBit, onSI5338Written, NULL);

  return Si5338ProgramStatusSuccess;
}

static void
onProgramDone(
  Si5338DeviceContext* pSi5338Ctx,
  Si5338ProgramRequest* pRequest,
  unsigned int msIndex,
  Si5338ProgramStatus status,
  void* pContextIgnored)
{
  Si5338ProgramRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, Si5338ProgramRequestSync, request);

  dfDebugPrint(4, ("onProgramDone: msIndex=%u status=%u\n", msIndex, status));

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

Si5338ProgramStatus
si5338ClockProgramSync(
  Si5338DeviceContext* pSi5338Ctx,
  unsigned int msIndex, /* Multisynth index */
  const CoreClockWord* pClockWord)
{
  Si5338ProgramRequestSync req;
  Si5338ProgramStatus status;

  dfDebugPrint(3, ("si5338ClockProgramSync: msIndex=%u\n", msIndex));

  dfEventInit(&req.ev);
  status = si5338ClockProgram(pSi5338Ctx, &req.request, msIndex, pClockWord, onProgramDone, NULL);
  if (Si5338ProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);

  return status;
}

Si5338ProgramStatus
si5338ClockProgramSteppingSync(
  Si5338DeviceContext* pSi5338Ctx,
  unsigned int msIndex,
  const CoreClockWord* pClockWord)
{
  Si5338Status status = Si5338StatusSuccess;
  int32_t currentStep, targetStep;
  boolean_t bIncrementing;

  dfDebugPrint(2, ("si5338StepFrequencyToTarget: msIndex=%u frequency=%lu targetStep=%ld\n",
    msIndex, (unsigned long)pClockWord->frequency, (long)(int32_t)pClockWord->d0));

  if (msIndex > 3) {
    return Si5338ProgramStatusInvalidIndex;
  }

  currentStep = pSi5338Ctx->multisynth[msIndex].currentStep;
  targetStep = (int32_t)pClockWord->d0;

  /* Apply the required number of steps */
  bIncrementing = (targetStep > currentStep) ? TRUE : FALSE;
  while (currentStep != targetStep) {
    /* Ping the increment/decrement field */
    dfDebugPrint(3, ("si5338StepFrequencyToTarget: Stepping %s\n", bIncrementing ? "UP" : "DOWN"));
    status = si5338WriteMaskSync(pSi5338Ctx, (uint16_t)SI5338_REGIDX_MSx_FIDCTL(msIndex), 0x70U, (uint8_t)(bIncrementing ? 0x40U : 0x60U));
    if (Si5338StatusSuccess != status) {
      dfDebugPrint(0, ("*** si5338StepFrequencyToTarget: failed to write SI5338 FIDCTL%u register\n", msIndex));
      goto done;
    }

    if (bIncrementing) {
      currentStep++;
    } else {
      currentStep--;
    }
    pSi5338Ctx->multisynth[msIndex].currentStep = currentStep;
  }

  dfDebugPrint(2, ("si5338StepFrequencyToTarget: currentStep=%ld\n", (long)currentStep));

  /* Save last programming value for restoring state on power-up */
  pSi5338Ctx->save[msIndex].bLastValid = TRUE;
  pSi5338Ctx->save[msIndex].last = *pClockWord;

done:
  return mapSi5338Status(status);
}

void
si5338SaveState(
  Si5338DeviceContext* pSi5338Ctx)
{
  /* Invalidate the SI5338 page bit shadow to force the page register to be reinitialized on restore */
  pSi5338Ctx->pageBit = 0xFFU;
}

Si5338ProgramStatus
si5338RestoreState(
  Si5338DeviceContext* pSi5338Ctx)
{
  Si5338ProgramStatus status = Si5338ProgramStatusSuccess;
  unsigned int msIndex;

  dfDebugPrint(2, ("si5338RestoreState: initializing frequency stepping\n"));
  initFrequencyStepping(pSi5338Ctx);
  for (msIndex = 0; msIndex < 4; msIndex++) {
    if (pSi5338Ctx->save[msIndex].bLastValid) {
      if (pSi5338Ctx->multisynth[msIndex].frequencyStep == 0) {
        dfDebugPrint(2, ("si5338RestoreState: restoring SI5338 output %u by multisynth programming\n", msIndex));
        status = si5338ClockProgramSync(pSi5338Ctx, msIndex, &pSi5338Ctx->save[msIndex].last);
      } else {
        dfDebugPrint(2, ("si5338RestoreState: restoring SI5338 output %u by multisynth stepping\n", msIndex));
        status = si5338ClockProgramSteppingSync(pSi5338Ctx, msIndex, &pSi5338Ctx->save[msIndex].last);
      }
      if (Si5338ProgramStatusSuccess != status) {
        return status;
      }
    }
  }

  return status;
}

ClockProgramStatus
si5338MapProgramStatus(
  Si5338ProgramStatus status)
{
  switch (status) {
  case Si5338ProgramStatusSuccess:
    return ClockProgramSuccess;

  case Si5338ProgramStatusI2CError:
    return ClockProgramHardwareError;

  case Si5338ProgramStatusInvalidIndex:
    return ClockProgramInvalidIndex;

  case Si5338ProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
si5338MapClockWordStatus(
  Si5338ClockWordStatus status)
{
  switch (status) {
  case Si5338ClockWordStatusSuccess:
    return ClockWordSuccess;

  case Si5338ClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case Si5338ClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
