/*
** File: serializer.h  
** Project: ADB3 core driver
** Purpose: Definitions for serialization mechanism
**
** (C) Copyright Alpha Data 2011
**
** Simple FIFO serialization mechanism where a client-supplied callback is
** invoked when a given request reaches the head of the queue. Restrictions:
**
** - Cancellation of asynchronous wait is not currently supported.
** - Interruption of synchronous wait (e.g. by a signal) is not currently
**   supported.
**
** Intended to be used for serializing access to hardware register groups,
** e.g. an I2C interface that requires several register accesses to transfer
** a byte of data.
*/

#if !defined(ADATA_CORE_SERIALIZER_H)
#define ADATA_CORE_SERIALIZER_H

#include <df.h>

struct _SerializerQueue;
typedef struct _SerializerQueue SerializerQueue;
struct _SerializerRequest;
typedef struct _SerializerRequest SerializerRequest;
struct _SerializerRequestSync;
typedef struct _SerializerRequestSync SerializerRequestSync;

struct _SerializerQueue { /* Client must not touch fields */
  DfSpinLock lock;
  DfList requests;
};

typedef void (SerializerCallback)(SerializerQueue* pQueue, SerializerRequest* pRequest, void* pContext);

struct _SerializerRequest { /* Client must not touch fields */
  DfListNode          node;
  SerializerCallback* pCallback;
  void*               pContext;
};

struct _SerializerRequestSync { /* Client must not touch fields */
  SerializerRequest   request;
  DfEvent*            pEvent;
};

extern void
_serializerSyncCallback(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContext);

extern void
serializerQueueInit(
  SerializerQueue* pQueue);

DF_DECLARE_INLINE_FUNC(SerializerRequest*, serializerGetHead)(
  SerializerQueue* pQueue)
{
  DfListNode* pHead = dfListGetHead(&pQueue->requests);
  
  if (NULL == pHead) {
    return NULL;
  } else {
    return DF_CONTAINER_OF(pHead, SerializerRequest, node);
  }
}

DF_DECLARE_INLINE_FUNC(void, serializerGet)(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  SerializerCallback* pCallback,
  void* pContext)
{
  DfSpinLockFlags f;
  boolean_t bEmpty;

  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  f = dfSpinLockGet(&pQueue->lock);
  bEmpty = dfListIsEmpty(&pQueue->requests);
  dfListAddToTail(&pQueue->requests, &pRequest->node);
  dfSpinLockPut(&pQueue->lock, f);
  if (bEmpty) {
    /* Queue was empty; can invoke the callback immediately */
    pCallback(pQueue, pRequest, pContext);
  }
}

DF_DECLARE_INLINE_FUNC(void, serializerPut)(
  SerializerQueue* pQueue)
{
  DfSpinLockFlags f;
  DfListNode* pHead;
  SerializerRequest* pNext;

  f = dfSpinLockGet(&pQueue->lock);
  pHead = dfListGetHead(&pQueue->requests);
  dfListRemove(pHead);
  pHead = dfListGetHead(&pQueue->requests);
  dfSpinLockPut(&pQueue->lock, f);
  if (NULL != pHead) {
    /* Invoke the callback of the next node */
    pNext = DF_CONTAINER_OF(pHead, SerializerRequest, node);
    pNext->pCallback(pQueue, pNext, pNext->pContext);
  }
}

DF_DECLARE_INLINE_FUNC(void, serializerGetSync)(
  SerializerQueue* pQueue,
  SerializerRequestSync* pRequest)
{
  DfEvent event;

  dfEventInit(&event);
  pRequest->pEvent = &event;
  serializerGet(pQueue, &pRequest->request, _serializerSyncCallback, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
}

/* Returns TRUE if 'pRequest' was successfully dequeued, otherwise FALSE */
extern boolean_t
serializerCancel(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest);

#endif
