/*
** File: model_admxrc4.c  
** Project: ADB3 core driver
** Purpose: Code, including function to bootstrap device context, specific to:
**   o ADM-XRC-4LX
**   o ADM-XRC-4SX
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "admxrc4.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "ics307_common.h"
#include "ics8430_61_common.h"
#include "model_boot.h"
#include "pci9xxx.h"
#include "pci9xxx_common.h"
#include "pci9xxx_93x6.h"

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData);

static void
onIcs307ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIgnored,
  Ics307ProgramRequest* pRequest,
  Ics307ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, ics307MapProgramStatus(status));
}

static void
onIcs8430_61ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIgnored,
  Ics8430_61ProgramRequest* pRequest,
  Ics8430_61ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 1, ics8430_61MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext*	pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return ics307MapProgramStatus(ics307Program(pDevCtx, &pDevCtx->model.admxrc4.ics307.context, &pDevCtx->model.admxrc4.ics307.programming, pClockWord, onIcs307ProgramDone));

  case 1:
    return ics8430_61MapProgramStatus(ics8430_61Program(pDevCtx, &pDevCtx->model.admxrc4.ics8430_61.context, &pDevCtx->model.admxrc4.ics8430_61.programming, pClockWord, onIcs8430_61ProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return ics307MapClockWordStatus(clockWordIcs307(flags, ADMXRC4_REF_OSC_FREQUENCY, frequencyReq, frequencyMin, frequencyMax, FALSE, pClockWord));

  case 1:
    return ics8430_61MapClockWordStatus(clockWordIcs8430_61(flags, ADMXRC4_REF_OSC_FREQUENCY, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}

static void
flashReset(
  Adb3CoreDeviceContext* pDevCtx)
{
  ModelRegsAdmxrc4* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl;

  dfDebugPrint(9, ("flashReset: entered\n"));

  pModelRegs = pDevCtx->hardware.model.pAdmxrc4;
  hModel = pDevCtx->hardware.hModel;

  /* Drive fl_rst_n, assert fl_rst_n */
	flctl = (0x1U << 27) | (0x1U << 31);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	/* Deassert fl_rst_n */
	flctl &= ~(0x1U << 31);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	/* Stop driving fl_rst_n */
	flctl &= ~(0x1U << 27);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

  /* Allow device time to recover */
  dfDelayThreadFor(dfMicrosecondsToTime(20));
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  ModelRegsAdmxrc4* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flpage, flctl;

  dfDebugPrint(9, ("flash: bankIndex=%lu bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    (unsigned long)bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0 || location >= 0x2000000U) {
    return;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc4;
  hModel = pDevCtx->hardware.hModel;

	flpage = (uint32_t)location & 0x1fe0000U;
	dfPciMemWrite32(hModel, &pModelRegs->flpage, flpage);
	(void)dfPciMemRead32(hModel, &pModelRegs->flpage);

	if (bWrite) {
		/* Select mode 0, set fl_a, drive fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
		flctl = ((uint32_t)location & 0x1fffeU) | (0x7U << 24) | (0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Set fl_d */
		dfPciMemWrite32(hModel, &pModelRegs->fldata, *pValue);
		(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

		/* Assert fl_we_n */
		flctl |= (0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_we_n */
		flctl &= ~(0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_ce_n */
		flctl &= ~(0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Stop driving fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n */
		flctl &= ~(0x7U << 24);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);
	} else {
		/* Select mode 0, set fl_a, Drive fl_a, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
		flctl = ((uint32_t)location & 0x1fffeU) | (0x3U << 25) | (0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Assert fl_oe_n */
		flctl |= (0x1U << 29);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Read fl_d */
		*pValue = (uint16_t)dfPciMemRead32(hModel, &pModelRegs->fldata);
		dfDebugPrint(9, ("flash: *pValue=0x%04x\n", (unsigned int)*pValue));

		/* Deassert fl_oe_n */
		flctl &= ~(0x1U << 29);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_ce_n */
		flctl &= ~(0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Stop driving fl_a, fl_ce_n, fl_oe_n, fl_we_n */
		flctl &= ~(0x3U << 25);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);
	}
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdmxrc4* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaControl: index=%u opCode=%lu pbValue=%p\n",
    index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc4;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADMXRC4_FPCTL_DONE) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    *pbValue = TRUE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat *pbValue=%s\n", *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADMXRC4_FPCTL_INIT) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = pDevCtx->model.admxrc4.shadow.fpctl;
    if (*pbValue) {
      val32 = (uint32_t)(val32 | ADMXRC4_FPCTL_PROG);
    } else {
      val32 = (uint32_t)(val32 & ~ADMXRC4_FPCTL_PROG);
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, &pModelRegs->fpctl, val32);
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const boolean_t bSelectMap32Bit = FALSE;
  const size_t mask32 = 0x3U;
  ModelRegsAdmxrc4* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaSelectMap: index=%u bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc4;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    if (bSelectMap32Bit) {
      /* Align pointer to 32-bit boundary using 8-bit transfers */
      align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
      if (align32 > length) {
        align32 = length;
      }
      length -= align32;
      while (align32--) {
        val8 = *pBuffer8++;
        dfPciMemWrite8(hModel, (uint8_t*)&pModelRegs->fpsmap[0], val8);
      }
      
      /* Do bulk of data using 32-bit transfers */
      length32 = length & ~mask32;
      pBuffer32 = (uint32_t*)pBuffer8;
      length -= length32;
      while (length32) {
        val32 = dfLe32ToCpu(*pBuffer32++);
        dfPciMemWrite32(hModel, &pModelRegs->fpsmap[0], val32);
        length32 -= 4;
      }
      
      /* Do remainder using 8-bit transfers */
      pBuffer8 = (uint8_t*)pBuffer32;
      while (length--) {
        val8 = *pBuffer8++;
        dfPciMemWrite8(hModel, (uint8_t*)&pModelRegs->fpsmap[0], val8);
      }
    } else {
      while (length--) {
        val8 = *pBuffer8++;
        dfPciMemWrite8(hModel, (uint8_t*)&pModelRegs->fpsmap[0], val8);
      }
    }
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, (uint8_t*)&pModelRegs->fpsmap[0]);
      *pBuffer8++ = val8;
    }
  }
  
  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  CoreVpdStatus status;
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc4* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  VpdAdmxrc4Rev0* pVpd, vpdRaw;
  uint32_t fOut;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc4;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      switch (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice) {
      case PCI_SUBDEVICE_PCI9656_ADMXRC4LX:
        pDevCtx->info.model = CoreModelAdmxrc4lx;
        break;

      case PCI_SUBDEVICE_PCI9656_ADMXRC4SX:
      default:
        pDevCtx->info.model = CoreModelAdmxrc4sx;
        break;
      }

      /* Initialize misc. registers in PCI9656 */
      pci9xxxInitRegs(pDevCtx, TRUE);

      /* Ensure that all interrupts are disabled in PCI9656 */
      pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaster = PLX_INTCSR_PCI_IE | PLX_INTCSR_LOC_IE | PLX_INTCSR_DMA0_IE | PLX_INTCSR_DMA1_IE;
      pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaskMaster = PLX_INTCSR_LOC_INT | PLX_INTCSR_DMA0_INT | PLX_INTCSR_DMA1_INT;
      dfPciMemWrite32(hBridge, &pRegs->INTCSR, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->INTCSR);
      dfPciMemWrite16(hBridge, &pRegs->DMACSR, 0x0808U);
      (void)dfPciMemRead16(hBridge, &pRegs->DMACSR);

      /* Enable target FPGA interrupt in Controller */
      pDevCtx->model.admxrc4.shadow.ictl = ADMXRC4_ICTL_ENFPGA;
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxrc4.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);
  
      /* Reset Flash and identify it */
      flashReset(pDevCtx);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      status = readWriteVpd(pDevCtx, FALSE, ADMXRC4_EEPROM_VPD_OFFSET, sizeof(vpdRaw), &vpdRaw);
      if (CoreVpdSuccess == status) {
        pDevCtx->info.vpd.bValid = TRUE;
        pVpd = &pDevCtx->info.vpd.pData->admxrc4.rev0;
        pVpd->modifications = dfLe16ToCpu(vpdRaw.modifications);
        pVpd->oscType = dfLe16ToCpu(vpdRaw.oscType);
        pVpd->flashType = dfLe16ToCpu(vpdRaw.flashType);
        pVpd->banks01 = dfLe8ToCpu(vpdRaw.banks01);
        pVpd->banks23 = dfLe8ToCpu(vpdRaw.banks23);
        pVpd->banks45 = dfLe8ToCpu(vpdRaw.banks45);
        pVpd->cardId = dfLe16ToCpu(vpdRaw.cardId);
        pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
        pVpd->serialLow = dfLe16ToCpu(vpdRaw.serialLow);
        pVpd->serialHigh = dfLe16ToCpu(vpdRaw.serialHigh);
        pVpd->logicRev = dfLe32ToCpu(vpdRaw.logicRev);
        pVpd->boardRev = dfLe32ToCpu(vpdRaw.boardRev);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to read VPD from EEPROM, status=0x%lx\n", (unsigned long)status));
      }

      /* Initialize ICS307 stuff */
      if (!ics307Init(
        pDevCtx,
        &pDevCtx->model.admxrc4.ics307.context,
        ADMXRC4_REF_OSC_FREQUENCY,
        Ics307InterfaceTypeXrc4,
        FALSE,
        &pDevCtx->hardware.model.pAdmxrc4->lclk,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS307 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      /* Initialize ICS8430-61 synthesizer stuff */
      if (!ics8430_61Init(
        pDevCtx,
        &pDevCtx->model.admxrc4.ics8430_61.context,
        ADMXRC4_REF_OSC_FREQUENCY,
        &pDevCtx->hardware.model.pAdmxrc4->mclk,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS8430-61 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[1].currentFrequency = fOut;
      }

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->LAS0BA);
      ls0r = dfPciMemRead32(hBridge, &pRegs->LAS0RR);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->LAS1BA);
      ls1r = dfPciMemRead32(hBridge, &pRegs->LAS1RR);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdmxrc4;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize ICS8430-61 stuff */
      ics8430_61Uninit(pDevCtx, &pDevCtx->model.admxrc4.ics8430_61.context);

      /* Deinitialize ICS307 stuff */
      ics307Uninit(pDevCtx, &pDevCtx->model.admxrc4.ics307.context);
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc4* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intcsr, clearMask, istat;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc4;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts, and mask the value with the interrupts we want */
  intcsr = dfPciMemRead32(hBridge, &pRegs->INTCSR) & pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMask;

  /* Clear the interrupts we saw */
  dfPciMemWrite32(hBridge, &pRegs->INTCSR, intcsr | pDevCtx->hardware.shadow.bridge.pci9xxx.intcsr);

  if (intcsr & PLX_INTCSR_LOC_INT) {
    dfDebugPrint(9, ("isr: MODEL interrupt\n"));
    istat = (uint8_t)(dfPciMemRead32(hModel, &pModelRegs->istat) & pDevCtx->model.admxrc4.shadow.ictl);
    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->istat, istat);
    if (istat & ADMXRC4_ISTAT_IRQFPGA) {
      dfDebugPrint(9, ("isr: target FPGA interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  }

  if (intcsr & (PLX_INTCSR_DMA0_INT | PLX_INTCSR_DMA1_INT)) {
    clearMask = 0x0U;
    if (intcsr & PLX_INTCSR_DMA0_INT) {
      dfDebugPrint(9, ("isr: DMA0 interrupt\n"));
      clearMask |= 0x0008U;
    }
    if (intcsr & PLX_INTCSR_DMA1_INT) {
      dfDebugPrint(9, ("isr: DMA1 interrupt\n"));
      clearMask |= 0x0800U;
    }
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, (uint16_t)(clearMask | pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr));
    if (intcsr & PLX_INTCSR_DMA0_INT) {
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[0].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, 0);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[0].dpc, (void*)(uintptr_t)0);
    }
    if (intcsr & PLX_INTCSR_DMA1_INT) {
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[1].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, 1);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[1].dpc, (void*)(uintptr_t)1);
    }
  }

  return intcsr ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated,
  boolean_t bUnmap)
{
  dfDebugPrint(2, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));
  
  if (bUnmap) {
    if (NULL != pDevCtx->hardware.bridge.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap PCI9656\n"));
      }
    }
    if (NULL != pDevCtx->hardware.model.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap CPLD\n"));
      }
    }
    if (NULL != pDevCtx->hardware.pBridged) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap FPGA space\n"));
      }
    }
    dfDebugPrint(2, ("mapBusResources: unmapped\n"));
  } else {
    pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, sizeof(Pci9656Registers), &pDevCtx->hardware.hBridge);
    if (NULL == pDevCtx->hardware.bridge.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map PCI9656\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped PCI9656 @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
    }
    pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hModel);
    if (NULL == pDevCtx->hardware.model.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map CPLD\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped CPLD @ 0x%08lx_%08lx to %p\n",
      dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.model.pGeneric));
    }
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hBridged);
    if (NULL == pDevCtx->hardware.pBridged) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map FPGA space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped FPGA space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.pBridged));
    }
    dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
  }
  
  return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  ModelRegsAdmxrc4* pModelRegs;
  DfMemoryHandle hModel;
  Ics307ProgramStatus ics307Status;
  Ics8430_61ProgramStatus ics8430_61Status;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdmxrc4;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save ICS307 state */
      ics307SaveState(pDevCtx, &pDevCtx->model.admxrc4.ics307.context);

      /* Save ICS8430-61 state */
      ics8430_61SaveState(pDevCtx, &pDevCtx->model.admxrc4.ics8430_61.context);
    } else { /* Powering up (from D3) */
      /* Reinitialize misc. registers in PCI9656 */
      pci9xxxInitRegs(pDevCtx, TRUE);

      /* Reset Flash device */
      flashReset(pDevCtx);

      /* Reenable target FPGA interrupt in Controller */
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxrc4.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Restore ICS8430-61 state */
      ics8430_61Status = ics8430_61RestoreState(pDevCtx, &pDevCtx->model.admxrc4.ics8430_61.context);
      if (Ics8430_61ProgramStatusSuccess != ics8430_61Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS8430-61 state, status=%u\n", ics8430_61Status));
      }

      /* Restore ICS307 state */
      ics307Status = ics307RestoreState(pDevCtx, &pDevCtx->model.admxrc4.ics307.context);
      if (Ics307ProgramStatusSuccess != ics307Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS307 state, status=%u\n", ics307Status));
      }
    }
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    return write93x6Buffer(pDevCtx, TRUE, addressLength9356, instructionLength9356, (uint32_t)address, (uint32_t)length, pData);
  } else {
    return read93x6Buffer(pDevCtx, TRUE, addressLength9356, instructionLength9356, (uint32_t)address, (uint32_t)length, pData);
  }
}

static boolean_t
validateBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated)
{
  if (pTranslated->variant.pci.numMemBar < 3) {
    dfDebugPrint(0, ("*** validateBusResources: less than 3 memory BARs\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[0].size < 0x200U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x200 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[1].size < 0x400000U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[2].size < sizeof(ModelRegsAdmxrc4)) {
    dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than %u bytes\n", sizeof(ModelRegsAdmxrc4)));
    return FALSE;
  }
  if (pTranslated->variant.pci.numIrq != 1) {
    dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
    return FALSE;
  }
  
  dfDebugPrint(2, ("validateBusResources: OK\n"));
  
  return TRUE;
}

void
adb3CoreBootstrapAdmxrc4(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc4: entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = pci9xxxDmaList;
  pDevCtx->methods.pDmaTransfer = pci9xxxDmaTransfer;
  pDevCtx->methods.pFlash = flash;
  pDevCtx->methods.pEnableInterrupts = pci9xxxEnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = pci9xxxWindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 2;
  pDevCtx->info.bootstrap.numDmaChannel = PCI9656_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numSensor = 0;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = eepromSize9356;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc4Rev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 16;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
