/*
** File: model_admxp.c  
** Project: ADB3 core driver
** Purpose: ADM-XP/ADP-XPI-specific code, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include "adb1_common.h"
#include "admxp.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "ics307_common.h"
#include "model_boot.h"

static boolean_t
admxpReadVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmxpRev0* pVpd = &pDevCtx->info.vpd.pData->admxp.rev0;
  VpdAdmxpRev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  unsigned int i;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->boardType = dfLe16ToCpu(vpdRaw.boardType);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->refClockFreq = dfLe32ToCpu(vpdRaw.refClockFreq);
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->ssram); i++) {
    pVpd->ssram[i].order = dfLe8ToCpu(vpdRaw.ssram[i].order);
  }
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->sdram); i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.sdram[i].order);
  }
  pVpd->cardId = dfLe32ToCpu(vpdRaw.cardId);
  pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
  pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->logicRev = (uint8_t)(0x10 + pDevCtx->pDevObj->hardwareId.variant.pci.revision);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static boolean_t
adpxpiReadVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdpxpiRev0* pVpd = &pDevCtx->info.vpd.pData->adpxpi.rev0;
  VpdAdpxpiRev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  unsigned int i;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->boardType = dfLe16ToCpu(vpdRaw.boardType);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->refClockFreq = dfLe32ToCpu(vpdRaw.refClockFreq);
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->sdram); i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.sdram[i].order);
  }
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->ssram); i++) {
    pVpd->ssram[i].order = dfLe8ToCpu(vpdRaw.ssram[i].order);
  }
  pVpd->cardId = dfLe32ToCpu(vpdRaw.cardId);
  pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
  pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->logicRev = (uint8_t)(0x10 + pDevCtx->pDevObj->hardwareId.variant.pci.revision);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
onIcs307ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307SynthCtx,
  Ics307ProgramRequest* pRequest,
  Ics307ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, ics307MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext*	pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return ics307MapProgramStatus(ics307Program(pDevCtx, &pDevCtx->model.admxp.ics307.context, &pDevCtx->model.admxp.ics307.programming, pClockWord, onIcs307ProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  VpdAdmxpRev0* pVpd = &pDevCtx->info.vpd.pData->admxp.rev0;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return ics307MapClockWordStatus(clockWordIcs307(flags, pVpd->refClockFreq, frequencyReq, frequencyMin, frequencyMax, TRUE, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}

static void
flashReset(
  Adb3CoreDeviceContext* pDevCtx)
{
  ModelRegsAdmxp* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t fldata = 0x3U << 26;

  dfDebugPrint(9, ("flashReset: entered\n"));

  pModelRegs = pDevCtx->hardware.model.pAdmxp;
  hModel = pDevCtx->hardware.hModel;

	/* Drive fl_rst_n, assert fl_rst_n */
	fldata |= (0x3U << 30);
	dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
	(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

	/* Deassert fl_rst_n */
	fldata &= ~(0x1U << 31);
	dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
	(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

	/* Stop driving fl_rst_n */
	fldata &= ~(0x1U << 30);
	dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
	(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

  /* Allow device time to recover */
  dfDelayThreadFor(dfMicrosecondsToTime(20));
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  ModelRegsAdmxp* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl, fldata = 0x3U << 26;
  uint16_t data16;

  dfDebugPrint(9, ("flash: bankIndex=%lu bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    (unsigned long)bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0 || location >= 0x2000000U) {
    return;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxp;
  hModel = pDevCtx->hardware.hModel;

	/* Drive fl_a, fl_sel_n, fl_oe_n, fl_we_n, assert fl_sel_n */
	flctl = (((uint32_t)location & 0x1FFFFFEU) >> 1) | (0x1fU << 25);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
  (void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	if (bWrite) {
		/* Drive fl_d (data to Flash) */
		fldata = (fldata & ~0xFFFFU) | (*pValue & 0xFFFFU) | 0x10000U;
		dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Assert fl_we_n (write enable for Flash) */
		flctl |= (0x1U << 31);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Deassert fl_we_n */
		flctl &= ~(0x1U << 31);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Stop driving fl_d (data to Flash) */
		fldata &= ~0x10000U;
		dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
    (void)dfPciMemRead32(hModel, &pModelRegs->fldata);
	} else {
		/* Assert fl_oe_n (output enable for Flash) */
		flctl |= (0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Read fl_d (data from Flash) */
		data16 = (uint16_t)dfPciMemRead32(hModel, &pModelRegs->fldata);
		/* Deassert fl_oe_n */
		flctl &= ~(0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		dfDebugPrint(9, ("flash: *pValue=0x%04X\n", (unsigned int)data16));
    *pValue = data16;
	}

	/* Stop driving fl_a, fl_sel_n, fl_oe_n, fl_we_n, assert fl_sel_n */
	flctl &= ~(0x1fU << 25);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
  (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdmxp* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxp;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADMXP_FPCTL_DONE) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone fpctl=0x%08lx *pbValue=%s\n", val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    *pbValue = TRUE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat *pbValue=%s\n", *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADMXP_FPCTL_INIT) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = pDevCtx->model.admxp.shadow.fpctl;
    if (*pbValue) {
      val32 = (uint32_t)(val32 | ADMXP_FPCTL_PROG);
    } else {
      val32 = (uint32_t)(val32 & ~ADMXP_FPCTL_PROG);
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, &pModelRegs->fpctl, val32);
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdmxp* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxp;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->fpsmap[0], val8);
    }
    
    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->fpsmap[0], val32);
      length32 -= 4;
    }
    
    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->fpsmap[0], val8);
    }
    
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->fpsmap[0]);
      *pBuffer8++ = val8;
    }
  }
  
  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxp* pModelRegs;
  DfMemoryHandle hModel;
  VpdAdmxpRev0* pVpd = &pDevCtx->info.vpd.pData->admxp.rev0;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  uint32_t fOut;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxp;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      switch (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice) {
      case PCI_SUBDEVICE_ADATA_ADPXPI_PCI:
      case PCI_SUBDEVICE_ADATA_ADPXPI_PCIX:
        pDevCtx->info.model = CoreModelAdpxpi;
        break;

      default:
        pDevCtx->info.model = CoreModelAdmxp;
        break;
      }

      /* Ensure that all interrupts are disabled in Bridge */
      pDevCtx->hardware.shadow.bridge.adb1.pictlMaster = ADB1_PICTL_ENMODEL | ADB1_PICTL_ENDMA_ALL | ADB1_PICTL_ENINTA;
      dfPciMemWrite32(hBridge, &pRegs->pictl, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->pictl);

      /* Enable target FPGA interrupt */
      pDevCtx->model.admxp.shadow.ictl = ADMXP_ICTL_ENFPGA;
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxp.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Enable DMA engines */
      adb1EnableDmaEngines(pDevCtx, TRUE);

      /* Establish FPGA control register shadow */
      pDevCtx->model.admxp.shadow.fpctl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpctl, pDevCtx->model.admxp.shadow.fpctl);

      /* Reset Flash and identify it */
      flashReset(pDevCtx);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = ADMXP_FLASH_TARGET_AREA_START;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - ADMXP_FLASH_TARGET_AREA_START;
        switch (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice) {
        case PCI_SUBDEVICE_ADATA_ADPXPI_PCI:
        case PCI_SUBDEVICE_ADATA_ADPXPI_PCIX:
          adpxpiReadVpd(pDevCtx);
          break;

        default:
          admxpReadVpd(pDevCtx);
          break;
        }
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize ICS307 stuff */
      if (!ics307Init(
        pDevCtx,
        &pDevCtx->model.admxp.ics307.context,
        pVpd->refClockFreq,
        Ics307InterfaceTypeXpl,
        TRUE,
        &pDevCtx->hardware.model.pAdmxp->clock,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS307 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->ls0b);
      ls0r = dfPciMemRead32(hBridge, &pRegs->ls0r);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->ls1b);
      ls1r = dfPciMemRead32(hBridge, &pRegs->ls1r);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdmxp;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize ICS307 stuff */
      ics307Uninit(pDevCtx, &pDevCtx->model.admxp.ics307.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Disable DMA engines */
        adb1EnableDmaEngines(pDevCtx, FALSE);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxp* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int channel, n;
  uint32_t pistat, istat;

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxp;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts, and mask the value with the interrupts we want */
  pistat = dfPciMemRead32(hBridge, &pRegs->pistat) & pDevCtx->hardware.shadow.bridge.adb1.pictl;
  dfDebugPrint(8, ("isr: pistat=0x%08lx\n", (unsigned long)pistat));

  /* Clear the interrupts that we saw */
  dfPciMemWrite32(hBridge, &pRegs->pistat, pistat);

  if (pistat & ADB1_PISTAT_IRQMODEL) {
    dfDebugPrint(9, ("isr: MODEL interrupt\n"));
    istat = dfPciMemRead32(hModel, &pModelRegs->istat) & pDevCtx->model.admxp.shadow.ictl;
    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->istat, istat);
    if (istat & ADMXP_ISTAT_IRQFPGA) {
      dfDebugPrint(9, ("isr: target FPGA interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (pistat & ADB1_PISTAT_IRQDMA(channel)) {
      dfDebugPrint(9, ("isr: DMA channel %u interrupt\n", channel));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, channel);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  return pistat ? TRUE : FALSE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Ics307ProgramStatus ics307Status;
  ModelRegsAdmxp* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdmxp;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save ICS307 state */
      ics307SaveState(pDevCtx, &pDevCtx->model.admxp.ics307.context);
    } else { /* Powering up (from D3) */
      /* Reset Flash device */
      flashReset(pDevCtx);

      /* Reenable target FPGA interrupt */
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxp.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Restore ICS307 state */
      ics307Status = ics307RestoreState(pDevCtx, &pDevCtx->model.admxp.ics307.context);
      if (Ics307ProgramStatusSuccess != ics307Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS307 state, status=%u\n", ics307Status));
      }
    }
  }
}

static boolean_t
mapBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated,
  boolean_t bUnmap)
{
  dfDebugPrint(2, ("admxpMapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));
  
  if (bUnmap) {
    if (NULL != pDevCtx->hardware.bridge.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
        dfDebugPrint(0, ("*** admxpMapBusResources: failed to unmap Bridge regs\n"));
      }
    }
    if (NULL != pDevCtx->hardware.model.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
        dfDebugPrint(0, ("*** admxpMapBusResources: failed to unmap Model regs\n"));
      }
    }
    if (NULL != pDevCtx->hardware.pBridged) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
        dfDebugPrint(0, ("*** admxpMapBusResources: failed to unmap FPGA space\n"));
      }
    }
    dfDebugPrint(2, ("admxpMapBusResources: unmapped\n"));
  } else {
    pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
    if (NULL == pDevCtx->hardware.bridge.pGeneric) {
      dfDebugPrint(0, ("*** admxpMapBusResources: failed to map Bridge regs\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("admxpMapBusResources: mapped Bridge regs @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
    }
    pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hModel);
    if (NULL == pDevCtx->hardware.model.pGeneric) {
      dfDebugPrint(0, ("*** admxpMapBusResources: failed to map Model regs\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("admxpMapBusResources: mapped Model regs @ 0x%08lx_%08lx to %p\n",
      dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.model.pGeneric));
    }
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hBridged);
    if (NULL == pDevCtx->hardware.pBridged) {
      dfDebugPrint(0, ("*** admxpMapBusResources: failed to map FPGA space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("admxpMapBusResources: mapped FPGA space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.pBridged));
    }
    dfDebugPrint(2, ("admxpMapBusResources: mapped OK\n"));
  }
  
  return TRUE;
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  CoreFlashStatus flashStatus;

  if (CHECK_BOUNDS(address, length, pDevCtx->info.flash[0].totalSize)) {
    return CoreVpdInvalidRegion;
  }
	if (bWrite) {
    flashStatus = flashWrite(pDevCtx, 0, address, length, pData);
	} else {
		flashStatus = flashRead(pDevCtx, 0, address, length, pData);
	}
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}

static CoreVpdStatus
syncVpd(
	Adb3CoreDeviceContext* pDevCtx)
{
  CoreFlashStatus flashStatus;

  flashStatus = flashSync(pDevCtx, 0);
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}

static boolean_t
validateBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated)
{
  if (pTranslated->variant.pci.numMemBar < 3) {
    dfDebugPrint(0, ("*** admxpValidateBusResources: less than 3 memory BARs\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[0].size < sizeof(Adb1BridgeRegisters)) {
    dfDebugPrint(0, ("*** admxpValidateBusResources: BAR0 smaller than 0x%lx bytes\n", (unsigned long)sizeof(Adb1BridgeRegisters)));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[1].size < 0x400000U) {
    dfDebugPrint(0, ("*** admxpValidateBusResources: BAR1 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[2].size < sizeof(ModelRegsAdmxp)) {
    dfDebugPrint(0, ("*** admxpValidateBusResources: BAR2 smaller than 0x%lx bytes\n", (unsigned long)sizeof(ModelRegsAdmxp)));
    return FALSE;
  }
  if (pTranslated->variant.pci.numIrq != 1) {
    dfDebugPrint(0, ("*** admxpValidateBusResources: not exactly 1 interrupt\n"));
    return FALSE;
  }
  
  dfDebugPrint(2, ("admxpValidateBusResources: OK\n"));
  
  return TRUE;
}

static void
bootstrap(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxp entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb1DmaList;
  pDevCtx->methods.pDmaTransfer = adb1DmaTransfer;
  pDevCtx->methods.pFlash = flash;
  pDevCtx->methods.pEnableInterrupts = adb1EnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pSyncVpd = syncVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb1WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 1;
  pDevCtx->info.bootstrap.numDmaChannel = ADMXP_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1U;
  pDevCtx->info.bootstrap.numSensor = 0;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = ADMXP_VPD_SIZE_IN_FLASH;
  pDevCtx->info.bootstrap.vpdOffset = ADMXP_VPD_OFFSET_IN_FLASH;
  switch (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice) {
  case PCI_SUBDEVICE_ADATA_ADPXPI_PCI:
  case PCI_SUBDEVICE_ADATA_ADPXPI_PCIX:
    pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdpxpiRev0);
    break;

  default:
    pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxpRev0);
    break;
  }
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}

void
adb3CoreBootstrapAdmxp(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxp entered\n"));
  bootstrap(pDevCtx);
}

void
adb3CoreBootstrapAdpxpi(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdpxpi entered\n"));
  bootstrap(pDevCtx);
}
