/*
** File: model_adb3.c  
** Project: ADB3 core driver
** Purpose: ADB3 generic code, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2009-2010
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "device.h"
#include "model_boot.h"
#include "flash.h"

boolean_t
adb3VerifyChecksum(
	Adb3CoreDeviceContext* pDevCtx,
  uint16_t vpdLength)
{
  uint8_t buff[16];
  uint8_t sum = 0, checksum;
  uint16_t offset = 0;
  uint16_t remaining = (uint16_t)(vpdLength - 1U);
  uint16_t i, chunk;

  if (vpdLength < 5) {
  	dfDebugPrint(0, ("*** verifyChecksum: invalid VPD length %lu(0x%lx)\n",
      (unsigned long)vpdLength, (unsigned long)vpdLength));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }

  /* Get checksum from VPD memory; always last byte of structure */
  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset + remaining, sizeof(checksum), &checksum);

  /* Compute checksum over all but last byte of structure */
  while (remaining) {
    chunk = (uint16_t)((remaining > sizeof(buff)) ? sizeof(buff) : remaining);
    flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset + offset, chunk, buff);
    for (i = 0; i < chunk; i++) {
      sum = (uint8_t)(sum + buff[i]);
    }
    offset = (uint16_t)(offset + chunk);
    remaining = (uint16_t)(remaining - chunk);
  }
  sum = (uint8_t)-sum;

  if (sum == checksum) {
    return TRUE;
  } else {
  	dfDebugPrint(0, ("*** verifyChecksum: checksum invalid, computed 0x%02lx found 0x%02lx\n",
      (unsigned long)sum, (unsigned long)checksum));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }
}

void
adb3DmaList(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWriteToDevice,
  boolean_t bFixedLocal,
  DfDmaMapperNode* pTable,
  unsigned int tableLength,
  DmaNodeTablePosition* pTablePosition,
  void* pDescriptorBuffer,
  unsigned int maxDescriptor,
  uint64_t descriptorBusAddress)
{
  const boolean_t bWorkAround4kCrossing = pDevCtx->info.bootstrap.bWorkAround4kCrossing;
  const unsigned int fourk = 0x1000U, fourkMask = fourk - 1U;
  Adb3DmaDescriptor* pDescriptor = (Adb3DmaDescriptor*)pDescriptorBuffer;
  DfDmaMapperNode* pTableTmp;
  uint64_t currDescBusAddr, nextDescBusAddr, dataBusAddr = 0U, localAddr = 0U;
  unsigned int i;
  uint32_t ctl;
  size_t remaining;
  unsigned int chunk, maxChunk;
  unsigned int numDesc = 0;
  unsigned int index;
  size_t offset;

  dfDebugPrint(7, ("adb3DmaList: entered, pDevCtx=%p bWriteToDevice=%s bFixedLocal=%s pTable=%p tableLength=%u *pTablePosition={%u, 0x%08lx_%08lx} pDescriptorBuffer=%p maxDescriptor=%u descriptorBusAddrss=0x%08lx_%08lx\n",
    (void*)pDevCtx, bWriteToDevice ? "TRUE" : "FALSE", bFixedLocal ? "TRUE" : "FALSE", (void*)pTable, tableLength, pTablePosition->index, dfSplitUint64((uint64_t)pTablePosition->offset), pDescriptorBuffer, maxDescriptor, dfSplitUint64(descriptorBusAddress)));

  ctl = bWriteToDevice ? ADB3DMACTL_PCI_TO_LOCAL : ADB3DMACTL_LOCAL_TO_PCI;
  ctl |= bFixedLocal ? ADB3DMACTL_FIXED_LOCAL_ADDR : 0;
  currDescBusAddr = descriptorBusAddress;
  index = pTablePosition->index;
  offset = pTablePosition->offset;
  pTableTmp = pTable + index;
  if (bWorkAround4kCrossing) {
    i = index;
    remaining = 0;
    for (i = index; i < tableLength && numDesc < maxDescriptor; numDesc++) {
      if (0 == remaining) {
        dataBusAddr = pTableTmp->busAddress + offset;
        if (!bFixedLocal) {
          localAddr = pTableTmp->localAddress + offset;
        }
        remaining = pTableTmp->length - offset;
        maxChunk = fourk - ((unsigned int)dataBusAddr & fourkMask);
      } else {
        maxChunk = fourk;
      }
      if (maxChunk > remaining) {
        maxChunk = (unsigned int)remaining; /* Cast is safe due to above comparison */
      }
      nextDescBusAddr = currDescBusAddr + sizeof(Adb3DmaDescriptor);
      /* Cast is safe since ((unsigned int)remaining) only used if remaining <= maxChunk */
      chunk = (remaining > maxChunk) ? maxChunk : (unsigned int)remaining;
      pDescriptor->pal = dfCpuToLe32((uint32_t)dataBusAddr);
      pDescriptor->pah = dfCpuToLe32((uint32_t)(dataBusAddr >> 32));
      pDescriptor->sizl = dfCpuToLe32((uint32_t)chunk);
      pDescriptor->sizh = dfCpuToLe32(0U);
      pDescriptor->ndl = dfCpuToLe32((uint32_t)nextDescBusAddr);
      pDescriptor->ndh = dfCpuToLe32((uint32_t)(nextDescBusAddr >> 32));
      pDescriptor->lal = dfCpuToLe32((uint32_t)localAddr);
      pDescriptor->ctl = dfCpuToLe32(ctl | ((uint32_t)(localAddr >> 32) & 0x7FU));
      pDescriptor++;
      currDescBusAddr = nextDescBusAddr;
      dataBusAddr += chunk;
      if (!bFixedLocal) {
        localAddr += chunk;
      }
      offset += chunk;
      remaining -= chunk;
      if (0 == remaining) {
        pTableTmp++;
        i++;
        offset = 0;
      }
    }
  } else {
    boolean_t bContinueDescriptor = FALSE;
    boolean_t bLastDescriptor = (index + 1 == tableLength || maxDescriptor <= 1) ? TRUE : FALSE;
    uint64_t endLocalAddr = 0U, endDataBusAddr = 0U;
    size_t size = 0U, entryLength;

    nextDescBusAddr = currDescBusAddr + sizeof(Adb3DmaDescriptor);
    i = index;
    for (;;) {
      entryLength = pTableTmp->length;
      if (bContinueDescriptor) {
        /* Continuation of last descriptor, as this table entry is contiguous in bus & local address w.r.t to previous table entry */
        size += entryLength;
        if (!bFixedLocal) {
          endLocalAddr += entryLength;
        }
        endDataBusAddr += entryLength;
      } else {
        /* This table entry is not contiguous both in bus & local address w.r.t to previous table entry */
        dataBusAddr = pTableTmp->busAddress + offset;
        localAddr = pTableTmp->localAddress;
        size = entryLength - offset;
        if (!bFixedLocal) {
          localAddr += offset;
          endLocalAddr = localAddr + size;
        } else {
          endLocalAddr = localAddr;
        }
        endDataBusAddr = dataBusAddr + size;
      }
      pTableTmp++;
      if (bLastDescriptor || pTableTmp->busAddress != endDataBusAddr || pTableTmp->localAddress != endLocalAddr) {
        pDescriptor->pal = dfCpuToLe32((uint32_t)dataBusAddr);
        pDescriptor->pah = dfCpuToLe32((uint32_t)(dataBusAddr >> 32));
        pDescriptor->sizl = dfCpuToLe32((uint32_t)size);
        pDescriptor->sizh = dfCpuToLe32(0U);
        pDescriptor->ndl = dfCpuToLe32((uint32_t)nextDescBusAddr);
        pDescriptor->ndh = dfCpuToLe32((uint32_t)(nextDescBusAddr >> 32));
        pDescriptor->lal = dfCpuToLe32((uint32_t)localAddr);
        pDescriptor->ctl = dfCpuToLe32(ctl | ((uint32_t)(localAddr >> 32) & 0x7FU));
        pDescriptor++;
        numDesc++;
        currDescBusAddr = nextDescBusAddr;
        nextDescBusAddr += sizeof(Adb3DmaDescriptor);
        bContinueDescriptor = FALSE;
      } else {
        bContinueDescriptor = TRUE;
      }
      if (bLastDescriptor) {
        break;
      }
      offset = 0;
      i++;
      if (numDesc + 1 == maxDescriptor || i + 1 == tableLength) {
        bLastDescriptor = TRUE;
      }
    }
  }
  pDescriptor--;
  pDescriptor->ndl = dfCpuToLe32(0U);
  pDescriptor->ctl |= dfCpuToLe32(ADB3DMACTL_ENDCHAIN | (bWriteToDevice ? ADB3DMACTL_IRQ_ON_FPGA_COMPLETION : ADB3DMACTL_IRQ_ON_HOST_COMPLETION));

  /* Cast is safe as pTableTmp cannot be much greater than pTable */
  pTablePosition->index = (unsigned int)(pTableTmp - pTable);
  pTablePosition->offset = offset;
  dfDebugPrint(7, ("adb3DmaList: finished, *pTablePosition={%u, 0x%08lx_%08lx}\n", pTablePosition->index, dfSplitUint64((uint64_t)pTablePosition->offset)));

#if DF_DBG_BUILD
  if (g_dfDebugLevel >= 7) {
    dfDebugPrint(0, ("adb3DmaList: dump of DMA engine linked list\n"));
    adb3DumpDmaDescriptors(pDevCtx, (Adb3DmaDescriptor*)pDescriptorBuffer, descriptorBusAddress, numDesc);
  }
#endif
}

void
adb3DmaTransfer(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channel,
  boolean_t bStart,
  void* pDescriptorBuffer,
  uint64_t descriptorBusAddress)
{
  const unsigned int maxAbortPoll = 20;
  Adb3DmaDescriptor* pDescriptor = (Adb3DmaDescriptor*)pDescriptorBuffer;
  Adb3GenericRegs* pRegs = pDevCtx->hardware.bridge.pAdb3;
  Adb3DmaRegs* pDmaRegs = &pRegs->dmaEngine[channel];
  uint32_t alignment;
  uint32_t lal, pal, ctl;
  DfMemoryHandle handle;
  unsigned int i;

  dfDebugPrint(6, ("adb3DmaTransfer: entered, pDevCtx=%p channel=%lu bStart=%s pDescriptorBuffer=%p busAddress=0x%08lx_%08lx\n",
    (void*)pDevCtx, (unsigned long)channel, bStart ? "TRUE" : "FALSE", pDescriptorBuffer, dfSplitUint64(descriptorBusAddress)));

  handle = pDevCtx->hardware.hBridge;

  if (bStart) {
    lal = dfLe32ToCpu(pDescriptor->lal & 0xfU);
    pal = dfLe32ToCpu(pDescriptor->pal & 0xfU);
    ctl = dfLe32ToCpu(pDescriptor->ctl);
    if (ctl & ADB3DMACTL_PCI_TO_LOCAL) {
      if (pal > lal) {
        alignment = ((0x10U + lal - pal) & 0xfU) | ADB3DMAALIGN_LOCAL_IGNORE_FIRST_WORD;
      } else {
        alignment = (lal - pal) | ADB3DMAALIGN_LOCAL_USE_FIRST_WORD;
      }
      dfPciMemWrite32(handle, &pDmaRegs->irqEnable, ADB3DMAIRQ_MPTL_WRITE_WORKER_COMPLETED | ADB3DMAIRQ_ABORT);
    } else {
      if (pal >= lal) {
        alignment = ((0x10U + lal - pal) & 0xf) | ADB3DMAALIGN_LOCAL_USE_FIRST_WORD;
      } else {
        alignment = (lal - pal) | ADB3DMAALIGN_LOCAL_IGNORE_FIRST_WORD;
      }
      dfPciMemWrite32(handle, &pDmaRegs->irqEnable, ADB3DMAIRQ_PCIE_WRITE_WORKER_COMPLETED | ADB3DMAIRQ_ABORT);
    }
    dfPciMemWrite32(handle, &pDmaRegs->alignment, alignment);
    dfPciMemWrite32(handle, &pDmaRegs->nextAddrHigh, (uint32_t)(descriptorBusAddress >> 32));

#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("adb3DmaTransfer: DMA channel %lu registers before starting on hardware:\n", (unsigned long)channel));
      adb3DumpDmaRegisters(pDevCtx, channel);
    }
#endif

    /* Writing to nextAddrLow starts the DMA transfer */
    dfPciMemWrite32(handle, &pDmaRegs->nextAddrLow, (uint32_t)descriptorBusAddress); 

#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("adb3DmaTransfer: DMA channel %lu registers after starting on hardware:\n", (unsigned long)channel));
      adb3DumpDmaRegisters(pDevCtx, channel);
    }
#endif
  } else {
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("adb3DmaTransfer: DMA channel %lu registers before attempt to abort:\n", (unsigned long)channel));
      adb3DumpDmaRegisters(pDevCtx, channel);
    }
#endif

    /* Write to ABORT register */ 
    dfPciMemWrite32(handle, &pDmaRegs->control.abort, 0);
    for (i = 0; i < maxAbortPoll; i++) {
      (void)dfPciMemRead32(handle, &pDmaRegs->control.status);
    }
    /* Assert cleanup strobe - should generate an interrupt */
    dfPciMemWrite32(handle, &pDmaRegs->cleanup.cleanup, 0);
    dfPciMemRead32(handle, &pDmaRegs->cleanup.cleanup);

#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("adb3DmaTransfer: DMA channel %lu registers after attempt to abort:\n", (unsigned long)channel));
      adb3DumpDmaRegisters(pDevCtx, channel);
    }
#endif
  }
}

unsigned int
adb3CountDmaEngines(
  Adb3CoreDeviceContext* pDevCtx)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  unsigned int i, n;
  uint32_t val32;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  n = 0;
  for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
    /*
    ** On the ADM-XRC-6T-ADV8, there may be up to 4 DMA channels, but we
    ** don't know how many until we get here. If a DMA channel is present,
    ** the nextAddrHigh register is writable and returns the last data
    ** written.
    ** Some DMA engines might only work in a particular direction, and we
    ** detect that by checking both 16-bit halves of the FIFO level registers.
    */
    dfPciMemWrite32(hBridge, &pRegs->dmaEngine[i].nextAddrHigh, 0xDEADBEEFU);
    val32 = dfPciMemRead32(hBridge, &pRegs->dmaEngine[i].nextAddrHigh);
    if (0xDEADBEEFU != val32) {
      /* DMA engine not present; don't look for any more DMA engines. */
      break;
    }
    n++;
    val32 = dfPciMemRead32(hBridge, &pRegs->dmaEngine[i].cleanup.fifoStatus);
    if (0U == (val32 & 0x0000ffffU)) {
      /* Does not support FPGA to Host transfers */
      pDevCtx->info.bootstrap.dmaRestrictions[i].bNoFpgaToHost = 1U;
    }
    if (0U == (val32 & 0xffff0000U)) {
      /* Does not support Host to FPGA transfers */
      pDevCtx->info.bootstrap.dmaRestrictions[i].bNoHostToFpga = 1U;
    }
  }
  return n;
}

void
adb3InitDmaEngines(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int count)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  unsigned int i;
  uint32_t val32;
  uint32_t mptlReadBLen = 0, mptlWriteBLen = 0; /* 0 => Don't change register */

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  /* Initialize DMA engine registers */
  dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "DmaMptlReadBurstLength", &mptlReadBLen);
  dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "DmaMptlWriteBurstLength", &mptlWriteBLen);
  for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
    /* Request up to 32 x 16-byte OCP words on PCIe side */
    dfPciMemWrite32(hBridge, &pRegs->dmaEngine[i].hostControl, 0x20U);

    /* Set MPTL read/write burst length for DMA engine OCP ports */
    val32 = 0x00080008U; /* Default read/write MPTL burst length for DMA engines is 8 x OCP words */
    if (0 != mptlWriteBLen) {
      val32 = (val32 & ~0x000003FFU) | ((mptlWriteBLen & 0x3FFU) << 0);
    }
    if (0 != mptlReadBLen) {
      val32 = (val32 & ~0x03FF0000U) | ((mptlReadBLen & 0x3FFU) << 16);
    }
    dfPciMemWrite32(hBridge, &pRegs->dmaEngine[i].fpgaControl, val32);
  }
}

void
adb3DumpDmaDescriptors(
	Adb3CoreDeviceContext* pDevCtx,
  Adb3DmaDescriptor* pDescriptors,
  uint64_t descBus,
  unsigned int count)
{
  Adb3DmaDescriptor* pCurrent;
  unsigned int i;

  if (sizeof(void*) == 4) {
    dfDebugPrint(0, ("  ### DescVirt   DescBus              [ PAH      PAL      SIZH     SIZL     NDH      NDL      LAL      CTL      ]\n"));
    for (i = 0; i < count; i++) {
      pCurrent = &pDescriptors[i];
      dfDebugPrint(0, ("  %03lu %p 0x%08lx_%08lx   %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx\n",
        (unsigned long)i, (void*)pCurrent, dfSplitUint64(descBus),
        (unsigned long)dfLe32ToCpu(pCurrent->pah), (unsigned long)dfLe32ToCpu(pCurrent->pal),
        (unsigned long)dfLe32ToCpu(pCurrent->sizh), (unsigned long)dfLe32ToCpu(pCurrent->sizl),
        (unsigned long)dfLe32ToCpu(pCurrent->ndh), (unsigned long)dfLe32ToCpu(pCurrent->ndl),
        (unsigned long)dfLe32ToCpu(pCurrent->lal), (unsigned long)dfLe32ToCpu(pCurrent->ctl)));
      descBus += sizeof(Adb3DmaDescriptor);
    }
  } else {
    dfDebugPrint(0, ("  ### DescVirt           DescBus              [ PAH      PAL      SIZH     SIZL     NDH      NDL      LAL      CTL      ]\n"));
    for (i = 0; i < count; i++) {
      pCurrent = &pDescriptors[i];
      dfDebugPrint(0, ("  %03lu %p 0x%08lx_%08lx   %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx\n",
        (unsigned long)i, (void*)pCurrent, dfSplitUint64(descBus),
        (unsigned long)dfLe32ToCpu(pCurrent->pah), (unsigned long)dfLe32ToCpu(pCurrent->pal),
        (unsigned long)dfLe32ToCpu(pCurrent->sizh), (unsigned long)dfLe32ToCpu(pCurrent->sizl),
        (unsigned long)dfLe32ToCpu(pCurrent->ndh), (unsigned long)dfLe32ToCpu(pCurrent->ndl),
        (unsigned long)dfLe32ToCpu(pCurrent->lal), (unsigned long)dfLe32ToCpu(pCurrent->ctl)));
      descBus += sizeof(Adb3DmaDescriptor);
    }
  }
}

void
adb3DumpDmaRegisters(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int channel)
{
  static const char* regNames[16] = {
    "status", "fifoStatus", "irqStatus", "irqEnable",
    "nextAddrLow", "nextAddrHigh", "hostControl", "fpgaControl",
    "alignment", "descPause", "reservedA", "reservedB",
    "reservedC", "reservedD", "reservedE", "reservedF"
  };
  DfMemoryHandle handle;
  uint32_t* pRegs = (uint32_t*)&pDevCtx->hardware.bridge.pAdb3->dmaEngine[channel];
  uint32_t regValue[16];
  unsigned int i;

  handle = pDevCtx->hardware.hBridge;
  for (i = 0; i < 16; i++) {
    regValue[i] = dfPciMemRead32(handle, pRegs + i);
  }
  for (i = 0; i < 4; i++) {
    dfDebugPrint(0, ("  %12s=0x%08lx %12s=0x%08lx %12s=0x%08lx %12s=0x%08lx\n",
      regNames[i], (unsigned long)regValue[i],
      regNames[i + 4], (unsigned long)regValue[i + 4],
      regNames[i + 8], (unsigned long)regValue[i + 8],
      regNames[i + 12], (unsigned long)regValue[i + 12]));
  }
}

void
adb3EnableInterrupts(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bEnable)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  DfInterruptObject* pInterruptObject;
  DfSpinLockFlags f;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  dfDebugPrint(1, ("adb3EnableInterrupts: entered bEnable=%s\n", bEnable ? "TRUE" : "FALSE"));

  pInterruptObject = &pDevCtx->interrupt.object;
  f = dfInterruptSpinLockGet(pInterruptObject);
  if (bEnable) {
    pDevCtx->hardware.shadow.bridge.adb3.irqEnable = pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster;
    if (pDevCtx->interrupt.bPolledMode) {
      dfInterruptSpinLockPut(pInterruptObject, f);
		  dfDebugPrint(2, ("adb3EnableInterrupts: interrupts NOT enabled due to operating in polled mode\n"));
    } else {
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);
      dfInterruptSpinLockPut(pInterruptObject, f);
		  dfDebugPrint(2, ("adb3EnableInterrupts: interrupts enabled, irqEnable=0x%08lx\n",
        (unsigned long)dfPciMemRead32(hBridge, &pRegs->irqEnable)));
    }
  } else {
    pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
    dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
    dfPciMemRead32(hBridge, &pRegs->irqEnable);
    dfInterruptSpinLockPut(pInterruptObject, f);
		dfDebugPrint(2, ("adb3EnableInterrupts: interrupts disabled, irqEnable=0x%08lx\n",
      (unsigned long)dfPciMemRead32(hBridge, &pRegs->irqEnable)));
	}
}

boolean_t
adb3NeedMsiWorkaround(
	Adb3CoreDeviceContext* pDevCtx)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  uint32_t feature;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  feature = dfPciMemRead32(hBridge, &pRegs->feature);

  return (feature & ADB3FEAT_MSIFIX) ? FALSE : TRUE;
}

boolean_t
adb3HasIcapInterface(
	Adb3CoreDeviceContext* pDevCtx)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  uint32_t feature;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  feature = dfPciMemRead32(hBridge, &pRegs->feature);

  return (feature & ADB3FEAT_ICAP_IF) ? TRUE : FALSE;
}

void
adb3ClearErrors(
	Adb3CoreDeviceContext* pDevCtx)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  dfPciMemWrite32(hBridge, &pRegs->pcieBrgStatus, 0);
  dfPciMemRead32(hBridge, &pRegs->pcieBrgStatus);
}

void
adb3GetDeviceStatus(
	Adb3CoreDeviceContext* pDevCtx,
  CoreDeviceStatus* pDeviceStatus)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  uint32_t pcieBrgStatus;
  unsigned int i;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  pcieBrgStatus = dfPciMemRead32(hBridge, &pRegs->pcieBrgStatus);

  dfZeroMemory(pDeviceStatus, sizeof(*pDeviceStatus));

  /* Window status */
  if (pDevCtx->info.bootstrap.numWindow >= 4) {
    if (pcieBrgStatus & (0x1U << 24)) {
      /* Timeout on BAR0 / Window 3 */
      pDeviceStatus->window |= (0x1U << 3);
    }
    if (pcieBrgStatus & (0x1U << 25)) {
      /* Timeout on BAR1 / Window 2 */
      pDeviceStatus->window |= (0x1U << 2);
    }
    if (pcieBrgStatus & (0x1U << 26)) {
      /* Timeout on BAR2&3 / Window 0&1 */
      pDeviceStatus->window |= (0x3U << 0);
    }
  } else {
    if (pcieBrgStatus & (0x1U << 24)) {
      /* Timeout on BAR0 / Window 2 */
      pDeviceStatus->window |= (0x1U << 2);
    }
    if (pcieBrgStatus & (0x1U << 25)) {
      /* Timeout on BAR1 / Window 1 */
      pDeviceStatus->window |= (0x1U << 1);
    }
    if (pcieBrgStatus & (0x1U << 26)) {
      /* Timeout on BAR2 / Window 0 */
      pDeviceStatus->window |= (0x1U << 0);
    }
  }

  /* DMA engine status */
  for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
    if (pcieBrgStatus & (0x1U << (i + 20))) {
      /* DMA engine PCI-E read timeout */
      pDeviceStatus->dmaEngine |= 0x1U << i;
    }
  }

  /* Direct Master status */
  if (pcieBrgStatus & (0x1U << 19)) {
    /* Direct Master PCI-E read timeout */
    pDeviceStatus->directMaster |= 0x1U << 0;
  }
  if (pcieBrgStatus & (0x1U << 27)) {
    /* Direct Master bad OCP command */
    pDeviceStatus->directMaster |= 0x1U << 1;
  }
}

CoreWindowConfigStatus
adb3GetWindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  CoreWindowConfig* pConfig)
{
  static const uint32_t maskHigh = 0x0U;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  DfSpinLockFlags f;
  uint32_t maskLow, pageLow, pageHigh;
  uint64_t mask, page;

  switch (windowIndex) {
  case 0:
  case 1:
    /* Following three members are not valid for ADB3 devices */
    pConfig->width = COREWINDOWCONFIG_WIDTH_DEFAULT;
    pConfig->prefetch = COREWINDOWCONFIG_PREFETCH_DEFAULT;
    pConfig->burst = COREWINDOWCONFIG_BURST_DEFAULT;
    pRegs = pDevCtx->hardware.bridge.pAdb3;
    hBridge = pDevCtx->hardware.hBridge;
    f = dfSpinLockGet(&pDevCtx->info.window[0].lock);
    maskLow = dfPciMemRead32(hBridge, &pRegs->fpgaMask);
    pageLow = dfPciMemRead32(hBridge, &pRegs->fpgaPageLow);
    pageHigh = dfPciMemRead32(hBridge, &pRegs->fpgaPageHigh);
    dfSpinLockPut(&pDevCtx->info.window[0].lock, f);
    mask = (uint64_t)maskLow | ((uint64_t)maskHigh << 32);
    page = (uint64_t)pageLow | ((uint64_t)pageHigh << 32);
    pConfig->base = page & ~mask;
    break;

  case 2:
  case 3:
    return CoreWindowConfigNotSupported;

  default:
    return CoreWindowConfigInvalidIndex;
  }

  return CoreWindowConfigSuccess;
}

CoreWindowConfigStatus
adb3SetWindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  uint32_t flags,
  const CoreWindowConfig* pConfig)
{
  static const uint32_t unsupportedFlags = COREWINDOWCONFIG_SET_WIDTH | COREWINDOWCONFIG_SET_PREFETCH | COREWINDOWCONFIG_SET_BURST;
  static const uint32_t maskHigh = 0x0U;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  DfSpinLockFlags f;
  uint32_t maskLow;
  uint64_t mask, page;

  switch (windowIndex) {
  case 0:
  case 1:
    if (flags & unsupportedFlags) {
      return CoreWindowConfigNotSupported;
    }
    if (flags & COREWINDOWCONFIG_SET_BASE) {
      pRegs = pDevCtx->hardware.bridge.pAdb3;
      hBridge = pDevCtx->hardware.hBridge;
      f = dfSpinLockGet(&pDevCtx->info.window[0].lock);
      maskLow = dfPciMemRead32(hBridge, &pRegs->fpgaMask);
      mask = (uint64_t)maskLow | ((uint64_t)maskHigh << 32);
      page = pConfig->base;
      if (page & mask) {
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
      dfPciMemWrite32(hBridge, &pRegs->fpgaPageLow, (uint32_t)page);
      dfPciMemWrite32(hBridge, &pRegs->fpgaPageHigh, (uint32_t)(page >> 32));
      pDevCtx->info.window[windowIndex].localBase = page;
      dfSpinLockPut(&pDevCtx->info.window[0].lock, f);
    }
    break;

  case 2:
  case 3:
    return CoreWindowConfigNotSupported;

  default:
    return CoreWindowConfigInvalidIndex;
  }

  return CoreWindowConfigSuccess;
}

CoreWindowConfigStatus
adb3WindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  boolean_t bConfigure,
  uint32_t flags,
  CoreWindowConfig* pConfig)
{
  if (bConfigure) {
    return adb3SetWindowConfig(pDevCtx, windowIndex, flags, pConfig);
  } else {
    return adb3GetWindowConfig(pDevCtx, windowIndex, pConfig);
  }
}

boolean_t
adb3ForcePcieGen(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int speed)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  uint32_t val32;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  switch (speed) {
  case 1U:
  case 2U:
    /*
    ** Force Bridge to retrain at PCIe Gen 1 or 2.
    **
    ** Relevant bits of register are connected to Xilinx Gen 2 PCIe core signals:
    ** 24    : pl_directed_link_auton
    ** 25:26 : pl_directed_link_change (25 => width, 26 => speed)
    ** 27    : pl_directed_link_speed
    ** 28:29 : pl_directed_link_width
    ** 30    : pl_upstream_prefer_deemph
    */
    val32 = (speed == 2) ? 0x0D000000U : 0x05000000U;
    dfPciMemWrite32(hBridge, &pRegs->pcieMisc, val32);
    /* Delay for 10 milliseconds, to allow retraining to finish. */
    dfDelayThreadFor(dfMicrosecondsToTime(10000));
    return TRUE;

  default:
    dfDebugPrint(0, ("+++ Force PCIe Gen N value of %lu(0x%lX) is not valid; ignoring.\n",
      (unsigned long)speed, (unsigned long)speed));
    return FALSE;
  }
}

static boolean_t
initHardware(
	Adb3CoreDeviceContext* pDevCtx,
	boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  unsigned int i;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelGeneric;

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Initialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Figure out if we need MSI bug workaround */
      pDevCtx->model.genericAdb3.bNeedMsiWorkaround = !pDevCtx->interrupt.bPolledMode && adb3NeedMsiWorkaround(pDevCtx);

      /* Initialize DMA engine registers */
      for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
        /* Request up to 32 x 16-byte words on PCIe side */
        dfPciMemWrite32(hBridge, &pRegs->dmaEngine[i].hostControl, 0x20U);    
        /* Allow up to 64 bytes read and 64 bytes written on MPTL */
        dfPciMemWrite32(hBridge, &pRegs->dmaEngine[i].fpgaControl, 0x80008U);
      }

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[0].localBase = 0;
      pDevCtx->info.window[0].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[1].localBase = 0;
      pDevCtx->info.window[1].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[3].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[3].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[3].localBase = 0;
      pDevCtx->info.window[3].localSize = 0;
      pDevCtx->info.window[3].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[3].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[3].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[3].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      break;

    case 1:
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    /* Nothing to do */
  }

  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  uint32_t irqStatus, dmaIrqStatus;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);
  dfIsrSpinLockPut(pInterruptObj, f);

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  if (pDevCtx->interrupt.bUsingMsi && pDevCtx->model.genericAdb3.bNeedMsiWorkaround) {
    /* MSI problem workaround - disable all interrupts then reenable */
    adb3MsiWorkaround(pDevCtx);
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(2, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
    pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, 0x1000U, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, 0x1000U, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
		if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
		}
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  unsigned int i;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Nothing to do yet */
    } else { /* Powering up (from D3) */
      /* Reinitialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Reinitialize DMA engine registers */
      for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
        /* Request up to 32 x 16-byte words on PCIe side */
        dfPciMemWrite32(hBridge, &pRegs->dmaEngine[i].hostControl, 0x20U);    
        /* Allow up to 64 bytes read and 64 bytes written on MPTL */
        dfPciMemWrite32(hBridge, &pRegs->dmaEngine[i].fpgaControl, 0x80008U);
      }
    }
  }
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

void
adb3CoreBootstrapGenericAdb3(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapGenericAdb3: entered\n"));

  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb3WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 0;
  pDevCtx->info.bootstrap.numDmaChannel = ADB3_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numSensor = 0;
  pDevCtx->info.bootstrap.numTargetFpga = 0;
  pDevCtx->info.bootstrap.numWindow = 4;
  pDevCtx->info.bootstrap.numFlashBank = 0;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = 0;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = (pDevCtx->pDevObj->hardwareId.variant.pci.revision <= 0x01U) ? TRUE : FALSE;
}
