/*
** File: main.c  
** Project: ADB3 core driver
** Purpose: OS-independent entry point into ADB3 core driver
**
** (C) Copyright Alpha Data 2009-2013
*/

#include "adb1.h"
#include "adb3.h"
#include "device.h"
#include "coreclock.h"
#include "corecombuf.h"
#include "corecoop.h"
#include "coredebugts.h"
#include "coredma.h"
#include "coredmabus.h"
#include "coreflash.h"
#include "corefpga.h"
#include "coreif.h"
#include "coremisc.h"
#include "coremodule.h"
#include "corenotify.h"
#include "coresensor.h"
#include "corevpd.h"
#include "corewindow.h"
#include "flash.h"
#include "flash_cfi.h"
#include "flash_legacy.h"
#include "hwmon.h"
#include "i2c_common.h"
#include "identify.h"
#include "interrupt.h"
#include "main.h"
#include "pci9xxx.h"
#include "resource.h"
#if defined(ADB3_DRIVER)
# include "prodtest.h"
#endif

#if !ADB3_MONOLITHIC
const char g_dfDriverName[] = "adb3core";
#endif

static Adb3CoreInterface g_coreInterface;

#if defined(ADB3_DRIVER)
/* User-mode interface for ADB3_* API functions */
static const char g_interfaceClassName[] = "adb3";
static DfInterfaceClass* g_pInterfaceClass = NULL;
static DfInterfaceMethods g_interfaceMethods;
#endif

#if defined(ADB3_DRIVER)
static DfIoStatus
onIoctl(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  unsigned int ioctlCode,
  void* pIoctl,
  unsigned int inSize,
  unsigned int outSize)
{
  DfIoStatus status = DfIoStatusSuccess;
  Adb3CoreClientContext* pClCtx = (Adb3CoreClientContext*)dfClientGetContext(pClObj);
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pClCtx->pDevCtx;

  dfDebugPrint(3, ("onIoctl(adb3): entered pDevObj=%p pClObj=%p pReq=%p ioctlCode=0x%lx pIoctl=%p inSize=0x%lx outSize=0x%lx\n",
    pDevObj, pClObj, pReq, (unsigned long)ioctlCode, pIoctl, (unsigned long)inSize, (unsigned long)outSize));

  switch (ioctlCode) {
  case ADB3_IOCTLCODE_GETPRODTESTINFO:
    status = ioctlGetProdTestInfo(pDevCtx, pIoctl, outSize);
    break;

  case ADB3_IOCTLCODE_GETUNIQUEID:
    status = ioctlGetUniqueId(pDevCtx, pIoctl, outSize);
    break;

  case ADB3_IOCTLCODE_SERVICEIRQ:
    status = ioctlServiceIrq(pDevCtx, pClCtx, pIoctl, outSize);
    break;

  case ADB3_IOCTLCODE_ABORTIPROG:
    status = ioctlAbortIprog(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADB3_IOCTLCODE_SCHEDULEIPROG:
    status = ioctlScheduleIprog(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADB3_IOCTLCODE_STATUSIPROG:
    status = ioctlStatusIprog(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADB3_IOCTLCODE_COMMANDAVR2:
    status = ioctlCommandAvr2(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADB3_IOCTLCODE_GETAVR2STATUS:
    status = ioctlGetAvr2Status(pDevCtx, pIoctl, outSize);
    break;

  default:
    status = DfIoStatusInvalidCode;
    break;
  }

  if (DfIoStatusDeferred != status) {
    dfRequestComplete(pReq, status);
  }
  return status;
}

#if DF_NEED_THUNK
static DfIoStatus
onIoctlThunk(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  unsigned int ioctlCode,
  void* pIoctl,
  unsigned int inSize,
  unsigned int outSize)
{
  DfIoStatus status = DfIoStatusSuccess;
  Adb3CoreClientContext* pClCtx = (Adb3CoreClientContext*)dfClientGetContext(pClObj);
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pClCtx->pDevCtx;

  dfDebugPrint(3, ("onIoctlThunk(adb3): entered pDevObj=%p pClObj=%p pReq=%p ioctlCode=0x%lx pIoctl=%p inSize=0x%lx outSize=0x%lx\n",
    pDevObj, pClObj, pReq, (unsigned long)ioctlCode, pIoctl, (unsigned long)inSize, (unsigned long)outSize));

  switch (ioctlCode) {
  case ADB3_IOCTLCODE_GETPRODTESTINFO:
    status = ioctlGetProdTestInfo(pDevCtx, pIoctl, outSize);
    break;

  case ADB3_IOCTLCODE_GETUNIQUEID:
    status = ioctlGetUniqueId(pDevCtx, pIoctl, outSize);
    break;

  case ADB3_IOCTLCODE_SERVICEIRQ:
    status = ioctlServiceIrq(pDevCtx, pClCtx, pIoctl, outSize);
    break;

  case ADB3_IOCTLCODE_ABORTIPROG:
    status = ioctlAbortIprog(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADB3_IOCTLCODE_SCHEDULEIPROG:
    status = ioctlScheduleIprog(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADB3_IOCTLCODE_STATUSIPROG:
    status = ioctlStatusIprog(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADB3_IOCTLCODE_COMMANDAVR2:
    status = ioctlCommandAvr2Thunk(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADB3_IOCTLCODE_GETAVR2STATUS:
    status = ioctlGetAvr2Status(pDevCtx, pIoctl, outSize);
    break;

  default:
    status = DfIoStatusInvalidCode;
    break;
  }

  if (DfIoStatusDeferred != status) {
    dfRequestComplete(pReq, status);
  }
  return status;
}
#endif

static DfIoStatus
onOpen(
	DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  DfInterface* pInterface,
  boolean_t bPrivileged)
{
  Adb3CoreDeviceContext* pDevCtx;
  Adb3CoreClientContext* pClCtx = NULL;
  DfIoStatus status = DfIoStatusSuccess;

  pDevCtx = (Adb3CoreDeviceContext*)dfInterfaceGetContext(pInterface);

  pClCtx = (Adb3CoreClientContext*)dfMalloc(sizeof(Adb3CoreClientContext));
  if (NULL == pClCtx) {
    status = DfIoStatusNoMemory;
    goto done;
  }
  
  pClCtx->pDevCtx = pDevCtx;
  pClCtx->bPrivileged = bPrivileged;

done:
  if (status != DfIoStatusDeferred) {
    if (DfIoStatusIsError(status)) {
      if (NULL != pClCtx) {
        dfFree(pClCtx);
      }
    } else {
      dfClientSetContext(pClObj, pClCtx);
    }
    dfRequestComplete(pReq, status);
  }
  return status;
}

static void
onClose(
	DfDeviceObject* pDevObj,
  DfClientObject* pClObj)
{
  Adb3CoreClientContext* pClCtx = (Adb3CoreClientContext*)dfClientGetContext(pClObj);

  /* Free the memory associated with the client's context */
  if (NULL != pClCtx) {
    dfFree(pClCtx);
    dfClientSetContext(pClObj, NULL);
  }
}
#endif

static void
coreTestDebugMsg(
  void* pDevice,
  int level,
  const char* pMessage)
{
  dfDebugPrint(level, ("%s", pMessage));
}

static void
dumpRawResources(
  DfBusType busType,
  DfBusResources* pResources)
{
  unsigned int i, n;
  DfPciIrqLine* pIrq;
  DfPciBar* pBar;

  switch (busType) {
  case DfBusPci:
    n = pResources->variant.pci.numMemBar;
    for (i = 0; i < n; i++) {
      pBar = &pResources->variant.pci.memBar[i];
      dfDebugPrint(0, ("Memory BAR%u base=0x%08lx_%08lx size=0x%08lx_%08lx flags=0x%x\n",
        i, dfSplitUint64(pBar->base), dfSplitUint64(pBar->size), (unsigned int)pBar->flags));
    }
    n = pResources->variant.pci.numIoBar;
    for (i = 0; i < n; i++) {
      pBar = &pResources->variant.pci.ioBar[i];
      dfDebugPrint(0, ("IO     BAR%u base=0x%08lx_%08lx size=0x%08lx_%08lx flags=0x%x\n",
        i, dfSplitUint64(pBar->base), dfSplitUint64(pBar->size), (unsigned int)pBar->flags));
    }
    n = pResources->variant.pci.numIrq;
    for (i = 0; i < n; i++) {
      pIrq = &pResources->variant.pci.irq[i];
      dfDebugPrint(0, ("Interrupt %u bMsgSignalled=%s bLevel=%s line=0x%x pin=0x%x\n",
        i, pIrq->bMessageSignalled ? "TRUE" : "FALSE", pIrq->bLevelSensitive ? "TRUE" : "FALSE", (unsigned int)pIrq->variant.raw.intLine, (unsigned int)pIrq->variant.raw.intPin));
    }
    break;

  default:
    dfDebugPrint(0, ("(unknown bus type)\n"));
    break;
  }
}

static void
freeDeviceContext(
  Adb3CoreDeviceContext* pDevCtx)
{
#if defined(ADB3_DRIVER)
  if (pDevCtx->userMode.bRegistered) {
    if (!dfInterfaceUnregister(pDevCtx->pDevObj, &pDevCtx->userMode.iface)) {
      dfDebugPrint(0, ("*** freeDeviceContext: failed to unregister user-mode interface\n"));
    } else {
      dfDebugPrint(1, ("freeDeviceContext: unregistered user-mode interface\n"));
      pDevCtx->userMode.bRegistered = FALSE;
    }
  }
#endif

  if (pDevCtx->bCoreInterfaceRegd) {
    dfUnexportInterface(pDevCtx->pDevObj, g_coreInterfaceName);
    pDevCtx->bCoreInterfaceRegd = FALSE;
    dfDebugPrint(1, ("freeDeviceContext: unregistered core interface\n"));
  }

  dfTimerUninit(&pDevCtx->hwMon.polling.timer);
}

static void
initResourceQueue(
  ResourceQueue* pQ)
{
  pQ->pHead = pQ->pTail = NULL;
  pQ->length = 0;
  pQ->users = 0;
}

static void
enableInterrupts(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bEnable)
{
  pDevCtx->methods.pEnableInterrupts(pDevCtx, bEnable);
  pDevCtx->interrupt.bEnabled = bEnable;
}

static void
stopDevice(
  DfDeviceObject* pDevObj,
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i, j;
  DfSpinLockFlags f;

#if defined(ADB3_DRIVER)
  if (pDevCtx->userMode.bEnabled) {
    if (!dfInterfaceEnable(pDevCtx->pDevObj, &pDevCtx->userMode.iface, FALSE)) {
      dfDebugPrint(0, ("*** stopDevice: failed to disable user-mode interface\n"));
    } else {
      dfDebugPrint(1, ("stopDevice: disabled user-mode interface\n"));
      pDevCtx->userMode.bEnabled = FALSE;
    }
  }
#endif

  /* Deinitialize hardware (phase 1, with interrupts enabled) */
  if (pDevCtx->hardware.bInitializedPhase1) {
    if (!pDevCtx->methods.pInitHardware(pDevCtx, FALSE, 1)) {
      dfDebugPrint(0, ("*** stopDevice: failed to uninitialize hardware (phase 1)\n"));
    } else {
      dfDebugPrint(1, ("stopDevice: uninitialized hardware (phase 1)\n"));
    }
    pDevCtx->hardware.bInitializedPhase1 = FALSE;
  }

  /* Sync caches with Flash devices, if necessary */
  for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
    if (pDevCtx->flashControl[i].cache.state == FlashCacheDirty) {
      if (pDevCtx->hardware.bCanTouch) {
        flashSync(pDevCtx, i);
        dfDebugPrint(1, ("stopDevice: sync'ed cache for Flash bank %u\n", i));
      } else {
        dfDebugPrint(0, ("stopDevice: not sync'ing cache for Flash bank %u because can't touch hardware\n", i));
      }
    }
    pDevCtx->flashControl[i].cache.state = FlashCacheInvalid;
  }

  /* Free I/O module VPD buffers */
  for (i = 0; i < MAX_NUM_IO_MODULE; i++) {
    if (NULL != pDevCtx->info.ioModule[i].pVpdBuffer) {
      dfFree(pDevCtx->info.ioModule[i].pVpdBuffer);
      pDevCtx->info.ioModule[i].pVpdBuffer = NULL;
      dfDebugPrint(1, ("stopDevice: freed I/O module buffer %u\n", i));
    }
  }

  /* Stop hardware monitor polling timer, if running */
  dfEventInit(&pDevCtx->hwMon.polling.shutdownEvent);
  f = dfSpinLockGet(&pDevCtx->hwMon.lock);
  pDevCtx->hwMon.polling.bShutdown = TRUE;
  if (pDevCtx->hwMon.polling.bTimerRunning) {
    dfSpinLockPut(&pDevCtx->hwMon.lock, f);
    dfDebugPrint(2, ("stopDevice: stoppping hardware monitor polling timer\n", i));
    if (!dfTimerCancel(&pDevCtx->hwMon.polling.timer)) {
      dfEventWait(&pDevCtx->hwMon.polling.shutdownEvent);
    }
    dfDebugPrint(1, ("stopDevice: stopped hardware monitor polling timer\n", i));
  } else {
    dfSpinLockPut(&pDevCtx->hwMon.lock, f);
    dfDebugPrint(2, ("stopDevice: hardware monitor polling timer not running\n", i));
  }
  dfEventUninit(&pDevCtx->hwMon.polling.shutdownEvent);

  if (pDevCtx->interrupt.bEnabled) {
    if (pDevCtx->hardware.bCanTouch) {
      /* Disable interrupts in hardware */
      dfDebugPrint(2, ("stopDevice: disabling interrupts\n"));
      enableInterrupts(pDevCtx, FALSE);
      dfDebugPrint(1, ("stopDevice: disabled interrupts\n"));
    } else {
      dfDebugPrint(1, ("stopDevice: not disabling interrupts because can't touch hardware\n"));
      pDevCtx->interrupt.bEnabled = FALSE;
    }

    /* Ensure no DPCs are scheduled */
    for (i = 0; i < MAX_NUM_TARGET_FPGA; i++) {
      if (pDevCtx->interrupt.fpga[i].bDpcInitialized) {
        dfDebugPrint(2, ("stopDevice: uninitializing target FPGA %u interrupt DPC\n", i));
        dfDpcUninit(&pDevCtx->interrupt.fpga[i].dpc);
        pDevCtx->interrupt.fpga[i].bDpcInitialized = FALSE;
      }
      if (pDevCtx->interrupt.hwMon[i].bDpcInitialized) {
        dfDebugPrint(2, ("stopDevice: uninitializing target FPGA %u hardware monitor interrupt DPC\n", i));
        dfDpcUninit(&pDevCtx->interrupt.hwMon[i].dpc);
        pDevCtx->interrupt.hwMon[i].bDpcInitialized = FALSE;
      }
    }
    for (i = 0; i < MAX_NUM_DMA_ENGINE; i++) {
      if (pDevCtx->interrupt.dma[i].bDpcInitialized) {
        dfDebugPrint(2, ("stopDevice: uninitializing DMA engine %u DPC\n", i));
        dfDpcUninit(&pDevCtx->interrupt.dma[i].dpc);
        pDevCtx->interrupt.dma[i].bDpcInitialized = FALSE;
#ifdef ADB3_DBG_IRQ_COUNT
        if (pDevCtx->interrupt.dma[i].count != pDevCtx->interrupt.dma[i].expected) {
          dfDebugPrint(0, ("*** stopDevice: DMA engine interrupt count=%lu expected=%lu\n",
            pDevCtx->interrupt.dma[i].count, pDevCtx->interrupt.dma[i].expected));
        }
#endif
      }
    }
    dfDebugPrint(1, ("stopDevice: uninitialized DPCs\n"));
  }

  /* Free primary common buffers */
  if (NULL != pDevCtx->comBuf.pComBufs) {
    unsigned int i, n = pDevCtx->comBuf.count;
    DfDmaBuffer* pDmaBuf;

    for (i = 0; i < n; i++) {
      pDmaBuf = pDevCtx->comBuf.pComBufs[i];
      dfAssert(NULL != pDmaBuf);
      if (NULL != pDmaBuf) {
        dfDmaBufferFree(pDevCtx->comBuf.pComBufs[i]);
        pDevCtx->comBuf.pComBufs[i] = NULL;
        pDevCtx->comBuf.count--;
        dfDebugPrint(2, ("stopDevice: freed primary common buffer %u\n", i));
      }
    }
    dfAssert(0 == pDevCtx->comBuf.count);
    dfFree(pDevCtx->comBuf.pComBufs);
    pDevCtx->comBuf.pComBufs = NULL;
    dfDebugPrint(1, ("stopDevice: freed primary common buffers\n"));
  }

  /* Free DMABus scatter-gather table buffers */
  for (i = 0; i < MAX_NUM_DMA_ENGINE; i++) {
    for (j = 0; j < 2; j++) {
      if (NULL != pDevCtx->dma.channel[i].dmaBus[j].pTable) {
        dfFree(pDevCtx->dma.channel[i].dmaBus[j].pTable);
        pDevCtx->dma.channel[i].dmaBus[j].pTable = NULL;
      }
    }
  }

  /* Free common buffers for DMA descriptors */
  for (i = 0; i < MAX_NUM_DMA_ENGINE; i++) {
    for (j = 0; j < 2; j++) {
      if (NULL != pDevCtx->dma.channel[i].descriptor[j].pDmaBuffer) {
        dfDmaBufferFree(pDevCtx->dma.channel[i].descriptor[j].pDmaBuffer);
        pDevCtx->dma.channel[i].descriptor[j].pDmaBuffer = NULL;
      }
    }
  }

  /* Free DMA mapper */
  if (pDevCtx->bDmaMapperAllocated) {
    dfDmaMapperFree(pDevObj);
    pDevCtx->bDmaMapperAllocated = FALSE;
    dfDebugPrint(1, ("stopDevice: freed DMA mapper\n"));
  }

  /* Detach interrupt handler */
  if (pDevCtx->interrupt.bConnected) {
    dfInterruptDisconnect(pDevCtx->pDevObj, &pDevCtx->interrupt.object);
    pDevCtx->interrupt.bConnected = FALSE;
    dfDebugPrint(1, ("stopDevice: disconnected interrupt handler\n"));
  }

#if defined(BUILD_DEBUGTS)
  /* Free debug timestamp buffers, if they were allocated. */
  for (i = 0; i < MAX_NUM_DMA_ENGINE; i++) {
    if (dfTimestampBufferEnabled(&pDevCtx->dma.channel[i].timestampBuffer)) {
      dfTimestampBufferUninit(&pDevCtx->dma.channel[i].timestampBuffer);
    }
  }
#endif

  for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
    Adb3CoreFlashInfo* pFlashInfo = &pDevCtx->info.flash[i];
    Adb3CoreFlashControl* pFlashControl = &pDevCtx->flashControl[i];

    /* Free Flash information structures */
    if (NULL != pFlashInfo->detail.pLegacy) {
      dfAssert(pFlashInfo->bPresent);
      flashLegacyCleanup(pFlashInfo->detail.pLegacy);
      pFlashInfo->detail.pLegacy = NULL;
    }
    if (NULL != pFlashInfo->detail.pCfi) {
      flashCfiCleanup(pFlashInfo->detail.pCfi);
      pFlashInfo->detail.pCfi = NULL;
    }
    /* Free Flash cache buffers */
    if (NULL != pFlashControl->cache.pBuffer) {
      dfFree(pFlashControl->cache.pBuffer);
      pFlashControl->cache.pBuffer = NULL;
    }
  }

  /* Deinitialize hardware (phase 0, with interrupts disabled) */
  if (pDevCtx->hardware.bInitializedPhase0) {
    if (!pDevCtx->methods.pInitHardware(pDevCtx, FALSE, 0)) {
      dfDebugPrint(0, ("*** stopDevice: failed to uninitialize hardware (phase 0)\n"));
    } else {
      dfDebugPrint(1, ("stopDevice: uninitialized hardware (phase 0)\n"));
    }
    pDevCtx->hardware.bInitializedPhase0 = FALSE;
  }

  /* Free VPD read buffer */
  if (NULL != pDevCtx->info.vpd.pData) {
    dfFree(pDevCtx->info.vpd.pData);
    pDevCtx->info.vpd.pData = NULL;
  }

  /* Unmap device from kernel address space */
  if (pDevCtx->hardware.bMapped) {
    if (!pDevCtx->methods.pMapBusResources(pDevCtx, NULL, NULL, TRUE)) {
      dfDebugPrint(0, ("*** stopDevice: failed to unmap bus resources\n"));
    } else {
      pDevCtx->hardware.bMapped = FALSE;
      dfDebugPrint(1, ("stopDevice: unmapped bus resources\n"));
    }
  }
}

boolean_t
adb3CoreOnStart(
  DfDeviceObject* pDevObj,
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRawRes,
  DfBusResources* pTranslatedRes)
{
  const unsigned int maxPriComBufCount = 65536U; /* Hard limit on number of primary common buffers */
  uint32_t dmaMapLength, priComBufCount, priComBufAlign, k, tmp32;
  uint64_t priComBufSize;
  unsigned int i, j;
  DfDmaBuffer** pComBufs = NULL;

  dfDebugPrint(1, ("adb3CoreOnStart: entered, pDevObj=%p pDevCtx=%p\n", pDevObj, pDevCtx));

  dfCopyKK(&pDevCtx->rawResources, pRawRes, sizeof(DfBusResources));
  dfCopyKK(&pDevCtx->translatedResources, pTranslatedRes, sizeof(DfBusResources));

  if (dfDebugLevel(2)) {
    dfDebugPrint(0, ("adb3CoreOnStart: raw resources:\n"));
    dumpRawResources(pDevObj->hardwareId.busType, &pDevCtx->rawResources);
  }

  /* Check that device's base address registers are OK */
  if (!pDevCtx->methods.pValidateBusResources(pDevCtx, &pDevCtx->rawResources, &pDevCtx->translatedResources)) {
    dfDebugPrint(0, ("*** adb3CoreOnStart: failed to validate bus resources\n"));
    goto failed;
  }
  dfDebugPrint(1, ("adb3CoreOnStart: validated bus resources\n"));

  /* Map device into kernel address space */
  if (!pDevCtx->methods.pMapBusResources(pDevCtx, &pDevCtx->rawResources, &pDevCtx->translatedResources, FALSE)) {
    dfDebugPrint(0, ("*** adb3CoreOnStart: failed to map bus resources\n"));
    goto failed;
	}
  pDevCtx->hardware.bMapped = TRUE;
  dfDebugPrint(1, ("adb3CoreOnStart: mapped bus resources\n"));

  /* Allocate VPD read buffer */
  if (pDevCtx->info.bootstrap.vpdBufferLength) {
    pDevCtx->info.vpd.pData = (Adb3CoreVpd*)dfMalloc(pDevCtx->info.bootstrap.vpdBufferLength);
    if (NULL == pDevCtx->info.vpd.pData) {
      dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate VPD read buffer\n"));
      goto failed;
    }
    dfDebugPrint(1, ("adb3CoreOnStart: allocated VPD read buffer, size=0x%08x\n", pDevCtx->info.bootstrap.vpdBufferLength));
  }

  /* Initialize hardware (phase 0, with interrupts disabled) */
  for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
    pDevCtx->info.flash[i].bPresent = FALSE;
  }
  if (!pDevCtx->methods.pInitHardware(pDevCtx, TRUE, 0)) {
    dfDebugPrint(0, ("*** adb3CoreOnStart: failed to initialize hardware\n"));
    goto failed;
  }
  pDevCtx->hardware.bInitializedPhase0 = TRUE;
  dfDebugPrint(1, ("adb3CoreOnStart: initialized hardware (phase 0)\n"));

  for (i = 0; i < pDevCtx->info.bootstrap.numFlashBank; i++) {
    Adb3CoreFlashInfo* pFlashInfo = &pDevCtx->info.flash[i];
    Adb3CoreFlashControl* pFlashControl = &pDevCtx->flashControl[i];

    if (pFlashInfo->bPresent) {
      if (pFlashInfo->bLegacyFlash) {
        LegacyFlashInformation* pDetail = pFlashInfo->detail.pLegacy;

        dfAssert(NULL != pDetail);
        pFlashControl->cache.pBuffer = (uint8_t*)dfMalloc(pDetail->largestBlockSize);
        if (NULL == pFlashControl->cache.pBuffer) {
          dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate Flash bank %u cache buffer\n", i));
          goto failed;
        }
        dfDebugPrint(1, ("adb3CoreOnStart: allocated Flash bank %u cache buffer, size=0x%08lx\n",
          i, (unsigned long)pDetail->largestBlockSize));
      } else {
        CfiInformation* pDetail = pFlashInfo->detail.pCfi;

        dfAssert(NULL != pDetail);
        if (pDetail->validation.largestBlock > (size_t)-1) {
          dfDebugPrint(0, ("*** adb3CoreOnStart: required Flash bank %u cache buffer too large, size=0x%08lx_%08lx\n",
            i, dfSplitUint64(pDetail->validation.largestBlock)));
          goto failed;
        }
        pFlashControl->cache.pBuffer = (uint8_t*)dfMalloc((size_t)pDetail->validation.largestBlock); /* Cast safe due to previous check */
        if (NULL == pFlashControl->cache.pBuffer) {
          dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate Flash bank %u cache buffer\n", i));
          goto failed;
        }
        dfDebugPrint(1, ("adb3CoreOnStart: allocated Flash bank %u cache buffer, size=0x%08lx_%08lx\n",
          i, dfSplitUint64(pDetail->validation.largestBlock)));
      }
    }
  }

  if (pDevCtx->info.vpd.bValid && pDevCtx->info.bootstrap.vpdBufferLength) {
    size_t n;

    dfDebugPrint(2, ("adb3CoreOnStart: dump of raw VPD information (max 0x100 bytes):\n"));
    n = pDevCtx->info.bootstrap.vpdBufferLength;
    if (n > 0x100U) {
      n = 0x100U;
    }
    dfDebugDumpMemory(2, 4, 16, 1, FALSE, pDevCtx->info.vpd.pData, n);
  }

  for (i = 0; i < pDevCtx->info.bootstrap.numIoModule; i++) {
    CoreIoModuleInfo* pInfo = &pDevCtx->info.ioModule[i];
    size_t n;

    if (pInfo->bPresent && NULL != pInfo->pVpdBuffer) {
      n = pInfo->vpdLength;
      if (n > 0x100U) {
        n = 0x100U;
      }
      dfDebugPrint(2, ("adb3CoreOnStart: dump of FRU ROM %u information (max 0x100 bytes):\n", i));
      dfDebugDumpMemory(2, 4, 16, 1, FALSE, pInfo->pVpdBuffer, n);
    }
  }

  /* Initialize DPCs that can be scheduled by interrupt handler */
  for (i = 0; i < MAX_NUM_TARGET_FPGA; i++) {
    dfDpcInit(&pDevCtx->interrupt.fpga[i].dpc, &pDevCtx->interrupt.object, fpgaInterruptDpc, pDevCtx);
    pDevCtx->interrupt.fpga[i].bDpcInitialized = TRUE;
    dfDpcInit(&pDevCtx->interrupt.hwMon[i].dpc, &pDevCtx->interrupt.object, hwMonAlertDpc, pDevCtx);
    pDevCtx->interrupt.hwMon[i].bDpcInitialized = TRUE;
  }
  for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
    pDevCtx->interrupt.dma[i].bDpcInitialized = TRUE;
    dfDpcInit(&pDevCtx->interrupt.dma[i].dpc, &pDevCtx->interrupt.object, dmaCompletedRoutine, (void*)pDevCtx);
#ifdef ADB3_DBG_IRQ_COUNT
    pDevCtx->interrupt.dma[i].count = 0;
    pDevCtx->interrupt.dma[i].expected = 0;
#endif
  }
  dfDebugPrint(1, ("adb3CoreOnStart: initialized DPCs\n"));

  /* Attach interrupt handler */
  if (NULL != pDevCtx->methods.pIsr) {
    switch (pDevCtx->pDevObj->hardwareId.busType) {
    case DfBusPci:
      if (pDevCtx->translatedResources.variant.pci.numIrq == 1) {
        if (!dfInterruptConnect(pDevCtx->pDevObj, &pDevCtx->interrupt.object, &pDevCtx->translatedResources, DfBusPci, 0, pDevCtx->methods.pIsr, pDevCtx)) {
          dfDebugPrint(0, ("*** adb3CoreOnStart: failed to connect interrupt handler\n"));
          goto failed;
        }
        pDevCtx->interrupt.bConnected = TRUE;
        pDevCtx->interrupt.bUsingMsi = dfInterruptUsingMsi(pDevCtx->pDevObj, &pDevCtx->interrupt.object);
        dfDebugPrint(1, ("adb3CoreOnStart: connected interrupt handler, bUsingMsi=%s\n", pDevCtx->interrupt.bUsingMsi ? "TRUE" : "FALSE"));
      } else {
        dfDebugPrint(1, ("adb3CoreOnStart: not connecting interrupt handler due to wrong number of interrupts (%lu)\n",
          (unsigned long)pDevCtx->translatedResources.variant.pci.numIrq));
      }
      break;

    default:
      dfDebugPrint(1, ("adb3CoreOnStart: not connecting interrupt handler due to unknown bus type (%lu)\n",
        (unsigned long)pDevCtx->pDevObj->hardwareId.busType));
      break;
    }
  } else {
    dfDebugPrint(1, ("adb3CoreOnStart: not connecting interrupt handler due to no ISR method\n"));
  }

  if (pDevCtx->info.bootstrap.numDmaChannel > 0) {
    /* Allocate DMA mapper */
    if (DfParameterSuccess != dfParameterGetUint32(pDevObj->pDrvObj, "DmaMapLength", &dmaMapLength)) {
      dmaMapLength = DEFAULT_DMA_MAP_LENGTH;
    }
    if (dmaMapLength < 1U) {
      dmaMapLength = 1U;
    }
    if (dmaMapLength > (g_dfDmaBufferMaxLength / pDevCtx->info.bootstrap.dmaDescriptorSize)) {
      dmaMapLength = (g_dfDmaBufferMaxLength / pDevCtx->info.bootstrap.dmaDescriptorSize);
    }
    if (!dfDmaMapperAllocate(
      pDevCtx->pDevObj,
      pDevCtx->info.bootstrap.numDmaChannel * 2,
      (unsigned int)dmaMapLength,
      pDevCtx->info.bootstrap.b64BitDmaAddress,
      pDevCtx->info.bootstrap.dmaTransferMaxSize,
      &pDevCtx->dma.mapLength))
    {
      dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate DMA mapper\n"));
      goto failed;
    }
    pDevCtx->bDmaMapperAllocated = TRUE;
    dfDebugPrint(1, ("adb3CoreOnStart: allocated DMA mapper, dmaMapLength=%lu\n", (unsigned long)pDevCtx->dma.mapLength));

    /* Allocate common buffers for DMA descriptors, 2 per DMA engine */
    for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
      for (j = 0; j < 2; j++) {
        /* Request 'dmaDescriptorSize - 1' extra bytes to enable alignment of the returned buffer */
        unsigned int size = pDevCtx->info.bootstrap.dmaDescriptorSize * pDevCtx->dma.mapLength;
        DfDmaBuffer* pDmaBuffer;

        pDmaBuffer = dfDmaBufferAllocate(pDevObj, size, pDevCtx->info.bootstrap.dmaDescriptorSize);
        if (NULL == pDmaBuffer) {
          dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate DMA descriptor buffer %lu.%lu\n", (unsigned long)i, (unsigned long)j));
          goto failed;
        }
        pDevCtx->dma.channel[i].descriptor[j].pDmaBuffer = pDmaBuffer;
        dfDebugPrint(1, ("adb3CoreOnStart: DMA descriptor buffer %lu.%lu pKernAddr=%p busAddr=0x%08lx_%08lx size=%lu(0x%lx)\n",
          (unsigned long)i, (unsigned long)j, (void*)pDmaBuffer->pKernelAddress,
          dfSplitUint64(pDmaBuffer->busAddress), (unsigned long)size, (unsigned long)size));
        pDevCtx->dma.channel[i].descriptor[j].buffer.pGeneric = pDmaBuffer->pKernelAddress;
        pDevCtx->dma.channel[i].descriptor[j].length = size;
        pDevCtx->dma.channel[i].descriptor[j].busAddress = pDmaBuffer->busAddress;
      }
    }

    /* Allocate buffers for holding DMABus scatter-gather tables, 2 per DMA engine */
    for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
      for (j = 0; j < 2; j++) {
        pDevCtx->dma.channel[i].dmaBus[j].pTable = (DfDmaMapperNode*)dfMalloc(sizeof(DfDmaMapperNode) * pDevCtx->dma.mapLength);
        if (NULL == pDevCtx->dma.channel[i].dmaBus[j].pTable) {
          dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate DMABus scatter-gather table buffer %lu.%lu\n", (unsigned long)i, (unsigned long)j));
          goto failed;
        }
      }
    }

#if defined(BUILD_DEBUGTS)
    {
      uint32_t channelMask;
      uint32_t nEntry;

      /* Allocate debug timestamp buffers, 1 per DMA engine, if requested via driver parameters. */
      channelMask = 0U;
      dfParameterGetUint32(pDevObj->pDrvObj, "DebugTSEnableDMA", &channelMask);
      nEntry = 0;
      dfParameterGetUint32(pDevObj->pDrvObj, "DebugTSBufferSizeDMA", &nEntry);
      if (nEntry) {
        if (nEntry > 0xFFFFFFU) {
          /* Don't allow buffer to exceed 256 MiB */
          nEntry = 0xFFFFFFU;
        }
        if (nEntry < 2) {
          nEntry = 2;
        }
        /* Allocate one more entry than requested because we use the difference between tail - head
        ** (modulo number of entries) to determine the number of entries currently in the buffer. */
        nEntry++;
        for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
          if (!(channelMask & (0x1U << i))) {
            continue;
          }
          if (!dfTimestampBufferInit(&pDevCtx->dma.channel[i].timestampBuffer, nEntry)) {
            dfDebugPrint(0, ("*** adb3CoreOnStart: failed to initialize debug timestamp buffer for DMA engine %u of size 0x%lX(%lu) entries\n",
              i, (unsigned long)nEntry, (unsigned long)nEntry));
            continue;
          }
        }
      }
    }
#endif
  }

  /* Allocate primary common buffers */
  if (DfParameterSuccess != dfParameterGetUint32(pDevObj->pDrvObj, "PrimaryCommonBufferCount", &priComBufCount)) {
    priComBufCount = 0;
  }
  if (priComBufCount > (unsigned int)-1) {
    dfDebugPrint(0, ("*** adb3CoreOnStart: invalid primary common buffer count %u(0x%x)\n",
      (unsigned long)priComBufCount, (unsigned long)priComBufCount));
    priComBufCount = 0;
  }
  if (priComBufCount > maxPriComBufCount) {
    dfDebugPrint(0, ("+++ adb3CoreOnStart: limiting primary common buffer count to %u(0x%x)\n",
      (unsigned long)maxPriComBufCount, (unsigned long)maxPriComBufCount));
    priComBufCount = maxPriComBufCount;
  }
  if (DfParameterSuccess != dfParameterGetUint32(pDevObj->pDrvObj, "PrimaryCommonBufferAlignment", &priComBufAlign)) {
    priComBufAlign = 1;
  }
  if (DfParameterSuccess != dfParameterGetUint32(pDevObj->pDrvObj, "PrimaryCommonBufferSizeLow", &tmp32)) {
    tmp32 = 0;
  }
  priComBufSize = (uint64_t)tmp32;
  if (DfParameterSuccess != dfParameterGetUint32(pDevObj->pDrvObj, "PrimaryCommonBufferSizeHigh", &tmp32)) {
    tmp32 = 0;
  }
  priComBufSize |= (uint64_t)tmp32 << 32;
  if (priComBufSize > (size_t)-1) {
    dfDebugPrint(0, ("*** adb3CoreOnStart: invalid primary common buffer size " DF_FMT_U64 "(0x%08lx_%08lx)\n",
      (unsigned long long)priComBufSize, dfSplitUint64(priComBufSize)));
    priComBufSize = 0;
  }
  if (priComBufSize != 0 && priComBufCount != 0) {
    dfDebugPrint(1, ("adb3CoreOnStart: allocating %lu primary common buffers of size " DF_FMT_U64 "(0x%lx_%08lx) bytes, align %lu(0x%lx)\n",
      (unsigned long)priComBufCount, (unsigned long long)priComBufSize, dfSplitUint64(priComBufSize), (unsigned long)priComBufAlign, (unsigned long)priComBufAlign));
    pComBufs = dfMalloc(sizeof(DfDmaBuffer*) * priComBufCount);
    if (NULL == pComBufs) {
      dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate descriptors for %lu primary common buffer(s)\n", (unsigned long)priComBufCount));
    }
    for (k = 0; k < priComBufCount; k++) {
      pComBufs[k] = NULL;
    }
    pDevCtx->comBuf.pComBufs = pComBufs;
    for (k = 0; k < priComBufCount; k++) {
      pComBufs[k] = dfDmaBufferAllocate(pDevObj, (size_t)priComBufSize /* cast safe due to above range check */, priComBufAlign);
      if (NULL == pComBufs[k]) {
        dfDebugPrint(0, ("*** adb3CoreOnStart: failed to allocate primary common buffer %lu, skipping remaining\n", (unsigned long)k));
        break;
      }
      dfDebugPrint(2, ("adb3CoreOnStart: allocated primary common buffer %u, pKernAddr=%p busAddr=0x%08lx_%08lx size=" DF_FMT_U64 "(0x%lx_%08lx)\n",
        (unsigned long)k, pComBufs[k]->pKernelAddress, dfSplitUint64(pComBufs[k]->busAddress), dfSplitUint64(priComBufSize)));
      pDevCtx->comBuf.count++;
    }
  }

  /* Enable interrupt in hardware */
  if (pDevCtx->interrupt.bConnected) {
    enableInterrupts(pDevCtx, TRUE);
    dfDebugPrint(1, ("adb3CoreOnStart: enabled interrupts\n"));
  }

  /* Initialize hardware (phase 1, with interrupts enabled) */
  if (!pDevCtx->methods.pInitHardware(pDevCtx, TRUE, 1)) {
    dfDebugPrint(0, ("*** adb3CoreOnStart: failed to initialize hardware (phase 1)\n"));
    goto failed;
  }
  pDevCtx->hardware.bInitializedPhase1 = TRUE;
  dfDebugPrint(1, ("adb3CoreOnStart: initialized hardware (phase 1)\n"));

#if defined(ADB3_DRIVER)
  if (!dfInterfaceEnable(pDevObj, &pDevCtx->userMode.iface, TRUE)) {
    dfDebugPrint(0, ("*** adb3CoreOnStart: failed to enable user-mode interface\n"));
    goto failed;
  } else {
    dfDebugPrint(1, ("adb3CoreOnStart: enabled user-mode interface\n"));
    pDevCtx->userMode.bEnabled = TRUE;
  }
#endif

  return TRUE;

failed:
  stopDevice(pDevObj, pDevCtx);
  return FALSE;
}

void
adb3CoreOnStop(
  DfDeviceObject* pDevObj,
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bCanTouchHardware)
{
  dfDebugPrint(1, ("adb3CoreOnStop: entered, pDevObj=%p pDevCtx=%p bCanTouchHardware=%s\n",
    pDevObj, pDevCtx, bCanTouchHardware ? "TRUE" : "FALSE"));

  pDevCtx->hardware.bCanTouch = bCanTouchHardware;
  stopDevice(pDevObj, pDevCtx);
}

boolean_t
adb3CoreOnAdd(
  DfDeviceObject* pDevObj,
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i, j;
  boolean_t bHwMatch = FALSE;

  dfDebugPrint(1, ("adb3CoreOnAdd: entered, pDevObj=%p pDevCtx=%p\n", pDevObj, pDevCtx));
  if (pDevObj->hardwareId.busType == DfBusPci) {
    dfDebugPrint(1, ("adb3CoreOnAdd: vid=0x%04x did=0x%04x subvid=0x%04x subdid=0x%04x rev=0x%02x\n",
      pDevObj->hardwareId.variant.pci.vendor,
      pDevObj->hardwareId.variant.pci.device,
      pDevObj->hardwareId.variant.pci.subVendor,
      pDevObj->hardwareId.variant.pci.subDevice,
      pDevObj->hardwareId.variant.pci.revision));
  }

  /* Initialize the device context */
  DF_ZERO_OBJECT(pDevCtx);
  pDevCtx->pDevObj = pDevObj;
  pDevCtx->bCoreInterfaceRegd = FALSE;
  pDevCtx->bDmaMapperAllocated = FALSE;

  dfSpinLockInit(&pDevCtx->cooperation.lock);
  pDevCtx->cooperation.pExclusiveClient = NULL;
  dfListInit(&pDevCtx->cooperation.clients);

#if defined(ADB3_DRIVER)
  /* Initialize user mode interface stuff */
  pDevCtx->userMode.bRegistered = FALSE;
  pDevCtx->userMode.bEnabled = FALSE;
#endif

  /* Initialize hardware stuff */
  pDevCtx->hardware.bridge.pGeneric = NULL;
  pDevCtx->hardware.model.pGeneric = NULL;
  pDevCtx->hardware.pBridged = NULL;
  pDevCtx->hardware.pBridgedPrefetch = NULL;
  pDevCtx->hardware.bInitializedPhase0 = FALSE;
  pDevCtx->hardware.bInitializedPhase1 = FALSE;
  pDevCtx->hardware.bMapped = FALSE;
  pDevCtx->hardware.bCanTouch = TRUE;

  /* Initialize the methods in the device context */
  pDevCtx->methods.pModelBootstrap = NULL;

  /* Initialize resource queues */
  dfSpinLockInit(&pDevCtx->resource.lock);
  initResourceQueue(&pDevCtx->resource.clockGen.queue);
  for (i = 0; i < MAX_NUM_CLOCK_GENERATOR; i++) {
    pDevCtx->resource.clockGen.users[i] = 0;
  }
  initResourceQueue(&pDevCtx->resource.dmaEngine.queue);
  for (i = 0; i < MAX_NUM_DMA_ENGINE; i++) {
    pDevCtx->resource.dmaEngine.users[i] = 0;
  }
  initResourceQueue(&pDevCtx->resource.eeprom.queue);
  pDevCtx->resource.eeprom.users[0] = 0;
  initResourceQueue(&pDevCtx->resource.flash.queue);
  pDevCtx->resource.flash.users[0] = 0;
  initResourceQueue(&pDevCtx->resource.targetFpga.queue);
  for (i = 0; i < MAX_NUM_TARGET_FPGA; i++) {
    pDevCtx->resource.targetFpga.users[i] = 0;
  }
  initResourceQueue(&pDevCtx->resource.sysMon.queue);
  pDevCtx->resource.sysMon.users[0] = 0;

  /* Initialize Flash stuff */
  for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
    pDevCtx->info.flash[i].detail.pGeneric = NULL;
    pDevCtx->info.flash[i].bPresent = FALSE;
    pDevCtx->info.flash[i].targetArea.start = 0;
    pDevCtx->info.flash[i].targetArea.length = 0;
    for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
      pDevCtx->flashControl[i].cache.pBuffer = NULL;
      pDevCtx->flashControl[i].cache.state = FlashCacheInvalid;
    }
  }

  /* Initialize VPD stuff */
  pDevCtx->info.vpd.pData = NULL;
  pDevCtx->info.vpd.bValid = FALSE;
  pDevCtx->info.vpd.bChecksumError = FALSE;

  /* Initialize clock stuff */
  for (i = 0; i < MAX_NUM_CLOCK_GENERATOR; i++) {
    ClockGeneratorContext* pDevClockCtx = &pDevCtx->clockGenerator[i];

    pDevClockCtx->index = i;
    pDevClockCtx->currentFrequency = 0;
    pDevClockCtx->program.state = 0;
  }

  /* Initialize interrupt stuff */
  pDevCtx->interrupt.bConnected = FALSE;
  pDevCtx->interrupt.bEnabled = FALSE;
  pDevCtx->interrupt.bPolledMode = FALSE;
  dfParameterGetBoolean(pDevObj->pDrvObj, "PolledInterrupts", &pDevCtx->interrupt.bPolledMode);
  pDevCtx->interrupt.bUsingMsi = FALSE;
  for (i = 0; i < MAX_NUM_TARGET_FPGA; i++) {
    pDevCtx->interrupt.fpga[i].bDpcInitialized = FALSE;
    pDevCtx->interrupt.hwMon[i].bDpcInitialized = FALSE;
  }
  for (i = 0; i < MAX_NUM_DMA_ENGINE; i++) {
    pDevCtx->interrupt.dma[i].bDpcInitialized = FALSE;
  }

  /* Initialize notification stuff */
  for (i = 0; i < MAX_NUM_TARGET_FPGA; i++) {
    dfSpinLockInit(&pDevCtx->notification.fpgaInterrupt[i].lock);
    dfListInit(&pDevCtx->notification.fpgaInterrupt[i].header);
    dfSpinLockInit(&pDevCtx->notification.fpgaAlert[i].lock);
    dfListInit(&pDevCtx->notification.fpgaAlert[i].header);
  }
  dfSpinLockInit(&pDevCtx->notification.sensorPoll.lock);
  dfListInit(&pDevCtx->notification.sensorPoll.header);

  /* Initialize DMA stuff */
  for (i = 0; i < MAX_NUM_DMA_ENGINE; i++) {
    for (j = 0; j < 2; j++) {
      pDevCtx->dma.channel[i].descriptor[j].pDmaBuffer = NULL;
      pDevCtx->dma.channel[i].dmaBus[j].pTable = NULL;
    }
    dfSpinLockInit(&pDevCtx->dma.channel[i].lock);
    pDevCtx->dma.channel[i].curr.pTicket = NULL;
    pDevCtx->dma.channel[i].curr.state.bCancel = FALSE;
    pDevCtx->dma.channel[i].curr.state.main = 0;
    pDevCtx->dma.channel[i].curr.state.setup.state = 0;
    pDevCtx->dma.channel[i].curr.state.setup.count = 0;
    pDevCtx->dma.channel[i].curr.state.transfer.state = 0;
    pDevCtx->dma.channel[i].curr.state.teardown.state = 0;
    pDevCtx->dma.channel[i].curr.state.teardown.count = 0;
  }

  /* Initialize hardware monitoring stuff */
  dfSpinLockInit(&pDevCtx->hwMon.lock);
  pDevCtx->hwMon.polling.bShutdown = FALSE;
  pDevCtx->hwMon.polling.bTimerRunning = FALSE;
  pDevCtx->hwMon.polling.enable = 0;
  dfTimerInit(&pDevCtx->hwMon.polling.timer, pDevCtx);

  /* Initialize I/O module stuff */
  for (i = 0; i < MAX_NUM_IO_MODULE; i++) {
    pDevCtx->info.ioModule[i].bPresent = FALSE;
    pDevCtx->info.ioModule[i].pVpdBuffer = NULL;
  }

  /* Initialize common buffer stuff */
  pDevCtx->comBuf.count = 0;
  pDevCtx->comBuf.pComBufs = NULL;

  /* Initialize window stuff */
  for (i = 0; i < MAX_NUM_LOCAL_BUS_WINDOW; i++) {
    dfSpinLockInit(&pDevCtx->info.window[i].lock);
  }

  /* From here on, things may fail */
	dfDebugPrint(1, ("adb3CoreOnAdd: infallible initialization completed\n"));

  identifyDevice(pDevCtx, &pDevObj->hardwareId, &bHwMatch);
  if (!bHwMatch) {
    dfDebugPrint(0, ("*** adb3CoreOnAdd: unrecognized hardware ID\n"));
    goto failed;
  }

  if (NULL != pDevCtx->methods.pModelBootstrap) {
    pDevCtx->methods.pModelBootstrap(pDevCtx);
  } else {
    dfDebugPrint(0, ("*** adb3CoreOnAdd: identifyDevice failed to set a ModelBootstrap method\n"));
    goto failed;
  }

  /* Check that bootstrap routine set the appropriate methods */
  for (i = 0; i < pDevCtx->info.bootstrap.numFlashBank; i++) {
    if (pDevCtx->info.bootstrap.flash[i].width != 0) {
      if (NULL == pDevCtx->methods.pFlash) {
        dfDebugPrint(0, ("*** adb3CoreOnAdd: ModelBootstrap failed to initialize pFlash method\n"));
        goto failed;
      }
    }
  }
  if (pDevCtx->info.bootstrap.numClockGen != 0) {
    if (NULL == pDevCtx->methods.pClockProgram) {
      dfDebugPrint(0, ("*** adb3CoreOnAdd: ModelBootstrap failed to initialize pClockProgram method\n"));
      goto failed;
    }
    if (NULL == pDevCtx->methods.pClockWord) {
      dfDebugPrint(0, ("*** onAdd: ModelBootstrap failed to initialize pClockWord method\n"));
      goto failed;
    }
  }
  if (pDevCtx->info.bootstrap.numTargetFpga != 0) {
    if (NULL == pDevCtx->methods.pFpgaControl) {
      dfDebugPrint(0, ("*** adb3CoreOnAdd: ModelBootstrap failed to initialize pFpgaControl method\n"));
      goto failed;
    }
    if (NULL == pDevCtx->methods.pFpgaSelectMap) {
      dfDebugPrint(0, ("*** adb3CoreOnAdd: ModelBootstrap failed to initialize pFpgaSelectMap method\n"));
      goto failed;
    }
  }
  if (pDevCtx->info.bootstrap.vpdLength != 0) {
    if (NULL == pDevCtx->methods.pReadWriteVpd) {
      dfDebugPrint(0, ("*** adb3CoreOnAdd: ModelBootstrap failed to initialize pReadWriteVpd method\n"));
      goto failed;
    }
  }
  if (pDevCtx->info.bootstrap.numSensor != 0) {
    if (NULL == pDevCtx->methods.pReadSensor) {
      dfDebugPrint(0, ("*** adb3CoreOnAdd: ModelBootstrap failed to initialize pReadSensor method\n"));
      goto failed;
    }
  }
  if (NULL == pDevCtx->methods.pEnableInterrupts ||
    NULL == pDevCtx->methods.pInitHardware ||
    NULL == pDevCtx->methods.pIsr ||
    NULL == pDevCtx->methods.pMapBusResources ||
    NULL == pDevCtx->methods.pValidateBusResources ||
    NULL == pDevCtx->methods.pWindowConfig)
  {
    dfDebugPrint(0, ("*** adb3CoreOnAdd: ModelBootstrap failed to initialize all mandatory methods\n"));
    dfDebugPrint(0, ("*** adb3CoreOnAdd:   pEnableInterrupts=%p\n", pDevCtx->methods.pEnableInterrupts));
    dfDebugPrint(0, ("*** adb3CoreOnAdd:   pInitHardware=%p\n", pDevCtx->methods.pInitHardware));
    dfDebugPrint(0, ("*** adb3CoreOnAdd:   pIsr=%p\n", pDevCtx->methods.pIsr));
    dfDebugPrint(0, ("*** adb3CoreOnAdd:   pMapBusResources=%p\n", pDevCtx->methods.pMapBusResources));
    dfDebugPrint(0, ("*** adb3CoreOnAdd:   pValidateBusResources=%p\n", pDevCtx->methods.pValidateBusResources));
    dfDebugPrint(0, ("*** adb3CoreOnAdd:   pWindowConfig=%p\n", pDevCtx->methods.pWindowConfig));
    goto failed;
  }

  dfAssert(pDevCtx->info.bootstrap.numDmaChannel <= MAX_NUM_DMA_ENGINE);
  dfAssert(pDevCtx->info.bootstrap.numClockGen <= MAX_NUM_CLOCK_GENERATOR);
  dfAssert(pDevCtx->info.bootstrap.numTargetFpga <= MAX_NUM_TARGET_FPGA);
  dfAssert(pDevCtx->info.bootstrap.numSensor <= MAX_NUM_SENSOR);
  dfAssert(pDevCtx->info.bootstrap.numWindow <= MAX_NUM_LOCAL_BUS_WINDOW);

  if (!dfExportInterface(pDevObj, g_coreInterfaceName, sizeof(Adb3CoreInterface), &g_coreInterface, pDevCtx)) {
    dfDebugPrint(0, ("*** adb3CoreOnAdd: failed to export core interface\n"));
    goto failed;
  }
  pDevCtx->bCoreInterfaceRegd = TRUE;
  dfDebugPrint(1, ("adb3CoreOnAdd: exported core interface\n"));
  
#if defined(ADB3_DRIVER)
  if (!dfInterfaceRegister(pDevObj, &pDevCtx->userMode.iface, g_pInterfaceClass, &g_interfaceMethods)) {
    dfDebugPrint(0, ("*** adb3CoreOnAdd: failed to register user-mode interface\n"));
    goto failed;
  }
  dfInterfaceSetContext(&pDevCtx->userMode.iface, pDevCtx);
  pDevCtx->userMode.bRegistered = TRUE;
  dfDebugPrint(1, ("adb3CoreOnAdd: registered user-mode interface\n"));
#endif

  return TRUE;
  
 failed:
  freeDeviceContext(pDevCtx);
  return FALSE;
}

void
adb3CoreOnRemove(
  DfDeviceObject* pDevObj,
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreOnRemove: entered. pDevObj=%p pDevCtx=%p\n", pDevObj, pDevCtx));
  
  freeDeviceContext(pDevCtx);
}

boolean_t
adb3CoreOnPowerQuery(
	DfDeviceObject* pDevObj,
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int requestedState,
  boolean_t bIsShutdown)
{
  dfDebugPrint(1, ("adb3CoreOnPowerQuery: entered, pDevObj=%p pDevCtx=%p currentState=D%u requestedState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, requestedState, bIsShutdown ? "TRUE" : "FALSE"));

  /* Never veto power change */
  return FALSE;
}

void
adb3CoreOnPowerSet(
	DfDeviceObject* pDevObj,
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState,
  boolean_t bIsShutdown)
{
  dfDebugPrint(1, ("adb3CoreOnPowerSet: entered, pDevObj=%p pDevCtx=%p currentState=D%u newState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, newState, bIsShutdown ? "TRUE" : "FALSE"));

  if (currentState != newState) {
    if (currentState < newState) {
      if (bIsShutdown) {
        /* Don't bother saving state if we're shutting down */
        return;
      }
      /* Powering down, so disable interrupts */
      if (pDevCtx->interrupt.bConnected) {
        enableInterrupts(pDevCtx, FALSE);
        dfDebugPrint(1, ("adb3CoreOnPowerSet: disabled interrupts\n"));
      }
      if (NULL != pDevCtx->methods.pPowerChange) {
        /* Save model-specific state of hardware */
        pDevCtx->methods.pPowerChange(pDevCtx, currentState, newState);
      }
      /* TO DO - cancel operations that touch the hardware */
    } else {
      dfAssert(!bIsShutdown);
      /* Powering up */
      if (NULL != pDevCtx->methods.pPowerChange) {
        /* Restore model-specific state to hardware */
        pDevCtx->methods.pPowerChange(pDevCtx, currentState, newState);
      }
      /* Reenable interrupts after restoring state */
      if (pDevCtx->interrupt.bConnected) {
        enableInterrupts(pDevCtx, TRUE);
      }
    }
  }
}

boolean_t
adb3CoreDriverEntry(
  DfDriverObject* pDrvObj)
{
  dfDebugPrint(1, ("adb3CoreDriverEntry: entered, pDrvObj=%p\n", pDrvObj));

#if defined(ADB3_DRIVER)
  dfAssert(sizeof(ModelRegsAdmxrc6tx) == 0x1000U);
  dfAssert(sizeof(ModelRegsAdpexrc6tadv) == 0x1000U);
#endif
#if defined(ADMXRC_DRIVER)
  dfAssert(sizeof(Pci9080Registers) == 0x100U);
  dfAssert(sizeof(Pci9656Registers) == 0x108U);
  dfAssert(sizeof(Adb1BridgeRegisters) == 0x100U);
  dfAssert(sizeof(ModelRegsAdmxrc) == 0x10U);
  dfAssert(sizeof(ModelRegsAdmxrc2l) == 0x10U);
  dfAssert(sizeof(ModelRegsAdmxrc2) == 0x10U);
  dfAssert(sizeof(ModelRegsAdpdrc2) == 0x14U);
  dfAssert(sizeof(ModelRegsAdmxpl) == 0x1000U);
  dfAssert(sizeof(ModelRegsAdmxrc4) == 0x100000U);
  dfAssert(sizeof(ModelRegsAdpexrc4fx) == 0x800U);
  dfAssert(sizeof(ModelRegsAdb1V4V5) == 0x100000U);
  dfAssert(sizeof(ModelRegsAdcbbp) == 0x40000U);
  dfAssert(sizeof(ModelRegsAdmpcie6s1) == 0x10U);
#endif

#if defined(ADMXRC_DRIVER)
  dfAssert(sizeof(VpdAdmxrcRev0) == 0x14U);
  dfAssert(sizeof(VpdAdmxrc2lRev0) == 0x14U);
  dfAssert(sizeof(VpdAdmxrc2Rev0) == 0x16U);
  dfAssert(sizeof(VpdAdpdrc2Rev0) == 0x16U);
  dfAssert(sizeof(VpdAdpwrc2Rev0) == 0x16U);
  dfAssert(sizeof(VpdAdmxplRev0) == 0x20U);
  dfAssert(sizeof(VpdAdmxpRev0) == 0x24U);
  dfAssert(sizeof(VpdAdpxpiRev0) == 0x24U);
  dfAssert(sizeof(VpdAdmxrc4Rev0) == 0x14U);
  dfAssert(sizeof(VpdAdcpxrc4lxRev0) == 0x34U);
  dfAssert(sizeof(VpdAdmxrc4fxRev0) == 0x34U);
  dfAssert(sizeof(VpdAdpexrc4fxRev0) == 0x34U);
  dfAssert(sizeof(VpdAdpexrc4fxRev1) == 0x36U);
  dfAssert(sizeof(VpdAdmxrc5lxRev0) == 0x34U);
  dfAssert(sizeof(VpdAdmxrc5lxaRev0) == 0x34U);
  dfAssert(sizeof(VpdAdmxrc5t1Rev0) == 0x34U);
  dfAssert(sizeof(VpdAdmxrc5t2Rev0) == 0x3CU);
  dfAssert(sizeof(VpdAdmxrc5tda1Rev0) == 0x3CU);
  dfAssert(sizeof(VpdAdmamc5a2Rev0) == 0x3CU);
  dfAssert(sizeof(VpdAdmxrc5tzRev0) == 0x3CU);
  dfAssert(sizeof(VpdAdcbbpRev0) == 0x50U);
  dfAssert(sizeof(VpdAdmpcie6s1Rev0) == 0x18U);
#endif

  /* Initialize the ADB3 core interface */
  g_coreInterface.pAcquireAsync = coreAcquireAsync;
  g_coreInterface.pAcquireAsyncPoll = coreAcquireAsyncPoll;
  g_coreInterface.pAcquireInitialize = coreAcquireInitialize;
  g_coreInterface.pAcquireSync = coreAcquireSync;
  g_coreInterface.pAcquireCancel = coreAcquireCancel;
  g_coreInterface.pRelease = coreRelease;
  g_coreInterface.pTestDebugMsg = coreTestDebugMsg;
  g_coreInterface.pClearErrors = coreClearDeviceErrors;
  g_coreInterface.pGetDeviceStatus = coreGetDeviceStatus;
  g_coreInterface.pGetModel = coreGetModel;
  g_coreInterface.pGetNumClockGen = coreGetNumClockGen;
  g_coreInterface.pGetNumDmaChannel = coreGetNumDmaChannel;
  g_coreInterface.pGetNumFlashBank = coreGetNumFlashBank;
  g_coreInterface.pGetNumIoModule = coreGetNumIoModule;
  g_coreInterface.pGetNumSensor = coreGetNumSensor;
  g_coreInterface.pGetNumTargetFpga = coreGetNumTargetFpga;
  g_coreInterface.pGetNumWindow = coreGetNumWindow;
  g_coreInterface.pGetNumCommonBuffer = coreGetNumCommonBuffer;
  g_coreInterface.pGetVpdPointer = coreGetVpdPointer;
  g_coreInterface.pGetFlashBlockInfo = coreGetFlashBlockInfo;
  g_coreInterface.pGetFlashInfo = coreGetFlashInfo;
  g_coreInterface.pGetIoModuleInfo = coreGetIoModuleInfo;
  g_coreInterface.pGetSensorInfo = coreGetSensorInfo;
  g_coreInterface.pGetWindowInfo = coreGetWindowInfo;
  g_coreInterface.pGetWindowAddress = coreGetWindowAddress;
  g_coreInterface.pGetWindowConfig = coreGetWindowConfig;
  g_coreInterface.pSetWindowConfig = coreSetWindowConfig;
  g_coreInterface.pGetCommonBufferInfo = coreGetCommonBufferInfo;
  g_coreInterface.pSyncCommonBuffer = coreSyncCommonBuffer;
  g_coreInterface.pClockTestFreq = coreClockTestFreq;
  g_coreInterface.pNotificationReg = coreNotificationReg;
  g_coreInterface.pNotificationUnreg = coreNotificationUnreg;
  g_coreInterface.pReadSensor = coreReadSensor;
  g_coreInterface.pSetCoopLevel = coreSetCoopLevel;
#if defined(BUILD_DEBUGTS)
  g_coreInterface.pGetDebugTSInfo = coreGetDebugTSInfo;
  g_coreInterface.pGetDebugTSStatus = coreGetDebugTSStatus;
  g_coreInterface.pReadDebugTS = coreReadDebugTS;
  g_coreInterface.pResetDebugTS = coreResetDebugTS;
#endif
  g_coreInterface.pClockGetFreq = coreClockGetFreq;
  g_coreInterface.pClockProgram = coreClockProgram;
  g_coreInterface.pDmaBusCancel = coreDmaBusCancel;
  g_coreInterface.pDmaBusInitialize = coreDmaBusInitialize;
  g_coreInterface.pDmaBusStart = coreDmaBusStart;
  g_coreInterface.pDmaCancel = coreDmaCancel;
  g_coreInterface.pDmaInitialize = coreDmaInitialize;
  g_coreInterface.pDmaStart = coreDmaStart;
  g_coreInterface.pFlashErase = coreFlashErase;
  g_coreInterface.pFlashRead = coreFlashRead;
  g_coreInterface.pFlashSync = coreFlashSync;
  g_coreInterface.pFlashWrite = coreFlashWrite;
  g_coreInterface.pFpgaControl = coreFpgaControl;
  g_coreInterface.pFpgaSelectMap = coreFpgaSelectMap;
  g_coreInterface.pVpdRead = coreVpdRead;
  g_coreInterface.pVpdSync = coreVpdSync;
  g_coreInterface.pVpdWrite = coreVpdWrite;

  /* Check that core interface has no NULL pointers */
  {
    const size_t n = sizeof(g_coreInterface) / sizeof(void*);
    void* pTmp = (void*)&g_coreInterface;
    void** pp = (void**)pTmp;
    size_t i;
    boolean_t bError = FALSE;

    for (i = 0; i < n; i++) {
      if (NULL == pp[i]) {
        dfDebugPrint(0, ("*** adb3CoreDriverEntry: core interface entry %lu(0x%lx) is NULL\n", (unsigned long)i, (unsigned long)i));
        bError = TRUE;
      }
    }
    if (bError) {
      return FALSE;
    }
  }

#if defined(ADB3_DRIVER)
  g_interfaceMethods.pOnCleanup = NULL;
  g_interfaceMethods.pOnClose = onClose;
  g_interfaceMethods.pOnIoctl = onIoctl;
# if DF_NEED_THUNK
  g_interfaceMethods.pOnIoctlThunk = onIoctlThunk;
# endif
  g_interfaceMethods.pOnOpen = onOpen;

  if (!dfInterfaceClassRegister(pDrvObj, "adb3i", &g_pInterfaceClass)) {
    dfDebugPrint(0, ("*** adb3CoreDriverEntry: failed to register interface class '%s'\n", g_interfaceClassName));
    return FALSE;
  }
  dfDebugPrint(1, ("adb3CoreDriverEntry: registered interface class '%s', pClass=%p\n", g_interfaceClassName, g_pInterfaceClass));
#endif

	return TRUE;
}

extern void
adb3CoreDriverExit(
  DfDriverObject* pDrvObj)
{
  dfDebugPrint(1, ("adb3CoreDriverExit: entered, pDrvObj=%p\n", pDrvObj));

#if defined(ADB3_DRIVER)
  if (NULL != g_pInterfaceClass) {
    dfDebugPrint(1, ("adb3CoreDriverExit: unregistering interface class '%s', pClass=%p\n", g_interfaceClassName, g_pInterfaceClass));
    dfInterfaceClassUnregister(g_pInterfaceClass);
    g_pInterfaceClass = NULL;
  }
#endif
}

/*
** ---------------------------------------------------------------------------------
** These functions are built into the modular driver, but not the monolithic driver
** ---------------------------------------------------------------------------------
*/

#if !ADB3_MONOLITHIC
static boolean_t
onAdd(
	DfDeviceObject* pDevObj)
{
	Adb3CoreDeviceContext* pDevCtx;

	dfDebugPrint(1, ("onAdd: entered, pDevObj=%p\n", pDevObj));

	/* Initialize the device context */
	pDevCtx = (Adb3CoreDeviceContext*)dfDeviceGetContext(pDevObj);
	pDevCtx->pDevObj = pDevObj;

  return adb3CoreOnAdd(pDevObj, pDevCtx);
}
#endif

#if !ADB3_MONOLITHIC
static void
onRemove(
	DfDeviceObject* pDevObj)
{
	Adb3CoreDeviceContext* pDevCtx;

	dfDebugPrint(1, ("onRemove: entered, pDevObj=%p\n", pDevObj));

	pDevCtx = (Adb3CoreDeviceContext*)dfDeviceGetContext(pDevObj);
  adb3CoreOnRemove(pDevObj, pDevCtx);
}
#endif

#if !ADB3_MONOLITHIC
static boolean_t
onStart(
	DfDeviceObject* pDevObj,
	DfBusResources* pRawRes,
	DfBusResources* pTranslatedRes)
{
  Adb3CoreDeviceContext* pDevCtx;

  dfDebugPrint(1, ("onStart: entered, pDevObj=%p\n", pDevObj));

  pDevCtx = (Adb3CoreDeviceContext*)dfDeviceGetContext(pDevObj);
  return adb3CoreOnStart(pDevObj, pDevCtx, pRawRes, pTranslatedRes);
}
#endif

#if !ADB3_MONOLITHIC
static void
onStop(
	DfDeviceObject* pDevObj,
  boolean_t bCanTouchHardware)
{
	Adb3CoreDeviceContext* pDevCtx;

	dfDebugPrint(1, ("onStop: entered, pDevObj=%p bCanTouchHardware=%s\n",
    pDevObj, bCanTouchHardware ? "TRUE" : "FALSE"));

	pDevCtx = (Adb3CoreDeviceContext*)dfDeviceGetContext(pDevObj);
  adb3CoreOnStop(pDevObj, pDevCtx, bCanTouchHardware);
}
#endif

#if !ADB3_MONOLITHIC
static boolean_t
onPowerQuery(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int requestedState,
  boolean_t bIsShutdown)
{
	Adb3CoreDeviceContext* pDevCtx;

	pDevCtx = (Adb3CoreDeviceContext*)dfDeviceGetContext(pDevObj);

  dfDebugPrint(1, ("onPowerQuery: entered, pDevObj=%p pDevCtx=%p currentState=D%u requestedState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, requestedState, bIsShutdown ? "TRUE" : "FALSE"));

  return adb3CoreOnPowerQuery(pDevObj, pDevCtx, currentState, requestedState, bIsShutdown);
}

static void
onPowerSet(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int newState,
  boolean_t bIsShutdown)
{
	Adb3CoreDeviceContext* pDevCtx;

	pDevCtx = (Adb3CoreDeviceContext*)dfDeviceGetContext(pDevObj);

  dfDebugPrint(1, ("onPowerSet: entered, pDevObj=%p pDevCtx=%p currentState=D%u newState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, newState, bIsShutdown ? "TRUE" : "FALSE"));

  adb3CoreOnPowerSet(pDevObj, pDevCtx, currentState, newState, bIsShutdown);
}
#endif

#if !ADB3_MONOLITHIC
boolean_t
dfDriverEntry(
	DfDriverObject* pDrvObj)
{
  dfDebugPrint(1, ("dfDriverEntry: ADB3 Core Driver, pDrvObj=%p version=%lu.%lu.%lu.%lu\n",
    pDrvObj,
    (unsigned long)ADB3_VERSION_0, (unsigned long)ADB3_VERSION_1, (unsigned long)ADB3_VERSION_2, (unsigned long)ADB3_VERSION_3));

	pDrvObj->deviceContextSize = sizeof(Adb3CoreDeviceContext);
	pDrvObj->methods.pOnAdd = onAdd;
	pDrvObj->methods.pOnRemove = onRemove;
	pDrvObj->methods.pOnStart = onStart;
	pDrvObj->methods.pOnStop = onStop;
  pDrvObj->methods.pOnPowerQuery = onPowerQuery;
  pDrvObj->methods.pOnPowerSet = onPowerSet;

  return adb3CoreDriverEntry(pDrvObj);
}
#endif

#if !ADB3_MONOLITHIC
extern void
dfDriverExit(
	DfDriverObject* pDrvObj)
{
	dfDebugPrint(1, ("dfDriverExit: ADB3 Core Driver exiting\n"));

  adb3CoreDriverExit(pDrvObj);
}
#endif
