/*
** File: ics8430_61.c  
** Project: ADB3 core driver
** Purpose: Functions for programming the ICS8430-61 clock generator.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "ics8430_61_common.h"

static const uint32_t g_fVcoMin = 250000000U; /* Minimum allowed VCO frequency in Hz, from datasheet */
static const uint32_t g_fVcoMax = 500000000U; /* Maximum allowed VCO frequency in Hz, from datasheet */

static const uint32_t g_fOutMin = 32550000U;  /* Minimum possible output frequency */
static const uint32_t g_fOutMax = 500001000U; /* Maximum possible output frequency */

/* N divide values */
static const uint32_t g_nVal[8] = {
  2, 3, 4, 6, 8, 12, 16, 24
};

/* N range values corresponding to N divide values */
static const uint32_t g_nRange[8][2] = {
  { 250000000U, 500001000U }, /* Min., max. freq */
  { 166660000U, 333330000U },
  { 125000000U, 250000000U },
  {  83330000U, 166660000U },
  {  62500000U, 125000000U },
  {  41660000U,  83330000U },
  {  31250000U,  62500000U },
  {         0U,  41660000U }
};

static const uint32_t g_tVal = 0x2U;

static const uint32_t g_pDiv = 16U;

/*
** Bits 0-16 of the returned Clock Word are the serial clock programming word
** for the ICS8430-61, but bit 17 is the reference clock select, which is
** driven to the SEL pins of the ICS8430-61 by ADB1-XRC4FX.
*/
Ics8430_61ClockWordStatus
clockWordIcs8430_61(
  uint32_t flags,
  uint32_t refFrequency,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  unsigned int mMin, mMax;
	unsigned int m, n;
	unsigned int bestM = 0, bestNIdx = 0;
  uint32_t refFreqDiv2;
	uint32_t f, bestFrequency = 0;
	uint32_t error, bestError = (uint32_t)-1;

  if (refFrequency == 0) {
		return Ics8430_61ClockWordStatusGeneralFailure;
  }

	if (frequencyReq < g_fOutMin || frequencyReq > g_fOutMax) {
		return Ics8430_61ClockWordStatusOutOfRange;
	}

  /*
  ** Calculate minimum and maximum allowable M in order to keep VCO frequency
  ** within the range 250 to 500 MHz. Formula for fvco is:
  **
  ** fvco = fref * M / P
  ** => M = fvco * P / fref
  ** => M = (fvco * (P/2)) / (fref / 2)
  **
  ** Using 'fvco * 8' instead of 'fvco * 16' ensures that arithmetic will not
  ** overflow a 32-bit unsigned integer.
  */
  refFreqDiv2 = refFrequency / 2U;
  mMin = (g_fVcoMin * (g_pDiv / 2) + refFreqDiv2 - 1U) / refFreqDiv2;
  mMax = (g_fVcoMax * (g_pDiv / 2)) / refFreqDiv2;

	for (m = mMin; m <= mMax; m++) {
		for (n = 0; n < 8; n++) {
			/*
			** Do not modify the order of operations in this formula unless
			** you ensure that the new order cannot result in arithmetic
			** overflow.
			*/
			f = refFrequency / ((g_pDiv / 2) * g_nVal[n]) * m;
			if (f < g_nRange[n][0] || f > g_nRange[n][1]) {
				/* Output frequency is outside limits for N current value */
				continue;
			}
      if ((flags & CoreClockMinimumFrequency) && f < frequencyMin) {
        continue;
      }
      if ((flags & CoreClockMaximumFrequency) && f > frequencyMax) {
        continue;
      }
			if (f > (uint32_t)frequencyReq) {
				error = f - (uint32_t)frequencyReq;
			} else {
				error = (uint32_t)frequencyReq - f;
			}
			if (error < bestError) {
				bestError = error;
				bestFrequency = f;
				bestM = m;
				bestNIdx = n;
			}
		}
	}

	if (bestM == 0) {
		/* Failed to find a suitable value */
		return Ics8430_61ClockWordStatusOutOfRange;
	}

	pClockWord->frequency = bestFrequency;
	pClockWord->d0 = (g_tVal << 12) | (bestNIdx << 9) | (bestM << 0);

	dfDebugPrint(3, ("clockWordIcs8430_61: bestFrequency=%lu clockWord=0x%08lx\n", (unsigned long)bestFrequency, (unsigned long)pClockWord->d0));

  return Ics8430_61ClockWordStatusSuccess;
}

boolean_t
ics8430_61Init(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIcs8430_61Ctx,
  uint32_t refFrequency,
  uint32_t* pReg,
  DfMemoryHandle hReg,
  uint32_t* pFOut)
{
  uint32_t val32;
  uint32_t refFreqDiv2, fVco, fOut;
  uint32_t m, mMin, mMax, nIdx, n;

  dfTimerInit(&pIcs8430_61Ctx->timer, pDevCtx);
  pIcs8430_61Ctx->pReg = pReg;
  pIcs8430_61Ctx->hReg = hReg;
  pIcs8430_61Ctx->save.bLastValid = FALSE;

  if (0 == refFrequency) {
    dfDebugPrint(0, ("*** ics8430_61Init: reference clock frequency is invalid, refFrequency=%lu\n", (unsigned long)refFrequency));
    return FALSE;
  }

  /* Figure out current output frequency of ICS8430-61 */
  val32 = dfPciMemRead32(pIcs8430_61Ctx->hReg, pIcs8430_61Ctx->pReg);
  dfDebugPrint(2, ("ics8430_61Init: MCLOCK=0x%08lX\n", (unsigned long)val32));
  nIdx = (val32 >> 9) & 0x7U;
  m = val32 & 0x1FFU;
  n = g_nVal[nIdx];
  refFreqDiv2 = refFrequency / 2U;
  mMin = (g_fVcoMin * (g_pDiv / 2) + refFreqDiv2 - 1) / refFreqDiv2;
  mMax = g_fVcoMax * (g_pDiv / 2) / refFreqDiv2;
  if (m < mMin || m > mMax) {
    dfDebugPrint(0, ("*** ics8430_61Init: m=%u, outside allowed range [%u, %u]\n",
      (unsigned int)m, (unsigned int)mMin, (unsigned int)mMax));
    return FALSE;
  }
  fVco = refFreqDiv2 * m / (g_pDiv / 2);
  fOut = fVco / n;
  dfDebugPrint(1, ("ics8430_61Init: fVco=%lu fOut=%lu\n", (unsigned long)fVco, (unsigned long)fOut));
  *pFOut = fOut;
  return TRUE;
}

void
ics8430_61Uninit(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIcs8430_61Ctx)
{
  dfTimerUninit(&pIcs8430_61Ctx->timer);
}

static void
onCheckIcs8430_61(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  Ics8430_61ProgramRequest* pRequest = (Ics8430_61ProgramRequest*)pCallbackContext;
  Ics8430_61DeviceContext* pIcs8430_61Ctx = pRequest->pIcs8430_61Ctx;
  uint32_t val32;
  Ics8430_61ProgramStatus status = Ics8430_61ProgramStatusSuccess;

  val32 = dfPciMemRead32(pIcs8430_61Ctx->hReg, pIcs8430_61Ctx->pReg);
  dfDebugPrint(5, ("onCheckIcs8430_61: mclkCtl=0x%lx\n", (unsigned long)val32));  
  if (!(val32 & 0x80000000U)) {
    status = Ics8430_61ProgramStatusHardwareError;
  }
  if (Ics8430_61ProgramStatusSuccess == status) {
    pIcs8430_61Ctx->save.bLastValid = TRUE;
    pIcs8430_61Ctx->save.last = pRequest->clockWord;
  }
  pRequest->pCallback(pDevCtx, pIcs8430_61Ctx, pRequest, status);
}

Ics8430_61ProgramStatus
ics8430_61Program(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIcs8430_61Ctx,
  Ics8430_61ProgramRequest* pRequest,
  const CoreClockWord* pClockWord,
  Ics8430_61ProgramCallback* pCallback)
{
  uint32_t* pReg;
  DfMemoryHandle hReg;
  uint32_t clockWord;

  pRequest->pIcs8430_61Ctx = pIcs8430_61Ctx;
  pRequest->pCallback = pCallback;
  pRequest->clockWord = *pClockWord;

  clockWord = pClockWord->d0;
  pReg = pIcs8430_61Ctx->pReg;
  hReg = pIcs8430_61Ctx->hReg;
  dfPciMemWrite32(hReg, pReg, (0x3U << 30) | (clockWord & 0x3fffU));
  dfPciMemRead32(hReg, pReg);
  dfTimerSchedule(&pIcs8430_61Ctx->timer, dfTimeGet() + dfMicrosecondsToTime(50000), onCheckIcs8430_61, pRequest);
  return Ics8430_61ProgramStatusSuccess;
}

static void
onProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pV6ClkSynthCtx,
  Ics8430_61ProgramRequest* pRequest,
  Ics8430_61ProgramStatus status)
{
  Ics8430_61ProgramRequestSync* pRequestSync = (Ics8430_61ProgramRequestSync*)pRequest;

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

Ics8430_61ProgramStatus
ics8430_61ProgramSync(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIcs8430_61Ctx,
  const CoreClockWord* pClockWord)
{
  Ics8430_61ProgramRequestSync req;
  Ics8430_61ProgramStatus status;

  dfEventInit(&req.ev);
  status = ics8430_61Program(pDevCtx, pIcs8430_61Ctx, &req.request, pClockWord, onProgramDone);
  if (Ics8430_61ProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

void
ics8430_61SaveState(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIcs8430_61Ctx)
{
  /* Nothing to do yet */
}

Ics8430_61ProgramStatus
ics8430_61RestoreState(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIcs8430_61Ctx)
{
  Ics8430_61ProgramStatus status = Ics8430_61ProgramStatusSuccess;
 
  /* If the ICS843034-01 had been programmed before power-down, reprogram it with the same thing */
  if (pIcs8430_61Ctx->save.bLastValid) {
    status = ics8430_61ProgramSync(pDevCtx, pIcs8430_61Ctx, &pIcs8430_61Ctx->save.last);
  }

  return status;
}

ClockProgramStatus
ics8430_61MapProgramStatus(
  Ics8430_61ProgramStatus status)
{
  switch (status) {
  case Ics8430_61ProgramStatusSuccess:
    return ClockProgramSuccess;

  case Ics8430_61ProgramStatusHardwareError:
    return ClockProgramHardwareError;

  case Ics8430_61ProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
ics8430_61MapClockWordStatus(
  Ics8430_61ClockWordStatus status)
{
  switch (status) {
  case Ics8430_61ClockWordStatusSuccess:
    return ClockWordSuccess;

  case Ics8430_61ClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case Ics8430_61ClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
