/*
** File: coreif.h  
** Project: ADB3 core driver
** Purpose: Defines direct-call interface between module drivers and core driver
**
** (C) Copyright Alpha Data 2009-2015
*/

#if !defined(ADATA_CORE_COREIF_H)
#define ADATA_CORE_COREIF_H

#include <df.h>

typedef enum _CoreModelType {
  CoreModelGeneric             =    0x0000, /* Generic / unknown model      */
  CoreModelAdmxrc              =    0x0001, /* ADM-XRC                      */
  CoreModelAdmxrcp             =    0x0002, /* ADM-XRC-P                    */
  CoreModelAdmxrc2l            =    0x0003, /* ADM-XRC-II-Lite              */
  CoreModelAdmxrc2             =    0x0004, /* ADM-XRC-II                   */
  CoreModelAdmxp               =    0x0005, /* ADM-XP                       */
  CoreModelAdmxpl              =    0x0006, /* ADM-XPL                      */
  CoreModelAdpwrc2             =    0x0007, /* ADP-WRC-II                   */
  CoreModelAdpdrc2             =    0x0008, /* ADP-DRC-II                   */
  CoreModelAdpxpi              =    0x0009, /* ADP-XPI                      */
  CoreModelAdmxrc4ls           =    0x000A, /* ADM-XRC-4LS                  */
  CoreModelAdmxrc4lx           =    0x000B, /* ADM-XRC-4LX                  */
  CoreModelAdmxrc4sx           =    0x000C, /* ADM-XRC-4SX                  */
  CoreModelAdpexrc4fx          =    0x000D, /* ADPE-XRC-4FX                 */
  CoreModelAdmxrc4fx           =    0x000E, /* ADM-XRC-4FX                  */
  CoreModelAdmxrc5lx           =    0x000F, /* ADM-XRC-5LX                  */
  CoreModelAdmxrc5t1           =    0x0010, /* ADM-XRC-5T1                  */
  CoreModelAdmxrc5t2           =    0x0011, /* ADM-XRC-5T2                  */
  CoreModelAdcpxrc4lx          =    0x0012, /* ADCP-XRC-4LX                 */
  CoreModelAdmamc5a2           =    0x0013, /* ADM-AMC-5A2                  */
  CoreModelAdmxrc5tz           =    0x0014, /* ADM-XRC-5TZ                  */
  CoreModelAdcbbp              =    0x0015, /* ADC-BBP                      */
  CoreModelAdmxrc5t2adv        =    0x0016, /* ADM-XRC-5T2-ADV              */
  CoreModelAdmxrc5tda1         =    0x0017, /* ADM-XRC-5T-DA1               */
  CoreModelAdmxrc5t2adv6       =    0x0018, /* ADM-XRC-5T2-ADV6             */
  CoreModelAdmxrc5t2advcc1     =    0x0019, /* ADM-XRC-5T2-ADV-CC1          */
  CoreModelAdmpcie6s1          =    0x001A, /* ADM-PCIE-6S1                 */
  /* Current hardware */
  CoreModelAdpexrc5t           =    0x0100, /* ADPe-XRC-5T                  */
  CoreModelAdmxrc6tl           =    0x0101, /* ADM-XRC-6TL                  */
  CoreModelAdmxrc6t1           =    0x0102, /* ADM-XRC-6T1                  */
  CoreModelAdmxrc6tge          =    0x0103, /* ADM-XRC-6TGE                 */
  CoreModelAdmxrc6tadv8        =    0x0104, /* ADM-XRC-6TADV8               */
  CoreModelAdpexrc6t           =    0x0105, /* ADPE-XRC-6T                  */
  CoreModelAdpexrc6tl          =    0x0106, /* ADPE-XRC-6T-L                */
  CoreModelAdpexrc6tadvControl =    0x0107, /* ADPE-XRC-6T-ADV (controller) */
  CoreModelAdpexrc6tadvTarget0 = 0x1000108, /* ADPE-XRC-6T-ADV (target 0)   */
  CoreModelAdpexrc6tadvTarget1 = 0x2000108, /* ADPE-XRC-6T-ADV (target 1)   */
  CoreModelAdmxrc7k1           =    0x0109, /* ADM-XRC-7K1                  */
  CoreModelAdmxrc6tda1         =    0x010A, /* ADM-XRC-6T-DA1               */
  CoreModelAdmxrc7v1           =    0x010B, /* ADM-XRC-7V1                  */
  CoreModelAdmvpx37v2          =    0x010C, /* ADM-VPX3-7V2                 */
  CoreModelAdmxrc6tgel         =    0x010D, /* ADM-XRC-6TGEL                */
  CoreModelAdmpcie7v3          =    0x010E, /* ADM-PCIE-7V3                 */
  CoreModelAdmxrc7z1           =    0x010F, /* ADM-XRC-7Z1                  */
  CoreModelAdmxrc7z2           =    0x0110, /* ADM-XRC-7Z2                  */
  CoreModelAdmxrcku1           =    0x0111, /* ADM-XRC-KU1                  */
  CoreModelAdmpcieku3          =    0x0113, /* ADM-PCIE-KU3                 */
  CoreModelAdmpcieku32nd       = 0x1000113, /* ADM-PCIE-KU3 (2nd PCIe endpoint) */
  CoreModelAdmxrcku1P5         =    0x0114, /* ADM-XRC-KU1 (P5 endpoint)    */
  CoreModelAdmxrcku1P6         =    0x0115, /* ADM-XRC-KU1 (P6 endpoint)    */
  CoreModelAdmpcie8v3          =    0x0116, /* ADM-PCIE-8V3                 */
  CoreModelAdmpcie8v32nd       = 0x1000116, /* ADM-PCIE-8V3 (2nd PCIe endpoint) */
  CoreModelAdmpcie8k5          =    0x0117, /* ADM-PCIE-8K5                 */
  /* In-house hardware - not delivered to customers */
  CoreModelAdmxrc6t1Bert       =   0x10102, /* ADM-XRC-6T1 in BERT configuration */
  CoreModelReserved10103       =   0x10103, /* Do not use */
  CoreModelReserved10104       =   0x10104, /* Do not use */
  CoreModelAdpexrc6tBert       =   0x10105, /* ADPE-XRC-6T-L in BERT configuration */
  CoreModelAdpexrc6tlBert      =   0x10106, /* ADPE-XRC-6T-L in BERT configuration */
  CoreModelAdmxrc6tda1Bert     =   0x1010A, /* ADM-XRC-6T-DA1 in BERT configuration */
  /* Force this type to be 32 bits */
  CoreModelForce32Bits         = 0x7FFFFFFF
} CoreModelType;

/*
** Ticket structure, required in order to acquire and release resources
**
** Resource identifiers are 32-bit integers with the following bit fields:
**   [ 7 :  0]  Resource code
**   [15 :  8]  Resource index
**   [29 : 16]  (Reserved)
**   [30     ]  Exclusivity flag
**   [31     ]  Modify intent flag
*/

/* Flags for a resource identifier */
#if 0
#define CORE_RESOURCE_READONLY    (0x0U << 31)             /* Resource state won't be modified */
#define CORE_RESOURCE_MODIFY      (0x1U << 31)             /* Resource may be modified */
#define CORE_RESOURCE_SHARE       (0x0U << 30)             /* Can share resource with other readers */
#define CORE_RESOURCE_EXCLUSIVE   (0x1U << 30)             /* Can't share resource with other readers */
#endif

/* This defines the resource index field in a resource identifier */
#define CORE_RESOURCE_INDEX_BITS  (8)
#define CORE_RESOURCE_INDEX_COUNT (1U << CORE_RESOURCE_INDEX_BITS)
#define CORE_RESOURCE_INDEX_MASK  (CORE_RESOURCE_INDEX_COUNT - 1U)
#define CORE_RESOURCE_INDEX_ANY   (CORE_RESOURCE_INDEX_MASK)

/* This defines the resource code field in a resource identifier */
#define CORE_RESOURCE_CODE_BITS   (8)
#define CORE_RESOURCE_CODE_COUNT  (1U << CORE_RESOURCE_CODE_BITS)
#define CORE_RESOURCE_CODE_MASK   (CORE_RESOURCE_CODE_COUNT - 1U)
#define CORE_RESOURCE_CLOCKGEN    (0) /* Clock generator */
#define CORE_RESOURCE_DMAENGINE   (1) /* DMA engine */
#define CORE_RESOURCE_EEPROM      (2) /* EEPROM */
#define CORE_RESOURCE_FLASH       (3) /* Flash */
#define CORE_RESOURCE_TARGETFPGA  (4) /* Target FPGA */
#define CORE_RESOURCE_SYSMON      (5) /* System monitor */
#define CORE_RESOURCE_VPD         (6) /* Vital product data */

#define CORE_RESOURCE_MAKE_ID(CODE, INDEX) (((CODE) << CORE_RESOURCE_CODE_SHIFT) | ((INDEX) << CORE_RESOURCE_INDEX_SHIFT))

/*
** Result of a call to pAcquireSync or pAcquireAsync methods
*/
typedef enum _CoreAcquireStatus {
  CoreAcquireSuccess         =  0, /* Acquired the the resource immediately */
  CoreAcquireWaiting         =  1, /* Waiting for the resource */
  CoreAcquireFailed          =  2, /* Failed to acquire the resources due to device failing */
  CoreAcquireRemoved         =  3, /* Failed to acquire the resources due to device disappearing / shutting down */
  CoreAcquireMustWait        =  4, /* Failed to acquire the resource immediately (if flags include CORE_ACQUIRE_NOWAIT) */
  CoreAcquireInvalid         =  5, /* An invalid resource ID was specified */
  CoreAcquireInvalidIndex    =  6, /* An invalid resource index was specified */
  CoreAcquireCancelled       =  7, /* The acquisition was cancelled */
  CoreAcquireUnexpectedError =  8  /* An unexpected error occurred */
} CoreAcquireStatus;

/*
** Wait behaviour flags, all of which are mutually exclusive to each other.
**
** CORE_ACQUIRE_NOWAIT               - Fail instead of waiting and don't use callback, if can't immediately get resource.
** CORE_ACQUIRE_SYNC_UNINTERRUPTIBLE - For synchronous acquisition, the thread wait is uninterruptible.
*/
#define CORE_ACQUIRE_NOWAIT           (0x1U << 0)
#define CORE_ACQUIRE_SYNC_UNINTERRUPTIBLE (0x1U << 1)

struct _CoreNotification;
typedef struct _CoreNotification CoreNotification;
struct _CoreTicketSystem;
typedef struct _CoreTicketSystem CoreTicketSystem;
struct _CoreTicket;
typedef struct _CoreTicket CoreTicket;
struct _CoreResourceDesc;
typedef struct _CoreResourceDesc CoreResourceDesc;

/* Interrupt service routine descriptor */

typedef void CoreNotificationCallback(CoreNotification* pNotification, unsigned int notifyType, void* pCallbackContext);

struct _CoreNotification {
  struct { /* Client must not touch */
    DfListNode node;
    unsigned int notifyType;
    CoreNotificationCallback* pCallback;
    void* pCallbackContext;
    unsigned int refCount;
    DfEvent* pRemoveEvent;
  } system;
};

/* Legal values for 'notifyType' */
#define CORE_NOTIFY_SENSOR_POLL                 (0x1)
#define CORE_NOTIFY_FPGA_INTERRUPT(targetIndex) ((targetIndex) | 0x100)
#define CORE_NOTIFY_FPGA_ALERT(targetIndex)     ((targetIndex) | 0x200)

/* Resource descriptor */

struct _CoreResourceDesc {
  struct { /* Client must not touch */
    CoreTicket*       pTicket;
    void*             pQueue;
    CoreResourceDesc* pPrev;
    CoreResourceDesc* pNext;
  } system;

  /* Client initializes following fields */
  uint8_t             code;  /* The requested resource code */
  uint8_t             index; /* The requested resource index; may be 'ANY' */

  /* System initializes following fields for client */
  uint8_t             actualIndex; /* Actual resource index (will be different to 'index' if 'index' == 'ANY') */
};

/* Callback for pAcquireAsync method */

typedef void (CoreAcquireCallback)(void* pDevice, CoreTicket* pTicket, CoreAcquireStatus result);

/* Resource ticket */

struct _CoreTicketSystem {
#if DF_DBG_BUILD
  boolean_t              bResourcesValid;
#endif
  boolean_t              bSync;
  boolean_t              bCancel;
  unsigned int           state;
  union {
    DfEvent*             pEvent;    /* bSync == TRUE */
    CoreAcquireCallback* pCallback; /* bSync == FALSE */
  } variant;
};

#define CoreTicketMakeStruct(NAME, N) \
struct NAME { \
  /* Client must not touch */ \
  CoreTicketSystem system; \
\
  /* Client initializes following fields */ \
  unsigned int             count; \
  CoreResourceDesc         resources[N]; /* Length array defined by macro parameter 'n' */ \
}

CoreTicketMakeStruct(_CoreTicket, 1);

typedef struct _CoreFlashBlockInfo {
  uint64_t address;
  uint64_t length;
  uint32_t flags;
} CoreFlashBlockInfo;

/* Flags for CoreFlashBlockInfo::flags */
#define COREFLASH_BOOTBLOCK (0x1U << 0)

typedef struct _CoreFlashInfo {
  uint64_t size;
  uint64_t largestBlock;
  uint64_t useableStart;
  uint64_t useableLength;
  uint16_t vendorId;
  uint16_t deviceId;
} CoreFlashInfo;

typedef enum _CoreFlashStatus {
  CoreFlashSuccess = 0,          /* Successful                             */
  CoreFlashInvalidIndex = 1,     /* Invalid Flash index was specified      */
  CoreFlashTimeout = 2,          /* Timeout occurred during operation      */
  CoreFlashVerifyFailed = 3,     /* Error verifying operation              */
  CoreFlashInvalidRegion = 4,    /* Invalid region was specified           */
  CoreFlashNotPresent = 5,       /* Hardware is not present                */
  CoreFlashInterrupted = 6,      /* Wait was interrupted                   */
  CoreFlashInvalidParameter = 7, /* An invalid parameter was passed        */
  CoreFlashEraseFailed = 8,      /* Erase failed                           */
  CoreFlashLockFailed = 9,       /* Lock / unlock failed                   */
  CoreFlashWriteFailed = 10,     /* Write failed                           */
  CoreFlashGeneralFailure = 11,  /* All other failures                     */
  CoreFlashInvalidTicket = 1000
} CoreFlashStatus;

typedef CoreFlashStatus CoreVpdStatus;
#define CoreVpdSuccess CoreFlashSuccess
#define CoreVpdTimeout CoreFlashTimeout
#define CoreVpdVerifyFailed CoreFlashVerifyFailed
#define CoreVpdInvalidRegion CoreFlashInvalidRegion
#define CoreVpdNotPresent CoreFlashNotPresent
#define CoreVpdInterrupted CoreFlashInterrupted
#define CoreVpdInvalidParameter CoreFlashInvalidParameter
#define CoreVpdEraseFailed CoreFlashEraseFailed
#define CoreVpdLockFailed CoreFlashLockFailed
#define CoreVpdWriteFailed CoreFlashWriteFailed
#define CoreVpdGeneralFailure CoreFlashGeneralFailure
#define CoreVpdInvalidTicket CoreFlashInvalidTicket

/* Configuration of local bus attributes of a window */

/* 'flags' for 'SetWindowConfig' method */
#define COREWINDOWCONFIG_SET_BASE         (0x1U << 0)
#define COREWINDOWCONFIG_SET_WIDTH        (0x1U << 1)
#define COREWINDOWCONFIG_SET_PREFETCH     (0x1U << 2)
#define COREWINDOWCONFIG_SET_BURST        (0x1U << 3)
/* Flags for fields of CoreWindowConfig struct */
#define COREWINDOWCONFIG_WIDTH_MASK       (0xFU)
#define COREWINDOWCONFIG_WIDTH_DEFAULT    (0U) /* Never returned by GetWindowConfig unless not supported by model */
#define COREWINDOWCONFIG_WIDTH_8BIT       (1U)
#define COREWINDOWCONFIG_WIDTH_16BIT      (2U)
#define COREWINDOWCONFIG_WIDTH_32BIT      (3U)
#define COREWINDOWCONFIG_WIDTH_64BIT      (4U)
#define COREWINDOWCONFIG_WIDTH_128BIT     (5U)
#define COREWINDOWCONFIG_WIDTH_256BIT     (6U)
#define COREWINDOWCONFIG_PREFETCH_MASK    (0x3U)
#define COREWINDOWCONFIG_PREFETCH_DEFAULT (0U) /* Never returned by GetWindowConfig unless not supported by model */
#define COREWINDOWCONFIG_PREFETCH_MIN     (1U)
#define COREWINDOWCONFIG_PREFETCH_NORMAL  (2U)
#define COREWINDOWCONFIG_PREFETCH_MAX     (3U)
#define COREWINDOWCONFIG_BURST_MASK       (0x3U)
#define COREWINDOWCONFIG_BURST_DEFAULT    (0U) /* Never returned by GetWindowConfig unless not supported by model */
#define COREWINDOWCONFIG_BURST_DISABLE    (1U)
#define COREWINDOWCONFIG_BURST_ENABLE     (2U)
typedef struct _CoreWindowConfig {
  uint64_t base;
  uint8_t  width;    /* See COREWINDOWCONFIG_WIDTH_* flags */
  uint8_t  prefetch; /* See COREWINDOWCONFIG_PREFETCH_* flags */
  uint8_t  burst;    /* See COREWINDOWCONFIG_BURST_* flags */
} CoreWindowConfig;

typedef enum _CoreWindowConfigStatus {
  CoreWindowConfigSuccess = 0,
  CoreWindowConfigInvalidIndex = 1, /* Window index was invalid */
  CoreWindowConfigNotSupported = 2, /* Setting of a particular parameter for specified window not supported by this model */
  CoreWindowConfigInvalidValue = 3  /* Value specified for a particular parameter specified window was invalid for this model */
} CoreWindowConfigStatus;

/* General information about a Window */

typedef struct _CoreWindowInfo {
  uint64_t busSize;
  uint64_t busBase;
  uint64_t localSize;
  uint64_t localBase;
  uint64_t translatedSize;
  uint64_t translatedBase;
  void*    pKernelBase;
  size_t   kernelSize;
} CoreWindowInfo;

typedef enum _CoreWindowStatus {
  CoreWindowSuccess = 0,
  CoreWindowInvalidIndex = 1
} CoreWindowStatus;

typedef enum _CoreWindowAddressStatus {
  CoreWindowAddressSuccess = 0,
  CoreWindowAddressInvalidIndex = 1,
  CoreWindowAddressInvalidRegion = 2
} CoreWindowAddressStatus;

typedef enum _CoreIoModuleStatus {
  CoreIoModuleSuccess = 0,
  CoreIoModuleInvalidIndex = 1
} CoreIoModuleStatus;

typedef struct _CoreIoModuleInfo {
  boolean_t bIsFmc; /* TRUE => FMC, FALSE => XRM */
  boolean_t bPresent;
  boolean_t bLegacyXrm;
  boolean_t bForce2v5;
  unsigned int vpdLength;
  uint8_t* pVpdBuffer;
} CoreIoModuleInfo;

typedef struct _CoreSensorInfo {
  char description[28];
  uint32_t capabilities;
  struct {
    int16_t intPart;
    uint16_t fracPart;
  } error;
  uint32_t datatype;
  uint32_t unit;
  int32_t exponent;
} CoreSensorInfo;

typedef union _CoreSensorValue {
  boolean_t booleanValue;
  struct {
    int32_t intPart;
    uint32_t fracPart;
  } doubleValue;
  int32_t int32Value;
  uint32_t uint32Value;
} CoreSensorValue;

typedef enum _CoreSensorStatus {
  CoreSensorSuccess = 0,
  CoreSensorInvalidIndex = 1,
  CoreSensorHardwareError = 2,
  CoreSensorUnexpectedError = 3
} CoreSensorStatus;

typedef enum _CoreSensorDatatype {
  CoreSensorDataBool = 0,
  CoreSensorDataDouble = 1,
  CoreSensorDataInt32 = 2,
  CoreSensorDataUInt32 = 3
} CoreSensorDatatype;

typedef enum _CoreSensorUnit {
  CoreSensorUnitNone = 0,
  CoreSensorUnitA = 1,
  CoreSensorUnitV = 2,
  CoreSensorUnitC = 3,
  CoreSensorUnitHz = 4,
  CoreSensorUnitRpm = 5,
  CoreSensorUnitS = 6
} CoreSensorUnit;

typedef enum _CoreFpgaControlStatus {
  CoreFpgaControlSuccess = 0,
  CoreFpgaControlInvalidIndex = 1,
  CoreFpgaControlInvalidOp = 2,
  CoreFpgaControlNotSupported = 3,
  CoreFpgaControlInvalidTicket = 1000
} CoreFpgaControlStatus;

typedef enum _CoreFpgaSelectMapStatus {
  CoreFpgaSelectMapSuccess = 0,
  CoreFpgaSelectMapInvalidIndex = 1,
  CoreFpgaSelectMapNotSupported = 2,
  CoreFpgaSelectMapInvalidTicket = 1000
} CoreFpgaSelectMapStatus;

typedef enum _CoreFpgaControlOp {
  CoreFpgaControlOpGetDone = 1,
  CoreFpgaControlOpGetLinkStatus = 2,
  CoreFpgaControlOpGetInit = 3,
  CoreFpgaControlOpSetProg = 4
} CoreFpgaControlOp;

static const uint32_t CoreClockMinimumFrequency = (0x1U << 0);
static const uint32_t CoreClockMaximumFrequency = (0x1U << 1);

typedef struct _CoreClockWord {
  uint64_t frequency;
  uint32_t d0, d1, d2;
} CoreClockWord;

typedef enum _CoreClockStatus {
  CoreClockSuccess = 0,
  CoreClockInvalidIndex = 1,
  CoreClockOutOfRange = 2,
  CoreClockHardwareError = 3,
  CoreClockGeneralFailure = 4,
  CoreClockInvalidTicket = 1000
} CoreClockStatus;

typedef void CoreClockProgCallback(CoreTicket* pTicket, void* pCallbackContext, CoreClockStatus status);

typedef struct _CoreDebugTSInfo {
  uint64_t frequency;
  uint64_t initTimestamp;
  DfTime initTime;
  size_t bufferSize;
} CoreDebugTSInfo;

typedef struct _CoreDebugTSStatus {
  size_t count;
  uint32_t status;
} CoreDebugTSStatus;

typedef enum _CoreReadDebugTSResult {
  CoreReadDebugTSSuccess = 0,
  CoreReadDebugTSNotEnabled = 1,
  CoreReadDebugTSInvalidFacility = 2,
  CoreReadDebugTSInvalidRegion = 3
} CoreReadDebugTSResult;

typedef enum _CoreDebugTSResult {
  CoreDebugTSSuccess = 0,
  CoreDebugTSNotEnabled = 1,
  CoreDebugTSInvalidFacility = 2
} CoreDebugTSResult;

static const uint32_t CoreDmaFixedLocal = (0x1U << 0);

typedef struct _CoreDmaParameters {
  DfBufferDescription* pBufferDescription;
  uint32_t modeFlags; /* May be ignored on some models */
  boolean_t bWriteToLocal;
  boolean_t bFixedLocalAddress;
  struct {
    size_t offset;
    size_t length;
    uint64_t localAddress;
  } region;
} CoreDmaParameters;

typedef struct _CoreDmaBusParameters {
  uint32_t modeFlags; /* May be ignored on some models */
  boolean_t bWriteToLocal;
  boolean_t bFixedLocalAddress;
  struct {
    size_t offset;
    size_t length;
    uint64_t localAddress;
    uint64_t busAddress;
  } region;
} CoreDmaBusParameters;

typedef enum _CoreDmaStatus {
  CoreDmaSuccess = 0,
  CoreDmaInvalidChannel = 1,
  CoreDmaInvalidRegion = 2,
  CoreDmaDirectionNotSupported = 3,
  CoreDmaHardwareError = 4,
  CoreDmaCancelled = 5,
  CoreDmaUnexpectedError = 6,
  CoreDmaMappingFailure = 7,
  CoreDmaInvalidTicket = 1000
} CoreDmaStatus;

typedef void CoreDmaCallback(CoreTicket* pTicket, void* pCallbackContext, CoreDmaStatus status);

typedef enum _CoreCommonBufferStatus {
  CoreCommonBufferSuccess = 0,
  CoreCommonBufferInvalidIndex = 1,
  CoreCommonBufferInvalidRegion = 2
} CoreCommonBufferStatus;

typedef struct _CoreCommonBufferInfo {
  uint64_t busBase;
  void*    pKernelBase;
  size_t   size;
} CoreCommonBufferInfo;

typedef struct _CoreDeviceStatus {
  uint32_t window;
  uint32_t dmaEngine;
  uint32_t directMaster;
  uint32_t model[2];
} CoreDeviceStatus;

typedef struct _CoreClient {
  DfListNode node;
  boolean_t bInList;
} CoreClient;

DF_DECLARE_INLINE_FUNC(void, adb3CoreClientInit)(
  CoreClient* pClient)
{
  pClient->bInList = FALSE;
}

typedef enum _CoreCoopLevel {
  CoreCoopLevelUnset = 0,
  CoreCoopLevelNormal = 1,
  CoreCoopLevelExclusive = 2
} CoreCoopLevel;

typedef struct _Adb3CoreInterface {
  /* Acquire and release resources */
  void                    (*pAcquireAsync)        (void* pDevice, CoreTicket* pTicket, uint32_t flags, CoreAcquireCallback* pCallback);
  CoreAcquireStatus       (*pAcquireAsyncPoll)    (void* pDevice, CoreTicket* pTicket, uint32_t flags, CoreAcquireCallback* pCallback);
  CoreAcquireStatus       (*pAcquireInitialize)   (void* pDevice, CoreTicket* pTicket);
  boolean_t               (*pAcquireCancel)       (void* pDevice, CoreTicket* pTicket);
  CoreAcquireStatus       (*pAcquireSync)         (void* pDevice, CoreTicket* pTicket, uint32_t flags);
  void                    (*pRelease)             (void* pDevice, CoreTicket* pTicket);

  /* Methods that do not require a ticket */
  void                    (*pTestDebugMsg)        (void* pDevice, int level, const char* pString);
  void                    (*pClearErrors)         (void* pDevice);
  void                    (*pGetDeviceStatus)     (void* pDevice, CoreDeviceStatus* pDeviceStatus);
  CoreModelType           (*pGetModel)            (void* pDevice);
  unsigned int            (*pGetNumClockGen)      (void* pDevice);
  unsigned int            (*pGetNumDmaChannel)    (void* pDevice);
  unsigned int            (*pGetNumFlashBank)     (void* pDevice);
  unsigned int            (*pGetNumIoModule)      (void* pDevice);
  unsigned int            (*pGetNumSensor)        (void* pDevice);
  unsigned int            (*pGetNumTargetFpga)    (void* pDevice);
  unsigned int            (*pGetNumWindow)        (void* pDevice);
  unsigned int            (*pGetNumCommonBuffer)  (void* pDevice);
  const void*             (*pGetVpdPointer)       (void* pDevice);
  CoreFlashStatus         (*pGetFlashInfo)        (void* pDevice, unsigned int bankIndex, CoreFlashInfo* pFlashInfo);
  CoreFlashStatus         (*pGetFlashBlockInfo)   (void* pDevice, unsigned int bankIndex, uint64_t address, CoreFlashBlockInfo* pBlockInfo);
  CoreIoModuleStatus      (*pGetIoModuleInfo)     (void* pDevice, unsigned int ioModuleIndex, CoreIoModuleInfo* pIoModuleInfo);
  CoreSensorStatus        (*pGetSensorInfo)       (void* pDevice, unsigned int sensorIndex, CoreSensorInfo* pSensorInfo);
  CoreWindowStatus        (*pGetWindowInfo)       (void* pDevice, unsigned int windowIndex, CoreWindowInfo* pWindowInfo);
  CoreWindowAddressStatus (*pGetWindowAddress)    (void* pDevice, unsigned int windowIndex, size_t offset, size_t length, void** pKernelAddress);
  CoreWindowConfigStatus  (*pGetWindowConfig)     (void* pDevice, unsigned int windowIndex, CoreWindowConfig* pConfig);
  CoreWindowConfigStatus  (*pSetWindowConfig)     (void* pDevice, unsigned int windowIndex, uint32_t flags, const CoreWindowConfig* pConfig);
  CoreCommonBufferStatus  (*pGetCommonBufferInfo) (void* pDevice, unsigned int commonBufferIndex, CoreCommonBufferInfo* pCommonBufferInfo);
  CoreCommonBufferStatus  (*pSyncCommonBuffer)    (void* pDevice, unsigned int commonBufferIndex, boolean_t bDmaWrite, size_t offset, size_t length);
  CoreClockStatus         (*pClockTestFreq)       (void* pDevice, unsigned int clockIndex, uint32_t flags, uint64_t frequencyReq, uint64_t frequencyMin, uint64_t frequencyMax, CoreClockWord* pClockWord);
  boolean_t               (*pNotificationReg)     (void* pDevice, unsigned int notifyType, CoreNotification* pNotification, CoreNotificationCallback* pCallback, void* pCallbackContext);
  boolean_t               (*pNotificationUnreg)   (void* pDevice, CoreNotification* pNotification);
  CoreSensorStatus        (*pReadSensor)          (void* pDevice, unsigned int sensorIndex, CoreSensorValue* pSensorValue);
  boolean_t               (*pSetCoopLevel)        (void* pDevice, CoreClient* pClientHandle, CoreCoopLevel coopLevel);
#if defined(BUILD_DEBUGTS)
  CoreDebugTSResult       (*pGetDebugTSInfo)      (void* pDevice, unsigned int facility, CoreDebugTSInfo* pDebugTSInfo);
  CoreDebugTSResult       (*pGetDebugTSStatus)    (void* pDevice, unsigned int facility, CoreDebugTSStatus* pDebugTSStatus);
  CoreReadDebugTSResult   (*pReadDebugTS)         (void* pDevice, unsigned int facility, size_t offset, size_t count, uint32_t flags, void* pBuffer);
  CoreDebugTSResult       (*pResetDebugTS)        (void* pDevice, unsigned int facility);
#endif

  /* Methods requiring a valid ticket */
  CoreClockStatus         (*pClockGetFreq)        (void* pDevice, CoreTicket* pTicket, unsigned int clockIndex, uint64_t* pFrequency);
  CoreClockStatus         (*pClockProgram)        (void* pDevice, CoreTicket* pTicket, unsigned int clockIndex, const CoreClockWord* pClockWord, CoreClockProgCallback* pCallback, void* pCallbackContext);
  CoreDmaStatus           (*pDmaBusCancel)        (void* pDevice, CoreTicket* pTicket, unsigned int channelIndex);
  CoreDmaStatus           (*pDmaBusInitialize)    (void* pDevice, CoreTicket* pTicket, unsigned int channelIndex, const CoreDmaBusParameters* pParameters, CoreDmaCallback* pCallback, void* pCallbackContext);
  void                    (*pDmaBusStart)         (void* pDevice, CoreTicket* pTicket, unsigned int channelIndex);
  CoreDmaStatus           (*pDmaCancel)           (void* pDevice, CoreTicket* pTicket, unsigned int channelIndex);
  CoreDmaStatus           (*pDmaInitialize)       (void* pDevice, CoreTicket* pTicket, unsigned int channelIndex, const CoreDmaParameters* pParameters, CoreDmaCallback* pCallback, void* pCallbackContext);
  void                    (*pDmaStart)            (void* pDevice, CoreTicket* pTicket, unsigned int channelIndex);
  CoreFlashStatus         (*pFlashErase)          (void* pDevice, CoreTicket* pTicket, unsigned int flashIndex, uint64_t address, uint64_t length);
  CoreFlashStatus         (*pFlashRead)           (void* pDevice, CoreTicket* pTicket, unsigned int flashIndex, uint64_t address, size_t length, void* pBuffer);
  CoreFlashStatus         (*pFlashSync)           (void* pDevice, CoreTicket* pTicket, unsigned int flashIndex);
  CoreFlashStatus         (*pFlashWrite)          (void* pDevice, CoreTicket* pTicket, unsigned int flashIndex, uint64_t address, size_t length, const void* pBuffer);
  CoreFpgaControlStatus   (*pFpgaControl)         (void* pDevice, CoreTicket* pTicket, unsigned int targetIndex, CoreFpgaControlOp operation, boolean_t* pbValue);
  CoreFpgaSelectMapStatus (*pFpgaSelectMap)       (void* pDevice, CoreTicket* pTicket, unsigned int targetIndex, boolean_t bWrite, size_t length, void* pBuffer);
  CoreVpdStatus           (*pVpdRead)             (void* pDevice, CoreTicket* pTicket, size_t address, size_t length, void* pBuffer);
  CoreVpdStatus           (*pVpdSync)             (void* pDevice, CoreTicket* pTicket);
  CoreVpdStatus           (*pVpdWrite)            (void* pDevice, CoreTicket* pTicket, size_t address, size_t length, const void* pBuffer);
} Adb3CoreInterface;

#endif
