/*
** File: coreflash.c  
** Project: ADB3 core driver
** Purpose: Implements functions for Flash programming in Core Interface.
**
** (C) Copyright Alpha Data 2009-2010
*/

#include <df.h>

#include "coreflash.h"
#include "device.h"
#include "flash.h"
#include "resource.h"

DF_ATTR_PASSIVE_CONTEXT
CoreFlashStatus
coreFlashErase(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int flashIndex,
  uint64_t offset,
  uint64_t length)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  CoreFlashStatus status = CoreFlashSuccess;
  uint64_t physicalAddr, chunk;

  dfAssertPassiveContext();

  if (flashIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    return CoreFlashInvalidIndex;
  }
  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_FLASH, flashIndex)) {
    return CoreFlashInvalidTicket;
  }

  while (length) {
    if (NULL != pDevCtx->methods.pFlashRemap) {
      /* On this model, physical and logical addresses are not the same. */
      pDevCtx->methods.pFlashRemap(pDevCtx, flashIndex, offset, length, &physicalAddr, &chunk);
      dfAssert(chunk <= length);
    } else {
      /* On this model, physical and logical addresses are the same. */
      physicalAddr = offset;
      chunk = length;
    }
    status = flashErase(pDevCtx, flashIndex, physicalAddr, chunk);
    if (CoreFlashSuccess != status) {
      break;
    }
    offset += chunk;
    length -= chunk;
  }

  return status;
}

DF_ATTR_PASSIVE_CONTEXT
CoreFlashStatus
coreFlashRead(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int flashIndex,
  uint64_t offset,
  size_t length,
  void* pBuffer)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  CoreFlashStatus status = CoreFlashSuccess;
  uint64_t physicalAddr, chunk;
  uint8_t* p8 = (uint8_t*)pBuffer;

  dfAssertPassiveContext();

  if (flashIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    return CoreFlashInvalidIndex;
  }
  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_FLASH, flashIndex)) {
    return CoreFlashInvalidTicket;
  }

  while (length) {
    if (NULL != pDevCtx->methods.pFlashRemap) {
      /* On this model, physical and logical addresses are not the same. */
      pDevCtx->methods.pFlashRemap(pDevCtx, flashIndex, offset, length, &physicalAddr, &chunk);
      dfAssert(chunk <= length);
    } else {
      /* On this model, physical and logical addresses are the same. */
      physicalAddr = offset;
      chunk = length;
    }
    status = flashRead(pDevCtx, flashIndex, physicalAddr, (size_t)chunk, (void*)p8); /* Cast is safe as chunk <= length (size_t) */
    if (CoreFlashSuccess != status) {
      break;
    }
    p8 += (size_t)chunk; /* Cast is safe as chunk <= length (size_t) */
    offset += chunk;
    length -= (size_t)chunk; /* Cast is safe as chunk <= length (size_t) */
  }

  return status;
}

DF_ATTR_PASSIVE_CONTEXT
CoreFlashStatus
coreFlashSync(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int flashIndex)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;

  dfAssertPassiveContext();

  if (flashIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    return CoreFlashInvalidIndex;
  }
  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_FLASH, flashIndex)) {
    return CoreFlashInvalidTicket;
  }

  return flashSync(pDevCtx, flashIndex);
}

DF_ATTR_PASSIVE_CONTEXT
CoreFlashStatus
coreFlashWrite(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int flashIndex,
  uint64_t offset,
  size_t length,
  const void* pBuffer)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  CoreFlashStatus status = CoreFlashSuccess;
  uint64_t physicalAddr, chunk;
  uint8_t* p8 = (uint8_t*)pBuffer;

  dfAssertPassiveContext();

  if (flashIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    return CoreFlashInvalidIndex;
  }
  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_FLASH, flashIndex)) {
    return CoreFlashInvalidTicket;
  }

  while (length) {
    if (NULL != pDevCtx->methods.pFlashRemap) {
      /* On this model, physical and logical addresses are not the same. */
      pDevCtx->methods.pFlashRemap(pDevCtx, flashIndex, offset, length, &physicalAddr, &chunk);
      dfAssert(chunk <= length);
    } else {
      /* On this model, physical and logical addresses are the same. */
      physicalAddr = offset;
      chunk = length;
    }
    status = flashWrite(pDevCtx, flashIndex, physicalAddr, (size_t)chunk, (void*)p8); /* Cast is safe as chunk <= length (size_t) */
    if (CoreFlashSuccess != status) {
      break;
    }
    p8 += (size_t)chunk; /* Cast is safe as chunk <= length (size_t) */
    offset += chunk;
    length -= (size_t)chunk; /* Cast is safe as chunk <= length (size_t) */
  }

  return status;
}

CoreFlashStatus
coreGetFlashInfo(
  void* pInterfaceContext,
  unsigned int flashIndex,
  CoreFlashInfo* pFlashInfo)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  Adb3CoreFlashInfo* pInfo;

  if (flashIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    return CoreFlashInvalidIndex;
  }

  pInfo = &pDevCtx->info.flash[flashIndex];
  if (!pInfo->bPresent) {
    return CoreFlashNotPresent;
  }
  pFlashInfo->size = pInfo->totalSize;
  pFlashInfo->useableStart = pInfo->targetArea.start;
  pFlashInfo->useableLength = pInfo->targetArea.length;
  if (pInfo->bLegacyFlash) {
    LegacyFlashInformation* pDetail = pInfo->detail.pLegacy;

    if (NULL == pDetail) {
      return CoreFlashNotPresent;
    }
    pFlashInfo->largestBlock = pDetail->largestBlockSize;
    pFlashInfo->vendorId = pDetail->vendorId;
    pFlashInfo->deviceId = pDetail->deviceId;
  } else {
    CfiInformation* pDetail = pInfo->detail.pCfi;

    if (NULL == pDetail) {
      return CoreFlashNotPresent;
    }
    pFlashInfo->largestBlock = pDetail->validation.largestBlock;
    pFlashInfo->vendorId = pDetail->validation.vendorId;
    pFlashInfo->deviceId = pDetail->validation.deviceId;
  }

  return CoreFlashSuccess;
}

CoreFlashStatus
coreGetFlashBlockInfo(
  void* pInterfaceContext,
  unsigned int flashIndex,
  uint64_t address,
  CoreFlashBlockInfo* pBlockInfo)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  CoreFlashStatus status;
  uint64_t blockStart, blockSize;
  uint64_t physicalAddr, dummy;

  if (NULL != pDevCtx->methods.pFlashRemap) {
    /* On this model, physical and logical addresses are not the same. */
    /* NOTE: we assume that remapping region boundaries are always on block boundaries. */
    pDevCtx->methods.pFlashRemap(pDevCtx, flashIndex, address, 1U, &physicalAddr, &dummy);
  } else {
    /* On this model, physical and logical addresses are the same. */
    physicalAddr = address;
  }

  status = flashIdentifyBlock(pDevCtx, flashIndex, physicalAddr, &blockStart, &blockSize);
  if (status == CoreFlashSuccess) {
    pBlockInfo->address = blockStart;
    pBlockInfo->length = blockSize;
    pBlockInfo->flags = 0; /* TO DO */
  }
  return status;
}

unsigned int
coreGetNumFlashBank(
  void* pInterfaceContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;

  return pDevCtx->info.bootstrap.numFlashBank;
}
