/*
** File: avr_sensor.c  
** Project: ADB3 core driver
** Purpose: Shared code for reading AVR uC sensors.
**
** (C) Copyright Alpha Data 2012
***/

#include <df.h>

#include "admxrc6tx_common.h"
#include "avr_sensor.h"

CoreSensorStatus
avrSensorCurrentRead(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int avrCurrentIndex,
  AvrCurrentScaling* pScaling,
  unsigned int count,
  CoreSensorValue* pReading)
{
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  uint16_t val16, fieldMask;
  int32_t factor, value;
  unsigned int fieldOffset;
  boolean_t bFieldSigned;

  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  if (avrCurrentIndex < count) {
    fieldOffset = pScaling[avrCurrentIndex].fieldOffset;
    fieldMask = pScaling[avrCurrentIndex].fieldMask;
    bFieldSigned = pScaling[avrCurrentIndex].bFieldSigned;
    val16 = dfPciMemRead16(hModel, (uint16_t*)(pModelRegs->sensor + fieldOffset));
    val16 = (uint16_t)(val16 & fieldMask);
    factor = pScaling[avrCurrentIndex].factor;
    if (bFieldSigned) {
      value = ((int16_t)val16 * factor) >> 10;
    } else {
      value = ((uint16_t)val16 * factor) >> 10;
    }
    pReading->doubleValue.intPart = value >> 16;
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }
  return CoreSensorInvalidIndex;
}

CoreSensorStatus
avrSensorMiscRead(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int avrMiscIndex,
  AvrMiscScaling* pScaling,
  CoreSensorValue* pReading)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;
  unsigned int fieldOffset;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  switch (avrMiscIndex) {
  case 0:
    fieldOffset = pScaling->manufTime.fieldOffset;
    val32 = dfPciMemRead32(hModel, (uint32_t*)(pModelRegs->model3.avr.sensor + fieldOffset));
    pReading->doubleValue.intPart = (int32_t)(val32 >> 2);
    pReading->doubleValue.fracPart = (val32 & 0x3) << 30;
    break;

  case 1:
    fieldOffset = pScaling->eventCount.fieldOffset;
    val32 = 0;
    if (pScaling->eventCount.bSillyByteOrder) {
      /* "Interesting" byte order: bits 23:16 @ offset n, 7:0 @ offset n+1, 15:8 @ offset n+2 */
      val32 |= (uint32_t)dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + fieldOffset + 0) << 16;
      val32 |= (uint32_t)dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + fieldOffset + 1) << 0;
      val32 |= (uint32_t)dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + fieldOffset + 2) << 8;
    } else {
      /* Little-endian byte order: bits 7:0 @ offset n, 15:8 @ offset n+1, 23:16 @ offset n+2 */
      val32 |= (uint32_t)dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + fieldOffset + 0) << 0;
      val32 |= (uint32_t)dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + fieldOffset + 1) << 8;
      val32 |= (uint32_t)dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + fieldOffset + 2) << 16;
    }
    pReading->uint32Value = val32;
    break;

  default:
    return CoreSensorInvalidIndex;
  }

  return CoreSensorSuccess;
}

CoreSensorStatus
avrSensorTempRead(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int avrTempIndex,
  AvrTempScaling* pScaling,
  unsigned int count,
  CoreSensorValue* pReading)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint16_t val16, fieldMask;
  int32_t factor, offset, value;
  unsigned int fieldOffset;
  boolean_t bFieldSigned;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (avrTempIndex < count) {
    fieldOffset = pScaling[avrTempIndex].fieldOffset;
    fieldMask = pScaling[avrTempIndex].fieldMask;
    bFieldSigned = pScaling[avrTempIndex].bFieldSigned;
    factor = pScaling[avrTempIndex].factor;
    offset = pScaling[avrTempIndex].offset;
    val16 = dfPciMemRead16(hModel, (uint16_t*)(pModelRegs->model3.avr.sensor + fieldOffset));
    val16 = (uint16_t)(val16 & fieldMask);
    if (bFieldSigned) {
      /* Raw ADC value is signed (two's complement); assumes that bit 15 of val16 is sign bit */
      value = ((int32_t)(int16_t)val16 * factor) >> 4;
    } else {
      /* Raw ADC value is unsigned */
      value = ((uint32_t)val16 * factor) >> 4;
    }
    value += offset;
    pReading->doubleValue.intPart = value >> 16;
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }
  return CoreSensorInvalidIndex;
}

CoreSensorStatus
avrSensorVoltageRead(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int avrVoltageIndex,
  AvrVoltageScaling* pScaling,
  unsigned int count,
  CoreSensorValue* pReading)
{
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  uint16_t val16, fieldMask;
  int32_t factor, value;
  unsigned int fieldOffset;
  boolean_t bFieldSigned;

  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  if (avrVoltageIndex < count) {
    fieldOffset = pScaling[avrVoltageIndex].fieldOffset;
    fieldMask = pScaling[avrVoltageIndex].fieldMask;
    bFieldSigned = pScaling[avrVoltageIndex].bFieldSigned;
    val16 = dfPciMemRead16(hModel, (uint16_t*)(pModelRegs->sensor + fieldOffset));
    val16 = (uint16_t)(val16 & fieldMask);
    factor = pScaling[avrVoltageIndex].factor;
    if (bFieldSigned) {
      value = ((int16_t)val16 * factor) >> 10;
    } else {
      value = ((uint16_t)val16 * factor) >> 10;
    }
    pReading->doubleValue.intPart = value >> 16;
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }
  return CoreSensorInvalidIndex;
}
