/*
** File: avr_flash.c  
** Project: ADB3 core driver
** Purpose: Functions for accessing Flash memory in AVR uC.
**
** (C) Copyright Alpha Data 2011
*/

#include "device.h"
#include "avr_common.h"
#include "avr_flash.h"

#define DEBUGLEVEL_INIT    (2) /* Level of debug messages relating to initialization / uninitialization */
#define DEBUGLEVEL_CANCEL  (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_ERROR   (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_WARNING (1) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_NORMAL  (6) /* Level of debug messages relating to normal operation */

/*
** This constant defines the AVR Flash size threshold where the new Flash
** commands ("r" / "w") are used instead of the old ones ("R" / "W"). This is
** necessary because the old Flash commands are limited to addresses below
** 64k, but some models use a larger AVR device with 128k or more of Flash
** memory.
*/
static const unsigned int g_oldFlashCmdThreshold = 0x10000U;

static AvrFlashStatus
mapAvrTransStatus(
  AvrTransStatus transStatus)
{
  switch (transStatus) {
  case AvrTransStatusSuccess:
    return AvrFlashSuccess;

  case AvrTransStatusTimeout:
    return AvrFlashTimeout;

  default:
    return AvrFlashGeneralFailure;
  }
}

/* Must be called in thread context */
extern AvrFlashStatus
avrFlashRead(
	Adb3CoreDeviceContext* pDevCtx,
  AvrDeviceContext* pAvrCtx,
  uint32_t address,
  uint32_t length,
  uint8_t* pBuffer)
{
  const uint32_t mask = pAvrCtx->flashPageSize - 1U;
  uint8_t* pMergeBuffer = pAvrCtx->vpd.pMergeBuffer;
  AvrTransStatus status;
  uint32_t offset, alignedAddress, chunk;
  unsigned int cmdLength;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avrFlashRead: address=0x%x length=%u(0x%x) pBuffer=%p\n",
    (unsigned int)address, (unsigned int)length, (unsigned int)length, pBuffer));

  if (CHECK_BOUNDS(address, length, pAvrCtx->flashSize)) {
    return AvrFlashInvalidRegion;
  }

  if (!avrIsServiceMode(pAvrCtx)) {
    return AvrFlashNotServiceMode;
  }

  /* AVR chips with > 64k of Flash need new AVR Flash command */
  cmdLength = (pAvrCtx->flashSize > g_oldFlashCmdThreshold) ? 5U : 3U;

  while (length) {
    offset = address & mask;
    alignedAddress = address & ~mask;
    chunk = pAvrCtx->flashPageSize - offset;
    if (chunk > length) {
      chunk = length;
    }
    if (cmdLength >= 5) {
      /* New Flash command */
      pMergeBuffer[0] = 'r';
      pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
      pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
      pMergeBuffer[3] = (uint8_t)(alignedAddress >> 16);
      pMergeBuffer[4] = 0;
    } else {
      /* Old Flash command */
      pMergeBuffer[0] = 'R';
      pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
      pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
    }
    status = avrTransactionSync(pAvrCtx, &pMergeBuffer[0], cmdLength, &pMergeBuffer[0], pAvrCtx->flashPageSize, 1000000U);
    if (status != AvrTransStatusSuccess) {
      return mapAvrTransStatus(status);
    }
    dfCopyKK(pBuffer, pMergeBuffer + offset, chunk);
    length -= chunk;
    address += chunk;
    pBuffer += chunk;
  }
  return AvrFlashSuccess;
}

/* Must be called in thread context */
extern AvrFlashStatus
avrFlashWrite(
	Adb3CoreDeviceContext* pDevCtx,
  AvrDeviceContext* pAvrCtx,
  uint32_t address,
  uint32_t length,
  const uint8_t* pData)
{
  const uint32_t mask = pAvrCtx->flashPageSize - 1U;
  uint8_t* pMergeBuffer = pAvrCtx->vpd.pMergeBuffer;
  AvrTransStatus status;
  uint32_t offset, alignedAddress, chunk;
  unsigned int cmdLength;
  uint8_t result;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avrFlashWrite: address=0x%x length=%u(0x%x) pData=%p\n",
    (unsigned int)address, (unsigned int)length, (unsigned int)length, pData));

  if (CHECK_BOUNDS(address, length, pAvrCtx->flashSize)) {
    return AvrFlashInvalidRegion;
  }

  if (!avrIsServiceMode(pAvrCtx)) {
    return AvrFlashNotServiceMode;
  }

  /* AVR chips with > 64k of Flash need new AVR Flash command */
  cmdLength = (pAvrCtx->flashSize > g_oldFlashCmdThreshold) ? 5U : 3U;

  while (length) {
    offset = address & mask;
    alignedAddress = address & ~mask;
    chunk = pAvrCtx->flashPageSize - offset;
    if (chunk > length) {
      chunk = length;
    }
    if (chunk != pAvrCtx->flashPageSize) {
      /* Not writing an entire page, so need to read page first, before merging in new data */
      if (cmdLength >= 5) {
        /* New Flash command */
        pMergeBuffer[0] = 'r';
        pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
        pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
        pMergeBuffer[3] = (uint8_t)(alignedAddress >> 16);
        pMergeBuffer[4] = 0;
      } else {
      /* Old Flash command */
        pMergeBuffer[0] = 'R';
        pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
        pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
      }
      status = avrTransactionSync(pAvrCtx, &pMergeBuffer[0], cmdLength, &pMergeBuffer[cmdLength], pAvrCtx->flashPageSize, 1000000U);
      if (status != AvrTransStatusSuccess) {
        return mapAvrTransStatus(status);
      }
    }
    if (cmdLength >= 5) {
      /* New Flash command */
      pMergeBuffer[0] = 'w';
      pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
      pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
      pMergeBuffer[3] = (uint8_t)(alignedAddress >> 16);
      pMergeBuffer[4] = 0;
    } else {
      /* Old Flash command */
      pMergeBuffer[0] = 'W';
      pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
      pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
    }
    /* Merge in new data (may or may not be an entire page of data) */
    dfCopyKK(pMergeBuffer + cmdLength + offset, pData, chunk);
    status = avrTransactionSync(pAvrCtx, &pMergeBuffer[0], cmdLength + pAvrCtx->flashPageSize, &pMergeBuffer[0], 1, 1000000U);
    if (status != AvrTransStatusSuccess) {
      return mapAvrTransStatus(status);
    }
    result = pMergeBuffer[0];
    if (0 != result) {
      return (AvrFlashStatus)((unsigned int)AvrFlashHardwareFailure + result);
    }
    length -= chunk;
    address += chunk;
    pData += chunk;
  }
  return AvrFlashSuccess;
}

/* Must be called in thread context */
extern AvrFlashStatus
avrFlashErase(
	Adb3CoreDeviceContext* pDevCtx,
  AvrDeviceContext* pAvrCtx,
  uint32_t address,
  uint32_t length)
{
  const uint32_t mask = pAvrCtx->flashPageSize - 1U;
  uint8_t* pMergeBuffer = pAvrCtx->vpd.pMergeBuffer;
  AvrTransStatus status;
  uint32_t offset, alignedAddress, chunk;
  unsigned int cmdLength;
  uint8_t result;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avrFlashRead: address=0x%x length=%u(0x%x)\n",
    (unsigned int)address, (unsigned int)length, (unsigned int)length));

  if (CHECK_BOUNDS(address, length, pAvrCtx->flashSize)) {
    return AvrFlashInvalidRegion;
  }

  if (!avrIsServiceMode(pAvrCtx)) {
    return AvrFlashNotServiceMode;
  }

  /* AVR chips with > 64k of Flash need new AVR Flash command */
  cmdLength = (pAvrCtx->flashSize > g_oldFlashCmdThreshold) ? 5U : 3U;

  while (length) {
    offset = address & mask;
    alignedAddress = address & ~mask;
    chunk = pAvrCtx->flashPageSize - offset; /* Cast is safe due to assertions in avrInit() */
    if (chunk > length) {
      chunk = length; /* Cast is safe due to above check */
    }
    if (chunk != pAvrCtx->flashPageSize) {
      if (cmdLength >= 5) {
        /* New Flash command */
        pMergeBuffer[0] = 'r';
        pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
        pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
        pMergeBuffer[3] = (uint8_t)(alignedAddress >> 16);
        pMergeBuffer[4] = 0;
      } else {
        /* Old Flash command */
        pMergeBuffer[0] = 'R';
        pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
        pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
      }
      status = avrTransactionSync(pAvrCtx, &pMergeBuffer[0], cmdLength, &pMergeBuffer[cmdLength], pAvrCtx->flashPageSize, 1000000U);
      if (status != AvrTransStatusSuccess) {
        return mapAvrTransStatus(status);
      }
    }
    if (cmdLength >= 5) {
      /* New Flash command */
      pMergeBuffer[0] = 'w';
      pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
      pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
      pMergeBuffer[3] = (uint8_t)(alignedAddress >> 16);
      pMergeBuffer[4] = 0;
    } else {
      /* Old Flash command */
      pMergeBuffer[0] = 'W';
      pMergeBuffer[1] = (uint8_t)(alignedAddress >> 0);
      pMergeBuffer[2] = (uint8_t)(alignedAddress >> 8);
    }
    dfSetMemory8(pMergeBuffer + cmdLength + offset, chunk, 0xFFU);
    status = avrTransactionSync(pAvrCtx, &pMergeBuffer[0], cmdLength + pAvrCtx->flashPageSize, &pMergeBuffer[0], 1, 1000000U);
    if (status != AvrTransStatusSuccess) {
      return mapAvrTransStatus(status);
    }
    result = pMergeBuffer[0];
    if (0 != result) {
      return (AvrFlashStatus)((unsigned int)AvrFlashHardwareFailure + result);
    }
    length -= chunk;
    address += chunk;
  }
  return AvrFlashSuccess;
}
