/*
** File: avr2_flash.c  
** Project: ADB3 core driver
** Purpose: Functions for accessing Flash memory in AVR uC via AVR2 interface.
**
** (C) Copyright Alpha Data 2015
*/

#include "device.h"
#include "avr2_common.h"
#include "avr2_flash.h"

#define DEBUGLEVEL_INIT    (2) /* Level of debug messages relating to initialization / uninitialization */
#define DEBUGLEVEL_CANCEL  (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_ERROR   (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_WARNING (1) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_NORMAL  (6) /* Level of debug messages relating to normal operation */

#define AVR2_FACILITY_BOOT_FLASH (0x06)
#define AVR2_COMMAND_FLASH_READ  (0x80)
#define AVR2_COMMAND_FLASH_WRITE (0x82)

static Avr2FlashStatus
mapAvr2TransStatus(
  Avr2TransStatus transStatus)
{
  switch (transStatus) {
  case Avr2TransStatusSuccess:
    return Avr2FlashSuccess;

  case Avr2TransStatusRXTimeout:
  case Avr2TransStatusTXTimeout:
  case Avr2TransStatusQueueTimeout:
    return Avr2FlashTimeout;

  default:
    return Avr2FlashGeneralFailure;
  }
}

/* Must be called in thread context */
extern Avr2FlashStatus
avr2FlashRead(
	Adb3CoreDeviceContext* pDevCtx,
  Avr2DeviceContext* pAvr2Ctx,
  uint32_t address,
  uint32_t length,
  uint8_t* pBuffer)
{
  const uint32_t mask = pAvr2Ctx->flashPageSize - 1U;
  const unsigned int cmdLength = 5U;
  const unsigned int respLength = 3U;
  uint8_t* pMergeBuffer = pAvr2Ctx->vpd.pMergeBuffer;
  Avr2TransStatus status;
  uint32_t pageIndex, pageOffset, alignedAddress, chunk;
  uint8_t result;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avr2FlashRead: address=0x%x length=%u(0x%x) pBuffer=%p\n",
    (unsigned int)address, (unsigned int)length, (unsigned int)length, pBuffer));

  if (CHECK_BOUNDS(address, length, pAvr2Ctx->flashSize)) {
    return Avr2FlashInvalidRegion;
  }

  if (!avr2IsServiceMode(pAvr2Ctx)) {
    return Avr2FlashNotServiceMode;
  }

  while (length) {
    pageIndex = address / pAvr2Ctx->flashPageSize;
    pageOffset = address & mask;
    alignedAddress = address & ~mask;
    chunk = pAvr2Ctx->flashPageSize - pageOffset;
    if (chunk > length) {
      chunk = length;
    }
    pMergeBuffer[0] = AVR2_FACILITY_BOOT_FLASH;
    pMergeBuffer[1] = AVR2_COMMAND_FLASH_READ;
    pMergeBuffer[2] = pAvr2Ctx->flashPageOrder;
    pMergeBuffer[3] = (uint8_t)(pageIndex & 0xFFU);
    pMergeBuffer[4] = (uint8_t)((pageIndex >> 8) & 0xFFU);
    status = avr2TransactionSync(pAvr2Ctx, &pMergeBuffer[0], cmdLength, &pMergeBuffer[0], respLength + pAvr2Ctx->flashPageSize, 1000000U, NULL);
    if (status != Avr2TransStatusSuccess) {
      return mapAvr2TransStatus(status);
    }
    result = pMergeBuffer[2];
    if (0 != result) {
      return (Avr2FlashStatus)((unsigned int)Avr2FlashHardwareFailure + result);
    }
    dfCopyKK(pBuffer, pMergeBuffer + respLength + pageOffset, chunk);
    length -= chunk;
    address += chunk;
    pBuffer += chunk;
  }
  return Avr2FlashSuccess;
}

/* Must be called in thread context */
extern Avr2FlashStatus
avr2FlashWrite(
	Adb3CoreDeviceContext* pDevCtx,
  Avr2DeviceContext* pAvr2Ctx,
  uint32_t address,
  uint32_t length,
  const uint8_t* pData)
{
  const uint32_t mask = pAvr2Ctx->flashPageSize - 1U;
  const unsigned int cmdLength = 5U;
  const unsigned int respLength = 3U;
  uint8_t* pMergeBuffer = pAvr2Ctx->vpd.pMergeBuffer;
  Avr2TransStatus status;
  uint32_t pageOffset, pageIndex, alignedAddress, chunk;
  uint8_t result;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avr2FlashWrite: address=0x%x length=%u(0x%x) pData=%p\n",
    (unsigned int)address, (unsigned int)length, (unsigned int)length, pData));

  if (CHECK_BOUNDS(address, length, pAvr2Ctx->flashSize)) {
    return Avr2FlashInvalidRegion;
  }

  if (!avr2IsServiceMode(pAvr2Ctx)) {
    return Avr2FlashNotServiceMode;
  }

  while (length) {
    pageIndex = address / pAvr2Ctx->flashPageSize;
    pageOffset = address & mask;
    alignedAddress = address & ~mask;
    chunk = pAvr2Ctx->flashPageSize - pageOffset;
    if (chunk > length) {
      chunk = length;
    }
    if (chunk != pAvr2Ctx->flashPageSize) {
      /* Not writing an entire page, so need to read page first, before merging in new data */
      pMergeBuffer[0] = AVR2_FACILITY_BOOT_FLASH;
      pMergeBuffer[1] = AVR2_COMMAND_FLASH_READ;
      pMergeBuffer[2] = pAvr2Ctx->flashPageOrder;
      pMergeBuffer[3] = (uint8_t)(pageIndex & 0xFFU);
      pMergeBuffer[4] = (uint8_t)((pageIndex >> 8) & 0xFFU);
      /* For the response buffer, we pass &pMergeBuffer[2] so that the page of
      ** read data lines up correctly with where the page of data needs to be
      ** for a write. */
      status = avr2TransactionSync(pAvr2Ctx, &pMergeBuffer[0], cmdLength, &pMergeBuffer[2], respLength + pAvr2Ctx->flashPageSize, 1000000U, NULL);
      if (status != Avr2TransStatusSuccess) {
        return mapAvr2TransStatus(status);
      }
      /* We look at pMergeBuffer[4] NOT pMergeBuffer[2] for the result of the operation,
      ** because we passed &pMergeBuffer[2] as the response buffer */
      result = pMergeBuffer[4];
      if (0 != result) {
        return (Avr2FlashStatus)((unsigned int)Avr2FlashHardwareFailure + result);
      }
    }
    pMergeBuffer[0] = AVR2_FACILITY_BOOT_FLASH;
    pMergeBuffer[1] = AVR2_COMMAND_FLASH_WRITE;
    pMergeBuffer[2] = pAvr2Ctx->flashPageOrder;
    pMergeBuffer[3] = (uint8_t)(pageIndex & 0xFFU);
    pMergeBuffer[4] = (uint8_t)((pageIndex >> 8) & 0xFFU);
    /* Merge in new data (may or may not be an entire page of data) */
    dfCopyKK(pMergeBuffer + cmdLength + pageOffset, pData, chunk);
    status = avr2TransactionSync(pAvr2Ctx, &pMergeBuffer[0], cmdLength + pAvr2Ctx->flashPageSize, &pMergeBuffer[0], respLength, 1000000U, NULL);
    if (status != Avr2TransStatusSuccess) {
      return mapAvr2TransStatus(status);
    }
    result = pMergeBuffer[2];
    if (0 != result) {
      return (Avr2FlashStatus)((unsigned int)Avr2FlashHardwareFailure + result);
    }
    length -= chunk;
    address += chunk;
    pData += chunk;
  }
  return Avr2FlashSuccess;
}

/* Must be called in thread context */
extern Avr2FlashStatus
avr2FlashErase(
	Adb3CoreDeviceContext* pDevCtx,
  Avr2DeviceContext* pAvr2Ctx,
  uint32_t address,
  uint32_t length)
{
  const uint32_t mask = pAvr2Ctx->flashPageSize - 1U;
  const unsigned int cmdLength = 5U;
  const unsigned int respLength = 3U;
  uint8_t* pMergeBuffer = pAvr2Ctx->vpd.pMergeBuffer;
  Avr2TransStatus status;
  uint32_t pageIndex, pageOffset, alignedAddress, chunk;
  uint8_t result;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avr2FlashRead: address=0x%x length=%u(0x%x)\n",
    (unsigned int)address, (unsigned int)length, (unsigned int)length));

  if (CHECK_BOUNDS(address, length, pAvr2Ctx->flashSize)) {
    return Avr2FlashInvalidRegion;
  }

  if (!avr2IsServiceMode(pAvr2Ctx)) {
    return Avr2FlashNotServiceMode;
  }

  while (length) {
    pageIndex = address / pAvr2Ctx->flashPageSize;
    pageOffset = address & mask;
    alignedAddress = address & ~mask;
    chunk = pAvr2Ctx->flashPageSize - pageOffset; /* Cast is safe due to assertions in avr2Init() */
    if (chunk > length) {
      chunk = length; /* Cast is safe due to above check */
    }
    if (chunk != pAvr2Ctx->flashPageSize) {
      /* Not writing an entire page, so need to read page first, before merging in new data */
      pMergeBuffer[0] = AVR2_FACILITY_BOOT_FLASH;
      pMergeBuffer[1] = AVR2_COMMAND_FLASH_READ;
      pMergeBuffer[2] = pAvr2Ctx->flashPageOrder;
      pMergeBuffer[3] = (uint8_t)(pageIndex & 0xFFU);
      pMergeBuffer[4] = (uint8_t)((pageIndex >> 8) & 0xFFU);
      status = avr2TransactionSync(pAvr2Ctx, &pMergeBuffer[0], cmdLength, &pMergeBuffer[0], respLength + pAvr2Ctx->flashPageSize, 1000000U, NULL);
      if (status != Avr2TransStatusSuccess) {
        return mapAvr2TransStatus(status);
      }
      result = pMergeBuffer[2];
      if (0 != result) {
        return (Avr2FlashStatus)((unsigned int)Avr2FlashHardwareFailure + result);
      }
    }
    pMergeBuffer[0] = AVR2_FACILITY_BOOT_FLASH;
    pMergeBuffer[1] = AVR2_COMMAND_FLASH_WRITE;
    pMergeBuffer[2] = pAvr2Ctx->flashPageOrder;
    pMergeBuffer[3] = (uint8_t)(pageIndex & 0xFFU);
    pMergeBuffer[4] = (uint8_t)((pageIndex >> 8) & 0xFFU);
    dfSetMemory8(pMergeBuffer + cmdLength + pageOffset, chunk, 0xFFU);
    status = avr2TransactionSync(pAvr2Ctx, &pMergeBuffer[0], cmdLength + pAvr2Ctx->flashPageSize, &pMergeBuffer[0], respLength, 1000000U, NULL);
    if (status != Avr2TransStatusSuccess) {
      return mapAvr2TransStatus(status);
    }
    result = pMergeBuffer[2];
    if (0 != result) {
      return (Avr2FlashStatus)((unsigned int)Avr2FlashHardwareFailure + result);
    }
    length -= chunk;
    address += chunk;
  }
  return Avr2FlashSuccess;
}
