/*
** File: admxrc6tx.h  
** Project: ADB3 core driver
** Purpose: Definitions common to ADM-XRC-6TL and ADM-XRC-6T1.
**
** (C) Copyright Alpha Data 2010-2012
*/

#ifndef _ADATA_CORE_ADMXRC6TX_H
#define _ADATA_CORE_ADMXRC6TX_H

#include <df.h>
#include "adb3.h"

#define ADMXRC6TX_FRU_ROM_SIZE (0x100U) /* In bytes */

/* States for system monitor alert state machine */
#define ADMXRC6Tx_HWMONSTATE_DISABLED (0)
#define ADMXRC6Tx_HWMONSTATE_ENABLED  (1)

/* Number of LM87 sensors, available in all firmware versions */
#define ADMXRC6TX_NUM_LM87_SENSOR  (10U)

/* Number of Xilinx system monitor sensors, only avaialable with 1.4 firmware (PCI rev ID 0x04) and later */
#define ADMXRC6TX_NUM_XSMON_SENSOR (3U)

/* Always LITTLE ENDIAN in hardware */
#pragma pack(1)
typedef struct _FlashRegsAdmxrc6tx {
	uint32_t ctl;                /* 0x0040 */
	uint32_t addr;               /* 0x0044 */
	uint32_t data;               /* 0x0048 */
	uint32_t _reserved3[1];      /* 0x004C */
} FlashRegsAdmxrc6tx;          /* 0x10 */
#pragma pack()

/* Always LITTLE ENDIAN in hardware */
#pragma pack(1)
typedef struct _IcapRegs {
	uint32_t wdata;              /* 0x00 */
	uint32_t rdata;              /* 0x04 */
	uint32_t cfgaddr;            /* 0x08 */
	uint32_t cfgdelay;           /* 0x0C */
	uint32_t ctl;                /* 0x10 */
	uint32_t icapstat;           /* 0x14 */
	uint32_t delaycnt;           /* 0x18 */
	uint32_t _reserved1;         /* 0x1C */
} IcapRegs;                    /* 0x20 */
#pragma pack()

/* Always LITTLE ENDIAN in hardware */
#pragma pack(1)
typedef struct _ModelRegsAdmxrc6tx {
  uint32_t pcbRev;             /* 0x0000 */
	uint32_t pwrStat;            /* 0x0004 */
	uint32_t buildDate;          /* 0x0008 */ /* Firmware ? (PCI rev ID ?) and later */
	uint32_t buildTime;          /* 0x000C */ /* Firmware ? (PCI rev ID ?) and later */
  union {
    struct {
      /* ADM-XRC-6TL */
  	  uint32_t lclkCtl;        /* 0x0010 */
	    uint32_t lclkStat1;      /* 0x0014 */
	    uint32_t lclkStat2;      /* 0x0018 */
	    uint32_t lclkStat3;      /* 0x001C */
    } v5ClkSynth;
    struct {
      /* ADM-XRC-6T1 firmware 1.6 (PCI rev 0x06) and later */
      /* ADM-XRC-6TGE */
	    uint32_t lclkCtl;        /* 0x0010 */
	    uint32_t lclkMmcmStat0;  /* 0x0014 */
	    uint32_t lclkMmcmStat1;  /* 0x0018 */
	    uint32_t lclkStat;       /* 0x001C */
    } v6ClkSynth;
  } model;
	uint8_t  selectMap[4];       /* 0x0020 */
	uint32_t fpgaCtl;            /* 0x0024 */
  uint32_t fpgaRead;           /* 0x0028 */
  uint32_t _reserved1[1];      /* 0x002C */
	uint32_t intCtl;             /* 0x0030 */
	uint32_t intStat;            /* 0x0034 */
	uint32_t _reserved2[2];      /* 0x0038 */
  FlashRegsAdmxrc6tx flash;    /* 0x0040 */
  uint32_t gpioBToT;           /* 0x0050 */
  uint32_t gpioTToT;           /* 0x0054 */
  uint32_t gtpStat;            /* 0x0058 */
	uint32_t _reserved4[1];      /* 0x005C */
  uint32_t mptlRecovery;       /* 0x0060 */
  uint32_t mptlStat;           /* 0x0064 */
  uint32_t mptlError;          /* 0x0068 */
	uint32_t _reserved5[5];      /* 0x006C */
  uint32_t xsmonCtl;           /* 0x0080 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
  uint32_t xsmonDI;            /* 0x0084 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
  uint32_t xsmonDO;            /* 0x0088 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
	uint32_t _reserved6[1];      /* 0x008C */
  uint32_t xsmonTemp1;         /* 0x0090 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
  uint32_t xsmonTemp2;         /* 0x0094 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
  uint32_t xsmonVInt1;         /* 0x0098 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
  uint32_t xsmonVInt2;         /* 0x009C */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
  uint32_t xsmonVAux1;         /* 0x00A0 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
  uint32_t xsmonVAux2;         /* 0x00A4 */ /* Firmware 1.4 (PCI rev ID 0x04) & later */
	uint32_t _reserved7[2];      /* 0x00A8 */
  union {                      /* 0x00B0 */
    union {
      /* Models with AVR uC */
      struct { /* Old interface for getting hardcoded board VPD, no longer used */
        uint32_t _reserved8[2];  /* 0x00B0 */
        uint32_t vpdAddr;        /* 0x00B8 */
        uint32_t vpdData;        /* 0x00BC */
      } obsolete;
      struct { /* Current uC interface */
        uint32_t ctlStat;        /* 0x00B0 */
        uint32_t _reserved9[3];  /* 0x00B4 */
      } curr;
    } avr;
  } model2;
  IcapRegs icap;               /* 0x00C0 ICAP interface may be present on certain Bridgeless models */
	uint32_t _reserved10[200];   /* 0x00E0 */
  uint32_t i2cCtl;             /* 0x0400 */
  uint32_t i2cStat;            /* 0x0404 */
  uint32_t sysMonStat;         /* 0x0408 */
  uint32_t xrmStat;            /* 0x040C */
	uint32_t _reserved11[60];    /* 0x0410 */
  uint8_t  sysMonBuf[256];     /* 0x0500 */
  uint8_t  xrmBuf[256];        /* 0x0600 ADM-PCIE-7V3/-KU3 VPD appears here */
  uint8_t  i2cBuf[256];        /* 0x0700 */
  union {                      /* 0x0800 */
    struct {
      /* Models with AVR uC */
      uint8_t vpd[256];        /* 0x0800 */
      uint8_t _reserved12[256]; /* 0x900 */
      uint8_t sensor[512];     /* 0x0A00 */
    } avr;
    struct {
      /* Models with UCD90120 system monitor */
      uint8_t _reserved12[512]; /* 0x0800 */
      uint8_t sensor[256];     /* 0x0A00 */
      uint8_t _reserved13[256]; /* 0x0B00 */
    } ucd90120;
  } model3;
  uint8_t _reserved14[1024];   /* 0x0C00 */
} ModelRegsAdmxrc6tx;          /* 0x1000 */
#pragma pack()

/* Temperature grade values for VPD */
#define ADMXRC6Tx_VPD_TEMPGRADE_COMMERCIAL  (0)
#define ADMXRC6Tx_VPD_TEMPGRADE_INDUSTRIAL  (1)
#define ADMXRC6Tx_VPD_TEMPGRADE_EXTENDED    (2)

/* Fields for ModelRegsAdmxrc6tx::pwrStat */
#define ADMXRC6Tx_V_PWRSTAT_PSU1OK      DF_VAL32S(0x1U, 0, 1) /* 1.5V & 2.5V supplies OK */
#define ADMXRC6Tx_V_PWRSTAT_1V0OK       DF_VAL32S(0x1U, 1, 1) /* 1.0V supply OK */
#define ADMXRC6Tx_V_PWRSTAT_1V8OK       DF_VAL32S(0x1U, 2, 1) /* 1.8V supply OK */
#define ADMXRC6Tx_V_PWRSTAT_VRAOK       DF_VAL32S(0x1U, 3, 1) /* VRef for DRAM banks 0, 1 OK */
#define ADMXRC6Tx_V_PWRSTAT_VRBOK       DF_VAL32S(0x1U, 4, 1) /* VRef for DRAM banks 2, 3 OK */
#define ADMXRC6Tx_V_PWRSTAT_BM1V0OK     DF_VAL32S(0x1U, 5, 1) /* Bridge MGT 1.0V supply OK */
#define ADMXRC6Tx_V_PWRSTAT_BM1V2OK     DF_VAL32S(0x1U, 6, 1) /* Bridge MGT 1.2V supply OK */
#define ADMXRC6Tx_V_PWRSTAT_TM1V0OK     DF_VAL32S(0x1U, 7, 1) /* Bridge MGT 1.0V supply OK */
#define ADMXRC6Tx_V_PWRSTAT_TM1V2OK     DF_VAL32S(0x1U, 8, 1) /* Bridge MGT 1.2V supply OK */
#define ADMXRC6Tx_V_PWRSTAT_XMVIOOK     DF_VAL32S(0x1U, 9, 1) /* XRM VI/O supply OK */
#define ADMXRC6Tx_S_PWRSTAT_XVSEL       16 /* XRM VI/O select inputs */
#define ADMXRC6Tx_L_PWRSTAT_XVSEL       3
#define ADMXRC6Tx_M_PWRSTAT_XVSEL       DF_MASK32S(ADMXRC6Tx_S_PWRSTAT_XVSEL, ADMXRC6Tx_M_PWRSTAT_XVSEL)
#define ADMXRC6Tx_V_PWRSTAT_XVEN        DF_VAL32S(0x1U, 19, 1) /* XRM VI/O enable pin asserted */
#define ADMXRC6Tx_V_PWRSTAT_XPRSNT      DF_VAL32S(0x1U, 24, 1) /* XRM is asserting presence pin */
#define ADMXRC6Tx_V_PWRSTAT_XFORCE2V5   DF_VAL32S(0x1U, 25, 1) /* XRM is legacy and requires 2.5V VI/O */
#define ADMXRC6Tx_V_PWRSTAT_XNVVLD      DF_VAL32S(0x1U, 26, 1) /* XRM nominal voltage read from PROM OK */
#define ADMXRC6Tx_V_PWRSTAT_XNVOOB      DF_VAL32S(0x1U, 27, 1) /* XRM nominal voltage read from PROM OK is out of bounds */
#define ADMXRC6Tx_V_PWRSTAT_XERROR      DF_VAL32S(0x1U, 31, 1) /* Error occurred when reading XRM PROM */

/* Fields for ModelRegsAdmxrc6tx::intCtl */
#define ADMXRC6Tx_V_INTCTL_FPENABLE     DF_VAL32S(0x1U, 0, 1)  /* 1 => FPGA interrupt is enabled */
#define ADMXRC6Tx_V_INTCTL_SYSENABLE    DF_VAL32S(0x1U, 2, 1)  /* 1 => system monutor interrupt is enabled */
#define ADMXRC6Tx_V_INTCTL_FPLEVEL      DF_VAL32S(0x1U, 8, 1)  /* 1 => FPGA interrupt is level-sensitive */
#define ADMXRC6Tx_V_INTCTL_SYSLEVEL     DF_VAL32S(0x1U, 10, 1) /* 1 => system monitor interrupt is level-sensitive */

/* Fields for ModelRegsAdmxrc6tx::intStat */
#define ADMXRC6Tx_V_INTSTAT_FP          DF_VAL32S(0x1U, 0, 1)
#define ADMXRC6Tx_V_INTSTAT_SYS         DF_VAL32S(0x1U, 2, 1)

/* Fields for ModelRegsAdmxrc6tx::i2cCtl */
#define ADMXRC6Tx_S_I2CCTL_DISDATA      (0) /* Data from discrete read command */
#define ADMXRC6Tx_L_I2CCTL_DISDATA      (8)
#define ADMXRC6Tx_M_I2CCTL_DISDATA      DF_MASK32S(ADMXRC6Tx_S_I2CCTL_DISDATA, ADMXRC6Tx_L_I2CCTL_DISDATA)
#define ADMXRC6Tx_V_I2CCTL_DISDATA(x)   DF_VAL32S(x, ADMXRC6Tx_S_I2CCTL_DISDATA, ADMXRC6Tx_L_I2CCTL_DISDATA)

#define ADMXRC6Tx_S_I2CCTL_DISADDR      (8) /* Address for discrete command */
#define ADMXRC6Tx_L_I2CCTL_DISADDR      (8)
#define ADMXRC6Tx_M_I2CCTL_DISADDR      DF_MASK32S(ADMXRC6Tx_S_I2CCTL_DISADDR, ADMXRC6Tx_L_I2CCTL_DISADDR)
#define ADMXRC6Tx_V_I2CCTL_DISADDR(x)   DF_VAL32S(x, ADMXRC6Tx_S_I2CCTL_DISADDR, ADMXRC6Tx_L_I2CCTL_DISADDR)

#define ADMXRC6Tx_S_I2CCTL_I2CADDR      (16) /* I2C chip address */
#define ADMXRC6Tx_L_I2CCTL_I2CADDR      (7)
#define ADMXRC6Tx_M_I2CCTL_I2CADDR      DF_MASK32S(ADMXRC6Tx_S_I2CCTL_I2CADDR, ADMXRC6Tx_L_I2CCTL_I2CADDR)
#define ADMXRC6Tx_V_I2CCTL_I2CADDR(x)   DF_VAL32S(x, ADMXRC6Tx_S_I2CCTL_I2CADDR, ADMXRC6Tx_L_I2CCTL_I2CADDR)

/* Values for ADMXRC6Tx_V_I2CCTL_I2CADDR() */
#define ADMXRC6Tx_I2C_SLOT_XRM          (0x50U)

#define ADMXRC6Tx_S_I2CCTL_IFADDR0      (23) /* I2C interface bus number bit 0 */
#define ADMXRC6Tx_L_I2CCTL_IFADDR0      (1)
#define ADMXRC6Tx_M_I2CCTL_IFADDR0      DF_MASK32S(ADMXRC6Tx_S_I2CCTL_IFADDR0, ADMXRC6Tx_L_I2CCTL_IFADDR0)
#define ADMXRC6Tx_V_I2CCTL_IFADDR0(x)   DF_VAL32S(x, ADMXRC6Tx_S_I2CCTL_IFADDR0, ADMXRC6Tx_L_I2CCTL_IFADDR0)

#define ADMXRC6Tx_S_I2CCTL_IFADDR1      (29) /* I2C interface bus number bit 1 */
#define ADMXRC6Tx_L_I2CCTL_IFADDR1      (1)
#define ADMXRC6Tx_M_I2CCTL_IFADDR1      DF_MASK32S(ADMXRC6Tx_S_I2CCTL_IFADDR1, ADMXRC6Tx_L_I2CCTL_IFADDR1)
#define ADMXRC6Tx_V_I2CCTL_IFADDR1(x)   DF_VAL32S(x, ADMXRC6Tx_S_I2CCTL_IFADDR1, ADMXRC6Tx_L_I2CCTL_IFADDR1)

#define ADMXRC6Tx_V_I2CCTL_IFADDR(x)    (ADMXRC6Tx_V_I2CCTL_IFADDR0((x) & 0x1U) | ADMXRC6Tx_V_I2CCTL_IFADDR1(((x) >> 1) & 0x1U))

#define ADMXRC6Tx_S_I2CCTL_CMD          (24) /* Command to interface */
#define ADMXRC6Tx_L_I2CCTL_CMD          (3)
#define ADMXRC6Tx_M_I2CCTL_CMD          DF_MASK32S(ADMXRC6Tx_S_I2CCTL_CMD, ADMXRC6Tx_L_I2CCTL_CMD)
#define ADMXRC6Tx_V_I2CCTL_CMD(x)       DF_VAL32S(x, ADMXRC6Tx_S_I2CCTL_CMD, ADMXRC6Tx_L_I2CCTL_CMD)
#define ADMXRC6Tx_V_I2CCTL_NOP          ADMXRC6Tx_V_I2CCTL_CMD(0x0U) /* No command */
#define ADMXRC6Tx_V_I2CCTL_DISREAD      ADMXRC6Tx_V_I2CCTL_CMD(0x2U) /* Discrete read command */
#define ADMXRC6Tx_V_I2CCTL_DISWRITE     ADMXRC6Tx_V_I2CCTL_CMD(0x3U) /* Discrete write command */
#define ADMXRC6Tx_V_I2CCTL_BLKREAD      ADMXRC6Tx_V_I2CCTL_CMD(0x4U) /* Block read command */
#define ADMXRC6Tx_V_I2CCTL_READY        DF_VAL32S(0x1U, 27, 1) /* I2C interface ready */
#define ADMXRC6Tx_V_I2CCTL_ERROR        DF_VAL32S(0x1U, 28, 1) /* 1 => error during command execution */

#endif
