/*******************************************************************************
 *	TmsTestData.cpp	TMS Test code for a Data Client
 *			T.Barnaby,	BEAM Ltd,	2007-10-02
 *******************************************************************************
 *
 *	This is a simpe test program to fetch data from the TMS system.
 *	This example fetches the Sigma/DeltaX/DeltaY levels from all
 *	pickups.
 */
#include <iostream>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/time.h>
#include <BFile.h>
#include <TmsD.h>
#include <TmsC.h>

#define	DEBUG		0

#if DEBUG
#define	dprintf(fmt, a...)       printf(fmt, ##a);
#else
#define	dprintf(fmt, a...)
#endif

using namespace Tms;
using namespace std;

const char*	cycleType = "SimBeam3";
const char*	testSignalFileName = "beam3-437000-8.psd";
int		quiet = 0;

// Get current time in seconds
double getTime()
{
	struct timeval	tp;
	
	gettimeofday(&tp, NULL);
	return ((double) tp.tv_sec + (double) tp.tv_usec * 1e-6);
}


// Function to set system to use simulated timing and data
BError tmsSimData(TmsControl& tmsControl, TmsProcess& tmsProcess, int simDataAll){
	BError		err;
	UInt32		cn;
	BString		ct;
	ConfigInfo	configInfo;
	unsigned int	n;
	BArray<UInt32>	data;
	BFile		file;
	int		nb;

	// Read the test signal file
	if(err = file.open(testSignalFileName, "r")){
		if(err = file.open(BString("/usr/tms/data/") + testSignalFileName, "r")){
			return err.set(1, BString("Error: Opening file: ") + testSignalFileName);
		}
	}
	data.resize(1024);

	if((nb = file.read(&data[0], 1024 * sizeof(UInt32))) <= 0){
		return err.set(1, BString("Error: Reading 1024 32bit data items from file: ") + testSignalFileName);
	}
	data.resize(nb / sizeof(UInt32));

	dprintf("Get Configuration\n");
	if(err = tmsControl.getConfiguration(configInfo)){
		return err.set(1, BString("Error: getting configuration TMS: ") + err.getString());
	}

	for(n = 0; n < configInfo.puReferences.size(); n++){
		dprintf("Set Test Data Channel: %d\n", n + 1);
		if(err = tmsControl.setTestData(configInfo.puReferences[n], 1, data)){
			printf("Error: Module: %d Pupe: %d Chan: %d %s\n",
				configInfo.puReferences[n].moduleNum,
				configInfo.puReferences[n].pupeNum,
				configInfo.puReferences[n].pupeChan,
				err.getString().retStr());
		}
	}

	return err;
}

void usage(void) {
	cerr << "Usage:\ttmsTestData [options] hostname\n";
	cerr << " -help             - Help on command line parameters\n";
	cerr << " -simdata          - Set the system to use simulated timing signals and test data\n";
	cerr << " -simdataall       - Sets all PUPE boards in the system to use simulated timing signals and test data\n";
	cerr << " -info             - Prints information on the processing cycle\n";
	cerr << " -numSamples <n>   - The number of samples to read\n";
	cerr << " -test <testName>  - The Test to be performed: single - single channel all bunches, all - all bunches all channels\n";
	cerr << "                   - allMean - Mean from all bunches all channels\n";
	cerr << " -check            - Check the data for validity\n";
	cerr << " -cont             - Continuous\n";
	cerr << " -quiet            - Only print errors\n";
	cerr << " -channel <n>      - Channel number\n";
}

static struct option options[] = {
		{ "?",			0, NULL, 0 },
		{ "h",			0, NULL, 0 },
		{ "help",		0, NULL, 0 },
		{ "simdata",		0, NULL, 0 },
		{ "simdataall",		0, NULL, 0 },
		{ "info",		0, NULL, 0 },
		{ "check",		0, NULL, 0 },
		{ "cont",		0, NULL, 0 },
		{ "quiet",		0, NULL, 0 },
		{ "numSamples",		1, NULL, 0 },
		{ "test",		1, NULL, 0 },
		{ "channel",		1, NULL, 0 },
		{ 0,0,0,0 }
};

int main(int argc, char** argv){
	BError			err;
	BString			hostName = "localhost";
	TmsProcess		tmsProcess;
	TmsControl		tmsControl;
	int			optIndex = 0;
	int			c;
	BString			s;
	int			simData = 0;
	int			simDataAll = 0;
	int			info = 0;
	int			check = 0;
	int			cont = 0;
	BString			test;
	int			numSamples = 0;
	int			channel = 1;

	while((c = getopt_long_only(argc, argv, "", options, &optIndex)) == 0){
		s = options[optIndex].name;
		if(s == "help" || s == "h" || s == "?"){
			usage();
			return 1;
		}
		else if(s == "simdata"){
			simData = 1;
		}
		else if(s == "simdataall"){
			simDataAll = 1;
		}
		else if(s == "info"){
			info = 1;
		}
		else if(s == "check"){
			check = 1;
		}
		else if(s == "cont"){
			cont = 1;
		}
		else if(s == "quiet"){
			quiet = 1;
		}
		else if(s == "test"){
			test = optarg;
		}
		else if(s == "numSamples"){
			numSamples = strtol(optarg, 0, 0);
		}
		else if(s == "channel"){
			channel = strtol(optarg, 0, 0);
		}
		else {
			usage();
			return 1;
		}
	}
	
	if(optind == argc){
		usage();
		return 1;
	}

	hostName = argv[optind++];

	// Connect to the Control service
	if(err = tmsControl.connectService(BString("//") + hostName + "/tmsControl1")){
		cerr << "Error: " << err.getString() << "\n";
		return 1;
	}

	// Connect to the Process service
	if(err = tmsProcess.connectService(BString("//") + hostName + "/tmsProcess1")){
		cerr << "Error: " << err.getString() << "\n";
		return 1;
	}

	if(err = tmsSimData(tmsControl, tmsProcess, simDataAll)){	
		cerr << "Error: " << err.getString() << "\n";
		return 1;
	}
	
	return 0;
}
