/*******************************************************************************
 *	TmsDataTestOffsetRaw.cpp	TMS data test to validate operation
 *			T.Barnaby,	BEAM Ltd,	2014-08-07
 *******************************************************************************
 *
 * This test loops once per second fetching cycle information and then fetches
 *  some Raw data with a time offset for this cycle for all cycle periods after injection.
 * The amount of data and the per sample timestamps are validated.
 * On error it will abort with appropriate error message.
 *
 */
#include <iostream>
#include <stdio.h>
#include <unistd.h>
#include <TmsD.h>
#include <TmsC.h>

using namespace Tms;
using namespace std;


// Function to read raw data, testing
BError tmsTest1(TmsProcess& tmsProcess){
	BError			err;
	DataInfo		dataInfo;
	UInt32			cn = 0;
	BString			ct;
	UInt32			n;
	Data			data;
	UInt32			chan;
	CycleInformation	ci;
	BUInt			c;
	BUInt			s;
	BUInt			p;
	BUInt			numValues;
	BUInt			startTime;
		
	// Find out the current cycle number and type
	if(err = tmsProcess.getCycleInfo(cn, ct)){
		return err.set(1, BString("Error: Getting Cycle Number: ") + err.getString());
	}
	
	// Get cycle information. Will await the completion of the cycle capture
	if(err = tmsProcess.getCycleInformation(cn, ci)){
		return err.set(1, BString("Error: Getting Cycle Information: ") + err.getString());
	}

	printf("Cycle information: %u %s\n", ci.cycleNumber, ci.cycleType.retStr());
	for(p = 0; p < ci.periods.number(); p++){
		printf("  Period: %2d  Times: %4d - %4d Harm: %2d  NumBunches: %2d BunchMask: 0x%05x NumValues: %8d\n", ci.periods[p].cyclePeriod, ci.periods[p].startTime,
			ci.periods[p].endTime, ci.periods[p].harmonic, ci.periods[p].numBunches, ci.periods[p].bunchMask, ci.periods[p].numValues);
	}

	// Fid period of interest
	for(p = 0; p < ci.periods.number(); p++){
		if(ci.periods[p].cyclePeriod == CyclePeriodEvent1)
			break;
	}

	// If this cycle does not have the period of interest, ignore
	if(p >= ci.periods.number())
		return err;

	// Start time 10ms before the end
	startTime = (ci.periods[p].endTime - ci.periods[p].startTime) - 10;

	printf("Fetch raw data from all pick-ups for cycle: %u\n", cn);

	// Setup dataInfo
	dataInfo.cycleNumber	= cn;
	dataInfo.channel	= 0;
	dataInfo.cyclePeriod	= CyclePeriodEvent1;
	dataInfo.startTime	= startTime;
	dataInfo.orbitNumber	= 0;
	dataInfo.bunchNumber	= 0;
	dataInfo.function	= DataFunctionRaw;
	dataInfo.argument	= 0;
	dataInfo.numValues	= 1000;
	dataInfo.beyondPeriod	= 0;

	if(err = tmsProcess.getData(dataInfo, data)){
		for(chan = 0; chan < data.errors.size(); chan++){
			printf("Error: Chan: %d ErrorNo: %d ErrorStr: %s\n", chan + 1, data.errors[chan].getErrorNo(), data.errors[chan].getString().retStr());
		}
		return err.set(1, BString("Error: Getting Data: ") + err.getString());
	}

	printf("Raw: Period: %d NumChannels: %d NumSamples: %d StartTimeOffset: %dms\n", dataInfo.cyclePeriod, data.numChannels, data.dataValues.size(), startTime);
	printf("Data Times: %d - %d\n", data.dataValues[0].time, data.dataValues[data.dataValues.size() - 1].time);

	numValues = (dataInfo.numValues / data.numChannels) - 1;
	printf("NumValues per channel expected: per chan: %d total: %d\n", numValues, data.numChannels * numValues);

	// Validate number of samples
	if(data.dataValues.size() != (numValues * data.numChannels)){
		printf("Expected numSamples: %d received numSamples: %d\n", numValues, data.dataValues.size());
		return err.set(1, "Error: number of samples returned incorrect");
	}

#ifdef ZAP
	for(s = 0; s < 8; s++){
		printf("Time: %d\n", data.dataValues[s].time);
	}
#endif

	return err;
}

int main(int argc, char** argv){
	BError			err;
	BString			host = "localhost";
	TmsProcess		tmsProcess;

	if(argc == 2)
		host = argv[1];

	// Connect to the Process service
	if(err = tmsProcess.connectService(BString("//") + host + "/tmsProcess1")){
		cerr << "Error: " << err.getString() << "\n";
		return 1;
	}
	
	while(1){
		// Run a normal data gathering cycle as a normal client would.
		if(err = tmsTest1(tmsProcess)){	
			cerr << "Error: " << err.getString() << "\n";
			return 1;
		}
		printf("\n");
		sleep(1);
	}
		
	return 0;
}
