/*
* Title:	tmsSink.cpp
* Author:	M.Thomas Beam Ltd
* Date:		2007-03-15
* Contents:	Datasinking class for stress testing
*
*
* Reads (current cycle number - 1) data packets from the server until an error is recieved.
* Then it gets the current cycle number and repeats. 
*
*/


#include "tmsSink.h"
#include <BError.h>
#include <errno.h>

using namespace std;

const int	MEGABYTE = 1024 * 1024;


TmsSink::TmsSink(BString host,int id): ohost(host) {
	opacketsRead = 0;
	obytesRead = 0;
	orate = 0;
	obytesPerCycle= 1024 * 1024 * 100;
	oid = id;
	ooverRuns = 0;
};

TmsSink::~TmsSink() {};


void*	TmsSink::function() {
	BError	err;

	while (true) {
		getData();
	}	
}

BError	TmsSink::connect() {
	BError	err;
	
	if(err = oprocess.connectService(BString("//") + ohost + "/tmsProcess")){
		cerr << "Id: (" << oid << ")  Error: cannot connect to tmsProcess: " << err.getString() << "\n";
		return err;
	}
	return err;
}


void	TmsSink::startTimer() {
	otimer.start();
}



TmsSinkStats	TmsSink::getStats() {
	TmsSinkStats	s;
	double		delta;
	
	olock.lock();
	
	delta = otimer.getElapsedTime() - oprior.duration;
	
	s.bytesPerCycle = obytesPerCycle;
	s.bytes = obytesRead;
	s.packets = opacketsRead;
	s.transferRate = (double)(obytesRead/MEGABYTE)/otimer.getElapsedTime();
	s.duration = otimer.getElapsedTime();
	s.dataInfo = odataInfo;
	s.overRuns = ooverRuns;
	
	s.deltaBytes = (obytesRead - oprior.bytes)/delta;
	s.deltaPackets = (opacketsRead - oprior.packets)/delta;
	s.deltaRate = (s.deltaBytes/MEGABYTE);
	s.deltaOverRuns = (ooverRuns - oprior.overRuns)/delta;
		
	oprior = s;
	olock.unlock();
	return s;
}

void	TmsSink::setRate(double bytesPerCycle) {
	obytesPerCycle = bytesPerCycle;
}

void	TmsSink::setupRequest(DataInfo dataInfo) {
	olock.lock();
	odataInfo = dataInfo;
	olock.unlock();
}

BError	TmsSink::getData() {
	BError		err;
	double		bytes = obytesPerCycle;
	double		bytesInPacket = (odataInfo.numValues * 8) + 8;

	setupRequest();
	while (1) {
		if(err = oprocess.getData(odataInfo, odata)){
			BString	s = err.getString();
			err.set(1, BString("Error: Getting Data: ") + err.getString());
			if (s.compareWild("*overw*")) {
					ooverRuns++;
			}
			else {	
				addError(err);
			}	
			break;
		}

		olock.lock();
		opacketsRead++;
		obytesRead += bytesInPacket;
		olock.unlock();
		bytes -= bytesInPacket;


		if ( bytes <= 0) {
			UInt32 		cycleNumber;
			BString		cycleType;
			UInt32		oldcycle = odataInfo.cycleNumber + 1;

			while (1) {
				if(err = oprocess.getCycleInfo(cycleNumber,cycleType)){
					return err;
				}
				if (cycleNumber > oldcycle) {
					return err;
				}	
				else {
					usleep(300000);
				}	
			}			
		}
	sched_yield();		
	}
	return err;
}



BError	TmsSink::setupRequest() {
	BError		err;
	UInt32 		cycleNumber;
	BString		cycleType;
	

	if(err = oprocess.getCycleInfo(cycleNumber,cycleType)){
		return err;
	}
	odataInfo.cycleNumber	= cycleNumber - 1;
	return err;
}


BError	TmsSink::saveToFile() {
	BError	err;
	FILE*	f;
	UInt32	i;

	printf("Data: NumValues: %d\n", odata.numValues);
	printf("Data: Type: %d\n", odata.dataType);
	
	if ((f = fopen("data.txt", "a")) == NULL) {
		BString msg;
	
		msg.printf("Error: Unable to open data file : %s\n",strerror(errno));
		err.set(1,msg);
		return err;
	}	
	for(i = 0; i < odata.numValues; i++){
		fprintf(f, "%d\n", odata.dataValues[i].sigma);
	}
	fclose(f);
	return err;
}

BError	TmsSink::getLastError() {
	BError	err;
	
	olock.lock();
	if (running())
		err =  oerr.queueGet();
	olock.unlock();	
	return err;	
}

void	TmsSink::addError(BError err) {
	olock.lock();
	if (oerr.number() > 10) 
		oerr.queueGet();
	oerr.queueAdd(err);
	olock.unlock();	
}





