/*******************************************************************************
 *	PupeDmaTest2.c	Simple Pupe DMA test program
 *			T.Barnaby,	BEAM Ltd,	2007-10-05
 *******************************************************************************
 *
 * Tests DMA from a board
 */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <admxrc2.h>
#include <pthread.h>

#ifdef linux
#include <unistd.h>
#endif


#define	DEBUG	0

#if DEBUG
#define	dprintf(fmt, a...)       printf(fmt, ##a);
#else
#define	dprintf(fmt, a...)
#endif

char		driver[]	= "/dev/admxrc0";
char		fpgafile[]	= "simple64-xrc4fx-4vfx100.bit";
int		BufferSize	= (1024*1024*2);

ADMXRC2_HANDLE		card;
ADMXRC2_STATUS		status;
ADMXRC2_DMADESC		dmaDesc0;
ADMXRC2_DMADESC		dmaDesc1;
ADMXRC2_CARD_INFO	ocardInfo;
ADMXRC2_SPACE_INFO	spInfo;
volatile uint*		fpga;

char*			recvbuf0;
char*			recvbuf1;

static int memcpy_64_mmx(void* to, const void* from, size_t len){
	size_t		i;

	// Check alignment and size
	if(((unsigned int)to & 0x07) || ((unsigned int)from & 0x07) || (len & 0x07)){
		fprintf(stderr, "memcpy_mmx: Memory pointers and length need to be aligned to 64bit boundary\n");
		return 1;
	}
	
	i = len / 8;
	while(i--){
		__asm__ __volatile__ (
			"movq (%0), %%mm0\n"
			"movq %%mm0, (%1)\n"
			:: "r" (from), "r" (to) : "memory");
		from = ((const unsigned char *)from) + 8;
		to = ((unsigned char *)to) + 8;
	}
	__asm__ __volatile__ ("emms":::"memory");
	return 0;
}

// Get current time in seconds
double getTime()
{
	struct timeval	tp;
	
	gettimeofday(&tp, NULL);
	return ((double) tp.tv_sec + (double) tp.tv_usec * 1e-6);
}

void hd8(void* data, int n){
	unsigned char* d = (unsigned char*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%2.2x ", *d++);
		if((i & 0xF) == 0xF)
			printf("\n");
	}
	printf("\n");
}

void hd32(void* data, int n){
	unsigned int* d = (unsigned int*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%8.8x ", *d++);
		if((i & 0x7) == 0x7)
			printf("\n");
	}
	printf("\n");
}

int dmaRead(int chan){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpgaAddress = 0;

	dprintf("Perform DMA\n");
	if(chan)
		status = ADMXRC2_DoDMA(card, dmaDesc1, 0, BufferSize, fpgaAddress, ADMXRC2_LOCALTOPCI, 0, mode, 0, 0, 0);
	else
		status = ADMXRC2_DoDMA(card, dmaDesc0, 0, BufferSize, fpgaAddress, ADMXRC2_LOCALTOPCI, 0, mode, 0, 0, 0);

	if(status != ADMXRC2_SUCCESS){
		printf("DMA: fail: status: %x\n", status);
	}
	dprintf("DMA complete\n");

	return BufferSize;
}


void testDma(int chan){
	int	i = 0;
	
	dmaRead(chan);
	hd32(recvbuf0, 64);
	
	while(1){
		if((i % 100) == 0)
			printf("%d\n", i);
		dmaRead(chan);
		i++;
	}
}


void testSpeed(int chan){
	int		num = 10;
	int		n;
	double		ts, te, r;

	printf("Test Read data rate: ChunkSize: %d\n", BufferSize);
	ts = getTime();
	for(n = 0; n < num; n++){
		dmaRead(chan);
	}
	te = getTime();
	
	r = (double(num) * BufferSize) / (te - ts);
	printf("DataRate: %f MBytes/s\n", r / (1024*1024));
}

int main(int argc, char *argv[])
{
//	float		clock_freq0 = 40.0e6f;
	float		clock_freq0 = 50.0e6f;
	float		clock_freq1 = 125.0e6f;
	ADMXRC2_STATUS	status;
	int		data;
	int		board = 0;
	pthread_t	thread1;

	recvbuf0 = (char*)ADMXRC2_Malloc(BufferSize);
	recvbuf1 = (char*)ADMXRC2_Malloc(BufferSize);
	memset(recvbuf0, 0x55, BufferSize);
	memset(recvbuf1, 0x55, BufferSize);
	
	if(argc == 2)
		board = atoi(argv[1]);

	printf("Testing Board: %d\n", board);
	status = ADMXRC2_OpenCardByIndex(board, &card);
	if(status != ADMXRC2_SUCCESS) {
		printf("Failed to open driver (%s) Status (%x)\n", driver, status);
		return 1;
	}

	if((status = ADMXRC2_GetCardInfo(card, &ocardInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to get info: %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}

	if((status = ADMXRC2_GetSpaceInfo(card, 0, &spInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to read info %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}
	fpga = (volatile uint*)spInfo.VirtualBase;

	// Set up Clocks
	status = ADMXRC2_SetClockRate(card, 0, clock_freq0, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set LCLK to %.1fMHz\n", clock_freq0 / 1.0e6f);
		return 1;
	}

	status = ADMXRC2_SetClockRate(card, 1, clock_freq1, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set MCLK to %.1fMHz\n", clock_freq1 / 1.0e6f);
		return 1;
	}

	printf("Set Clocks to: Lclk: %.2fMHz MClk: %.2fMHz\n", clock_freq0 / 1.0e6f, clock_freq1 / 1.0e6f);
        printf("Programming FPGA: %s\n", fpgafile);
        if((status = ADMXRC2_ConfigureFromFile(card, fpgafile)) != ADMXRC2_SUCCESS) {
                printf("Unable to load fpga: status(%x)\n", status);
                return 1;
        }

	status = ADMXRC2_SetupDMA(card, recvbuf0, BufferSize, 0, &dmaDesc0);
	if(status != ADMXRC2_SUCCESS){
		printf("Failed to set up receive DMA buffer:\n");
		return 1;
	}

	status = ADMXRC2_SetupDMA(card, recvbuf1, BufferSize, 0, &dmaDesc1);
	if(status != ADMXRC2_SUCCESS){
		printf("Failed to set up receive DMA buffer:\n");
		return 1;
	}

	// Wait a bit
	sleep(1);


	BufferSize	= 1024000;

	// Test DMA Speed
	testSpeed(0);

#ifndef ZAP
	testDma(0);
#endif

#ifdef ZAP	
	// Test two threads reading
	writeTestPattern(0, 0);

	pthread_create(&thread1, 0, testContThread1, 0);
	
	sleep(4);
	testCont(0);
#endif
	// Close the device
	ADMXRC2_CloseCard(card);

	return 0;
}
