/*******************************************************************************
 *	PupeCycleTest3.c	Simple Pupe Cycle Test program
 *				T.Barnaby,	BEAM Ltd,	2007-10-04
 *******************************************************************************
 */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <pthread.h>
#include <sys/time.h>
#include <admxrc2.h>

#ifdef linux
#include <unistd.h>
#endif

// FPGA Control Registers
const int	IMEM_REG	= 0x0000 >> 2;	// Block RAM Index
const int	LOCKED		= 0x0008 >> 2;	// DCM Locked Status
const int	ADDR_REG	= 0x0010 >> 2;	// PAGE Register for SDRAM access
const int	MEM_REG		= 0x0018 >> 2;	// SDRAM Bank Select Register and BlockRam enable
const int	IER		= 0x0020 >> 2;	// Interrupt Enable Register
const int	ISR		= 0x0028 >> 2;	// Interrupt Status Register
const int	MEM_RAS		= 0x0030 >> 2;	// SDRAM Read Address Scaling
const int	MEM_RAO		= 0x0038 >> 2;	// SDRAM Read Address Offset
const int	MEM_GNT0	= 0x0400 >> 2;	// Memory Grant Registers
const int	MEM_GNT1	= 0x0408 >> 2;	// Memory Grant Registers
const int	MEM_GNT2	= 0x0410 >> 2;	// Memory Grant Registers
const int	MEM_GNT3	= 0x0408 >> 2;	// Memory Grant Registers

const int	FIRMWARE	= 0x0800 >> 2;	// Firmware ID code "CN" + version numbers
const int	ADC		= 0x0808 >> 2;	// ADC Control Register
const int	TIMING_IO	= 0x0810 >> 2;	// Timing I/O Register
const int	TESTCTRL	= 0x0818 >> 2;	// Test Data Control Register
const int	TESTLEN		= 0x0820 >> 2;	// Test Data Pattern Length

// Per Pick-Up registers
const int	PU0_REG_BASE	= 0x0880 >> 2;	// PickUp 0 Register Base
const int	PU1_REG_BASE	= 0x0900 >> 2;	// PickUp 1 Register Base
const int	PU2_REG_BASE	= 0x0980 >> 2;	// PickUp 2 Register Base

const int	CONTROL		= 0x0000 >> 2;	// PU General Control and Status register
const int	CYCLE		= 0x0008 >> 2;	// Cycle number
const int	TIME		= 0x0010 >> 2;	// Time in ms from start of cycle
const int	TIME_TBLADDR	= 0x0018 >> 2;	// Last write address in timing table
const int	PLL_FREQUENCY	= 0x0020 >> 2;	// PLL Reference orbit frequency
const int	PLL_FREQDELAY	= 0x0028 >> 2;	// PLL frequency load delay
const int	PLL_PHASEDELAY	= 0x0030 >> 2;	// PLL phase delay
const int	PLL_GAIN	= 0x0038 >> 2;	// PLL gain
const int	DDS_FREQ_MIN	= 0x0040 >> 2;	// PLL DDS minimum frequency
const int	DDS_FREQ_MAX	= 0x0048 >> 2;	// PLL DDS maximum frequency
const int	DIAG_CTRL	= 0x0050 >> 2;	// Diagnostics Control/Status
const int	DIAG_TRIGGER	= 0x0058 >> 2;	// Diagnostics Trigger
const int	TEST		= 0x0060 >> 2;	// Timing Test

// Memory access window
const int	FPGA_DATA_ADDRESS	= 0x200000;	// The FPGA memory
const int	FPGA_MEM		= 0x200000 >> 2;	// The FPGA memory

// SDRAM Data Banks
const int	SDRAM_DATA_PU0		= 0x08;	// PickUp 0 data
const int	SDRAM_DATA_PU1		= 0x09;	// PickUp 0 data
const int	SDRAM_DATA_PU2		= 0x0A;	// PickUp 0 data
const int	SDRAM_DATA_TEST		= 0x0B;	// Test input data
const int	SDRAM_BLOCKRAM		= 0x00;	// Test input data

const int	PU0_BRAM_BASE		= 0;	// PickUp 0 Block Ram ID base
const int	PU1_BRAM_BASE		= 8;	// PickUp 0 Block Ram ID base
const int	PU2_BRAM_BASE		= 16;	// PickUp 0 Block Ram ID base

const int	BRAM_CYCLE_TIMING	= 0;	// Cycle Timing Table
const int	BRAM_CYCLE_INFO		= 1;	// Cycle Information Table
const int	BRAM_PHASE_TABLE	= 2;	// Timing Phase Table
const int	BRAM_SWITCH_TABLE	= 3;	// Timing Switch Table
const int	BRAM_DIAG_TABLE		= 4;	// Diagnostics table
const int	BRAM_BUNCH_MEAN0	= 5;	// Bunch Mean Table #0
const int	BRAM_BUNCH_MEAN1	= 6;	// Bunch Mean Table #1

// Control register bits
const int	CONTROL_INIT				= 0x01;	// Reset
const int	CONTROL_LOOP_CONTROL			= 0x02;	// 
const int	CONTROL_DDS_FREQUENCY_LIMIT_ENABLE	= 0x04;	// 

// Interrupt bits
const int	INT_PU0_CYCLE_START	(1<<0);		// 0 PU #0 CYCLE_START
const int	INT_PU0_CYCLE_STOP	(1<<1);		// 1 PU #0 CYCLE_STOP
const int	INT_PU0_ERROR		(1<<2);		// 2 PU #0 ERROR
const int	INT_PU0_DIAG		(1<<3);		// 3 PU #0 DIAGNOSTIC INFO CAPTURED
const int	INT_PU0_WRITE_FIFO	(1<<4);		// 4 PU #0 SDRAM Write FIFO Half Full
const int	INT_PU0_USER0		(1<<5);		// 5 PU #0 User Interrupts (Switch Table bits 5)
const int	INT_PU0_USER1		(1<<6);		// 6 PU #0 User Interrupts (Switch Table bits 6)
const int	INT_PU0_USER2		(1<<7);		// 7 PU #0 User Interrupts (Switch Table bits 7)

const int	INT_PU1_CYCLE_START	(1<<8);		// 8 PU #1 CYCLE_START
const int	INT_PU1_CYCLE_STOP	(1<<9);		// 9 PU #1 CYCLE_STOP
const int	INT_PU1_ERROR		(1<<10);	// 10 PU #1 ERROR
const int	INT_PU1_DIAG		(1<<11);	// 11 PU #1 DIAGNOSTIC INFO CAPTURED
const int	INT_PU1_WRITE_FIFO	(1<<12);	// 12 PU #1 SDRAM Write FIFO Half Full
const int	INT_PU1_USER0		(1<<13);	// 13 PU #1 User Interrupts (Switch Table bits 5)
const int	INT_PU1_USER1		(1<<14);	// 14 PU #1 User Interrupts (Switch Table bits 6)
const int	INT_PU1_USER2		(1<<15);	// 15 PU #1 User Interrupts (Switch Table bits 7)

const int	INT_PU2_CYCLE_START	(1<<16);	// 16 PU #2 CYCLE_START
const int	INT_PU2_CYCLE_STOP	(1<<17);	// 17 PU #2 CYCLE_STOP
const int	INT_PU2_ERROR		(1<<18);	// 18 PU #2 ERROR
const int	INT_PU2_DIAG		(1<<19);	// 19 PU #2 DIAGNOSTIC INFO CAPTURED
const int	INT_PU2_WRITE_FIFO	(1<<20);	// 20 PU #2 SDRAM Write FIFO Half Full
const int	INT_PU2_USER0		(1<<21);	// 21 PU #2 User Interrupts (Switch Table bits 5)
const int	INT_PU2_USER1		(1<<22);	// 22 PU #2 User Interrupts (Switch Table bits 6)
const int	INT_PU2_USER2		(1<<23);	// 23 PU #2 User Interrupts (Switch Table bits 7)


enum TimingSig		{
			TimingSigClock=0x01, TimingSigCycleStart=0x02, TimingSigCycleStop=0x04,
			TimingSigCalStart=0x08, TimingSigCalStop=0x10, TimingSigInjection=0x20,
			TimingSigHChange=0x40, TimingSigFRef=0x80
			};

#define	DEBUG	0

#if DEBUG
#define	dprintf(fmt, a...)       printf(fmt, ##a);
#else
#define	dprintf(fmt, a...)
#endif

struct DataValue {
	int16_t		sigma;	///< The Sigma value
	int16_t		deltaX;	///< The DeltaX value
	int16_t		deltaY;	///< The DeltaY value
	uint16_t	time;	///< The Time in ms this sample was processed
};

int		BufferSize	= (1024*1024*2);
int		MemoryArea	= SDRAM_DATA_TEST;

char		driver[] = "/dev/admxrc0";
char		fpgafile[] = "tms-fpga.bit";

ADMXRC2_HANDLE		card;
ADMXRC2_STATUS		status;
ADMXRC2_DMADESC		dmaDesc;
ADMXRC2_CARD_INFO	ocardInfo;
ADMXRC2_SPACE_INFO	spInfo;
volatile uint*		fpga;

char*			recvbuf;

// Get current time in seconds
double getTime()
{
	struct timeval	tp;
	
	gettimeofday(&tp, NULL);
	return ((double) tp.tv_sec + (double) tp.tv_usec * 1e-6);
}


void hd8(void* data, int n){
	unsigned char* d = (unsigned char*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%2.2x ", *d++);
		if((i & 0xF) == 0xF)
			printf("\n");
	}
	printf("\n");
}

void hd32(void* data, int n){
	unsigned int* d = (unsigned int*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%8.8x ", *d++);
		if((i & 0x7) == 0x7)
			printf("\n");
	}
	printf("\n");
}

void hd64(void* data, int n){
	unsigned int* d = (unsigned int*)data;
	int	i = 0;
	
	printf("%8.8x: ", i/2);
	for(i = 0; i < n; i++){
		printf("%8.8x ", *d++);
		if((i & 0x7) == 0x7){
			printf("\n");
			printf("%8.8x: ", (i+1)/2);
		}
	}
	printf("\n");
}

int readBlock(){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[MEM_REG] = MemoryArea;
	fpga[ADDR_REG] = 0;
	fpgaAddress = FPGA_MEM << 2;

	dprintf("Perform DMA\n");	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, BufferSize, fpgaAddress, ADMXRC2_LOCALTOPCI, 0, mode, 0, 0, 0);
	if(status != ADMXRC2_SUCCESS){
		printf("DMA: fail: status: %x\n", status);
	}
	dprintf("DMA complete\n");

	return BufferSize;
}

int writeBlock(){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[MEM_REG] = MemoryArea;
	fpga[ADDR_REG] = 0;
	
	fpgaAddress = FPGA_MEM << 2;

	dprintf("Perform DMA: To %x\n", fpgaAddress);	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, BufferSize, fpgaAddress, ADMXRC2_PCITOLOCAL, 0, mode, 0, 0, 0);
	if(status != ADMXRC2_SUCCESS){
		printf("DMA: fail: status: %x\n", status);
	}
	dprintf("DMA complete\n");

	return BufferSize;
}

void writeTestPattern(){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	for(n = 0; n < BufferSize / sizeof(uint32_t); n++){
		p[n] = n;
	}
	
	writeBlock();
}

void readTestPattern(){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	readBlock();
	
	for(n = 0; n < BufferSize / sizeof(uint32_t); n++){
		if(p[n] != n){
			printf("Error at: %x = %x\n", n, p[n]);
			hd32(recvbuf, 64);
			break;
		}
	}
}

void readMem(){
	readBlock();
	hd32(recvbuf, 32);
}

void testm1(){
	writeTestPattern();
	readTestPattern();

//	readMem();
	printf("\n");
}

int fpgaWriteData64(void* fromAddress, uint32_t fpgaOffset, int nbytes){
	ADMXRC2_STATUS	status;
	uint32_t	dmaMode;
	
	dmaMode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType,
				  ADMXRC2_IOWIDTH_64, 0,
				  ADMXRC2_DMAMODE_USEREADY |
				  ADMXRC2_DMAMODE_USEBTERM |
				  ADMXRC2_DMAMODE_BURSTENABLE);
				  
	memcpy(recvbuf, fromAddress, nbytes);

//	fpga[MEM_GNT0] = 0x00;	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, nbytes,
			FPGA_DATA_ADDRESS + fpgaOffset, ADMXRC2_PCITOLOCAL, 0, dmaMode, 0, 0, 0);
//	fpga[MEM_GNT0] = 0x01;	

	return status;
}

int fpgaReadData64(uint32_t fpgaOffset, void* toAddress, int nbytes){
	ADMXRC2_STATUS	status;
	uint32_t	dmaMode;
	
	dmaMode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType,
				  ADMXRC2_IOWIDTH_64, 0,
				  ADMXRC2_DMAMODE_USEREADY |
				  ADMXRC2_DMAMODE_USEBTERM |
				  ADMXRC2_DMAMODE_BURSTENABLE);
				  
//	fpga[MEM_GNT0] = 0x00;	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, nbytes,
			FPGA_DATA_ADDRESS + fpgaOffset, ADMXRC2_LOCALTOPCI, 0, dmaMode, 0, 0, 0);
//	fpga[MEM_GNT0] = 0x02;	

	memcpy(toAddress, recvbuf, nbytes);

	return status;
}

int fpgaCopyData(int pupeChan, uint32_t address, uint32_t nsamples, DataValue* data){
	int			err;
	uint32_t		pageSize = 2*1024*1024;
	uint32_t		dataTablePage;
	uint32_t		dataTableOffset;
	uint32_t		ncopy;
	uint32_t		addressOut = 0;
	
	dprintf("fpgaCopyData\n");
	switch(pupeChan){
	case 1:	fpga[MEM_REG] = SDRAM_DATA_PU0; break;
	case 2:	fpga[MEM_REG] = SDRAM_DATA_PU1; break;
	case 3:	fpga[MEM_REG] = SDRAM_DATA_PU2; break;
	default:	return 1;
	}
	
	while(nsamples){
		dataTablePage = address >> 21;		// 2MByte Region
		dataTableOffset = address & 0x001FFFFF;	// Offset
		
		ncopy = nsamples * 8;
		if(ncopy > (pageSize - dataTableOffset)){
			ncopy = pageSize - dataTableOffset;
		}
		
		fpga[ADDR_REG] = dataTablePage << 19;
#if DEBUG
		printf("MEM_RAO: %d MEM_RAS: %d\n", fpga[MEM_RAO], fpga[MEM_RAS]);
		printf("ReadData: Chan: %d Address: %x Page: %x Offset: %x NBytes: %d\n", pupeChan, address, dataTablePage, dataTableOffset, ncopy);
#endif

		if(err = fpgaReadData64(dataTableOffset, &data[addressOut], ncopy)){
			fpga[MEM_REG] = SDRAM_BLOCKRAM;
			return err;
		}

		dataTableOffset = 0;
		address += ncopy;
		addressOut += (ncopy / 8);
		nsamples -= (ncopy / 8);
	}
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	dprintf("Pupe::fpgaCopyData: End\n");

	return 0;
}

void pupeLoadTestData(){
	FILE*		file;
	uint32_t	data[1024];
	uint32_t	rdata[1024];
	int		nb;
	int		i;

	printf("Load test data\n");	
	fpga[TESTCTRL] = 0;
	fpga[MEM_RAO] = 0;
	fpga[MEM_RAS] = 1;
	fpga[MEM_REG] = SDRAM_DATA_TEST;
	fpga[ADDR_REG] = 0;

	// Delay needed so that TEST DATA engine has released access to SDRAM
	usleep(1000);

	if(!(file = fopen("beam3-437000-8.psd", "r"))){
		fprintf(stderr, "Cannot open file beam3-437000-8.psd\n");
		exit(1);
	}
	
	if((nb = fread(data, 1, sizeof(data), file)) <= 0){
		fprintf(stderr, "Error: Reading data items from file\n");
		exit(1);
	}
	fpgaWriteData64(data, 0, nb);

#ifndef ZAP
	fpgaReadData64(0, rdata, nb);
	for(i = 0; i < nb/4; i++){
		if(data[i] != rdata[i]){
			printf("TestData Error at: %x (%x != %x)\n", i, rdata[i], data[i]);
			break;
		}
	}
#endif
	fclose(file);
	
	fpga[TESTLEN] = nb / 8;
	fpga[TESTCTRL] = (1 << 25);
	fpga[ADC] = 0x00;
	fpga[PU0_REG_BASE + TEST] = fpga[PU0_REG_BASE + TEST] | 0x8000;
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
}

void pupeLoadStatePhaseTables1(){
	int		n;
	uint32_t	states[] = { 0x0e31ef15, 0x1eee2f1d, 0x2e3eef15, 0x34eeef1d, 0x4eeeef15, 0x00 };
	uint8_t		phase0[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa };
	uint8_t		phase1[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xcb,0xcb,0xcb,0xcb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa };
	uint8_t		phase2[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa };
	uint8_t		phase3[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xcb,0xcb,0xcb,0xcb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb };
	uint8_t		phase4[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xcb,0xcb,0xcb,0xca,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3 };
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	
	printf("Load state/phase tables\n");	
	fpga[regBase + PLL_PHASEDELAY] = 128;
	fpga[regBase + PLL_FREQUENCY]  = 15015205;
	fpga[regBase + PLL_FREQDELAY]  = 0;
	fpga[regBase + PLL_GAIN]       = (9 << 16) | 4096;
	fpga[regBase + DDS_FREQ_MIN]   = 0;
	fpga[regBase + DDS_FREQ_MAX]   = 0;
	
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_SWITCH_TABLE;
	fpgaWriteData64(states, 0, sizeof(states));

	fpga[IMEM_REG] = brBase + BRAM_PHASE_TABLE;
	fpgaWriteData64(phase0, 0 * 512, 512);
	fpgaWriteData64(phase1, 1 * 512, 512);
	fpgaWriteData64(phase2, 2 * 512, 512);
	fpgaWriteData64(phase3, 3 * 512, 512);
	fpgaWriteData64(phase4, 4 * 512, 512);

}

void pupeLoadStatePhaseTables(){
	int		n;
	uint32_t	states[] = { 0x0f31ff15,0x1fff2f1d,0x2f3fff15,0x34ffff1d,0x4fffff23,0x0eeeee00,0x0eeeee00,0x0eeeee00,0x00 };
	uint8_t		phase0[] = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };
	uint8_t		phase1[] = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0x41,0x41,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x01,0x01,0x01,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0x41,0x41,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x01,0x01,0x01,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0x41,0x41,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x01,0x01,0x01,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0xc1,0xc1,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00 };
	uint8_t		phase2[] = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };
	uint8_t		phase3[] = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0x41,0x41,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0x41,0x41,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0x41,0x41,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x05,0x05,0x05,0x05,0xc1,0xc1,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x03,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00 };
	uint8_t		phase4[] = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x0a,0x0a,0x0a,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x0a,0x0a,0x0a,0x08,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0x48,0x48,0x0a,0x0a,0x0a,0x08,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0x48,0x48,0x0a,0x0a,0x0a,0x08,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0x48,0x48,0x0a,0x0a,0x0a,0x08,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0c,0x0c,0xc8,0xc8,0x0a,0x0a,0x0a,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x0a,0x0a,0x0a,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x08,0x0a,0x0a,0x0a,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00 };
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	
	printf("Load state/phase tables\n");	
	fpga[regBase + PLL_PHASEDELAY] = 118;
	fpga[regBase + PLL_FREQUENCY]  = 15015205;
	fpga[regBase + PLL_FREQDELAY]  = 0;
	fpga[regBase + PLL_GAIN]       = (9 << 16) | 4096;
	fpga[regBase + DDS_FREQ_MIN]   = 0;
	fpga[regBase + DDS_FREQ_MAX]   = 0;
	
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_SWITCH_TABLE;
	fpgaWriteData64(states, 0, sizeof(states));

	fpga[IMEM_REG] = brBase + BRAM_PHASE_TABLE;
	fpgaWriteData64(phase0, 0 * 512, 512);
	fpgaWriteData64(phase1, 1 * 512, 512);
	fpgaWriteData64(phase2, 2 * 512, 512);
	fpgaWriteData64(phase3, 3 * 512, 512);
	fpgaWriteData64(phase4, 4 * 512, 512);

}

void pupeDisplayStatePhaseTables(){
	uint32_t	states[8];
	uint8_t		phase[8][512];
	int		i;
	int		p;

	printf("PLL_PHASEDELAY: %d\n", fpga[PU0_REG_BASE + PLL_PHASEDELAY]);
	printf("PLL_FREQUENCY:  %d\n", fpga[PU0_REG_BASE + PLL_FREQUENCY]);
	printf("PLL_FREQDELAY:  %d\n", fpga[PU0_REG_BASE + PLL_FREQDELAY]);
	printf("PLL_GAIN:       %x\n", fpga[PU0_REG_BASE + PLL_GAIN]);
	printf("DDS_FREQ_MIN:   %d\n", fpga[PU0_REG_BASE + DDS_FREQ_MIN]);
	printf("DDS_FREQ_MAX:   %d\n", fpga[PU0_REG_BASE + DDS_FREQ_MAX]);

	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = PU0_BRAM_BASE + BRAM_SWITCH_TABLE;
	fpgaReadData64(0, states,  sizeof(states));
	printf("States: ");
	for(i = 0; i < 8; i++)
		printf("0x%8.8x,", states[i]);
	printf("\n");

	fpga[IMEM_REG] = PU0_BRAM_BASE + BRAM_PHASE_TABLE;
	for(i = 0; i < 8; i++){
		fpgaReadData64(i * 512, phase[i], 512);
		printf("Phase%d: ", i);
		for(p = 0; p < 512; p++)
			printf("0x%2.2x,", phase[i][p]);
		printf("\n");
	}
}

void pupeSetDiagnostics(int trigger = TimingSigInjection){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	uint32_t	v;
	
	printf("Start Diagnostics capture\n");	
	v = (0 << 2);		// Source
	v |= (0 << 4);		// Trigger Store
	v |= (0 << 8);		// Clock Select
	v |= (0 << 16);		// Post trigger delay
	v |= 0x1;		// Enable

//	fpga[regBase + DIAG_TRIGGER]	= TimingSigClock;
//	fpga[regBase + DIAG_TRIGGER]	= TimingSigCycleStart;
//	fpga[regBase + DIAG_TRIGGER]	= TimingSigInjection;
//	fpga[regBase + DIAG_TRIGGER]	= TimingSigCycleStart | TimingSigInjection;
//	fpga[regBase + DIAG_TRIGGER]	= 0xFF;

	fpga[regBase + DIAG_TRIGGER]	= trigger;
	fpga[regBase + DIAG_CTRL]	= v;
	
	printf("Trig: %x\n", fpga[regBase + DIAG_TRIGGER]);
	printf("Diag: %x\n", fpga[regBase + DIAG_CTRL]);
}

int	getBitValue(uint64_t value, int startBit, int nBits, int s){
	int64_t		v;
	uint64_t	bm = (1ULL << nBits) - 1;
	uint64_t	sm = (1ULL << (nBits - 1));
	
	v = (value >> startBit) & bm;
	
	if(s){
		v = -(v & sm) | v;
	}
	
	return v;
}

void pupeDumpDiagnostics(){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	uint64_t	data[1024];
	FILE*		file;
	int		i;
	
	printf("Dump Diagnostics\n");	
#ifdef ZAP
	if(! (file = fopen("diag0.txt", "w"))){
#else
	if(! (file = fopen("/data/diag0.txt", "w"))){
#endif
		fprintf(stderr, "Unable to open diag0.txt\n");
		exit(1);
	}
	
	// Wait for data
	while((fpga[regBase + DIAG_CTRL] & 0x01)){
		printf("Diag: %x\n", fpga[regBase + DIAG_CTRL]);
		usleep(100000);
	}
	
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_DIAG_TABLE;
	fpgaReadData64(0, data, sizeof(data));
	
	// Now dump data
//	fprintf(file, "FrefIn PllMsb PhaseTableMsb Lo1 Sigma Lo2 Gate Blr Mean1 Mean2 RfSelect1 RfSelect2 SelFilter SwitchState DdsFreq\n");
	for(i = 0; i < 1024; i++){
#ifndef ZAP
		fprintf(file, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d\n",
			i,
			getBitValue(data[i], 63, 1, 0),		// FREF
			!getBitValue(data[i], 62, 1, 0),	// PllMsb (FRef Reference)
			!getBitValue(data[i], 61, 1, 0),	// PhaseTableMsb (Local FREF)
			getBitValue(data[i], 60, 1, 0),		// LO1
//			getBitValue(data[i], 57, 3, 1),		// Sigma (3 bits)
			getBitValue(data[i], 32, 12, 1),	// Sigma (12 bits)
			getBitValue(data[i], 56, 1, 0),		// LO2

			getBitValue(data[i], 49, 1, 0),		// Gate
			getBitValue(data[i], 48, 1, 0),		// Blr
			getBitValue(data[i], 50, 1, 0),		// Mean1
			getBitValue(data[i], 51, 1, 0),		// Mean2

			getBitValue(data[i], 52, 1, 0),		// RfSelect1
			getBitValue(data[i], 53, 1, 0),		// RfSelect2
			getBitValue(data[i], 54, 1, 0),		// SelFilter
			getBitValue(data[i], 55, 1, 0),		// InfoStobe

			getBitValue(data[i], 44, 4, 0),		// SwitchState

			getBitValue(data[i], 0, 32, 1)		// DDS_FREQ
			);
#else
		fprintf(file, "%d %d %d %d %d %d %d %d %d\n",
			i,
			getBitValue(data[i], 7, 1, 0),		// FREF
			getBitValue(data[i], 6, 1, 0),		// 
			getBitValue(data[i], 5, 1, 0),		// 
			getBitValue(data[i], 4, 1, 0),		// 
			getBitValue(data[i], 3, 1, 0),		// 
			getBitValue(data[i], 2, 1, 0),		// 
			getBitValue(data[i], 1, 1, 0),		// 
			getBitValue(data[i], 0, 1, 0)		// 
			);
#endif
	}
	fclose(file);
}

void pupeCycle1(){
	printf("Perform Pupe Cycle\n");	
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigCycleStart;	// Cycle Start
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigCycleStart;	// Cycle Start End

	usleep(100000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigInjection;		// Injection
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigInjection;	// Injection End

	usleep(200000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigHChange;		// HChange
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigHChange;		// HChange End

	usleep(700000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigCycleStop;		// Cycle Stop
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigCycleStop;	// Cycle Stop End
	usleep(200000);
}

void pupeCycle(){
	printf("Perform Pupe Cycle\n");	
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigCycleStart;	// Cycle Start
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigCycleStart;	// Cycle Start End

	usleep(100000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigInjection;		// Injection
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigInjection;	// Injection End

	usleep(400000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigHChange;		// HChange
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigHChange;		// HChange End

	usleep(400000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigCycleStop;		// Cycle Stop
	usleep(10000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigCycleStop;	// Cycle Stop End
	usleep(200000);
}

const int	FpgaCycleInfoTableSize		= 512;
const int	FpgaCycleTimingTableSize	= 4096;
const int	FpgaCycleTimingTableMask	= (4096-1);

void pupeGetData(DataValue* data, int numValues){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	uint32_t	ocycleInfoTable[FpgaCycleInfoTableSize];
	uint64_t	ocycleTimingTable[FpgaCycleTimingTableSize];
	uint32_t	cycleTimingAddress = 0;
	uint64_t	cycleTimingEntry;
	uint32_t	cycleDataAddress;
	uint32_t	cycle = 0;
	int		i;

	cycle = fpga[regBase + CYCLE] - 1;
	
	dprintf("PupeGetData: Cycle: %u\n", cycle);
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_CYCLE_INFO;
	fpgaReadData64(0, ocycleInfoTable, sizeof(ocycleInfoTable));

	fpga[IMEM_REG] = brBase + BRAM_CYCLE_TIMING;
	fpgaReadData64(0, ocycleTimingTable, sizeof(ocycleTimingTable));

	cycleTimingAddress = ocycleInfoTable[(((cycle & 0x03)) << 4) + 1] & FpgaCycleTimingTableMask;	// Hard coded Injection entry

	// Get Cycle Data Table Address
	cycleTimingEntry = ocycleTimingTable[cycleTimingAddress];
	cycleDataAddress = cycleTimingEntry >> 32;

#if DEBUG
	printf("CycleInfoTable:\n");
	hd32(ocycleInfoTable, 4 * 16);

	printf("CycleTimingAddress: %u(0x%x) CycleTime: %u CycleDataAddress: %u(0x%x)\n", cycleTimingAddress, cycleTimingAddress, uint32_t(cycleTimingEntry & 0xFFFFFFFF), cycleDataAddress, cycleDataAddress);
	printf("CycleTimingTable:\n");
	hd64(&ocycleTimingTable[cycleTimingAddress], 16);
#endif

	// Fetch Data
	fpga[MEM_REG] = SDRAM_DATA_PU0;
	fpga[ADDR_REG] = 0x00;
	fpga[MEM_RAO] = 0;
	fpga[MEM_RAS] = 1;
	fpgaCopyData(1, cycleDataAddress * 8, numValues, data);

#if DEBUG
	printf("CycleDataTable:\n");
	hd64(data, 64);
	
	for(i = 0; i < 16; i++){
		printf("Data: %d,%d,%d,%d\n", data[i].sigma, data[i].deltaX, data[i].deltaY, data[i].time);
	}
#endif
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
}

void pupeDataView(){
	DataValue	data[1024];
	int		i;

	printf("PupeDataView\n");
	pupeGetData(data, 1024);
	
	printf("CycleDataTable:\n");
	hd64(data, 64);
	
	for(i = 0; i < 16; i++){
		printf("Data: %d,%d,%d,%d\n", data[i].sigma, data[i].deltaX, data[i].deltaY, data[i].time);
	}
}

void pupeDisplayInfoTable(){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	uint32_t	ocycleInfoTable[FpgaCycleInfoTableSize];
	int		i;
	int		c;
	uint32_t*	p;

	// Check entries for this cycle
	c = fpga[regBase + CYCLE] - 1;

	printf("View Data: Cycle: %d\n", c);	
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_CYCLE_INFO;
	fpgaReadData64(0, ocycleInfoTable, sizeof(ocycleInfoTable));

	printf("CycleInfoTable:\n");
	hd32(ocycleInfoTable, 4 * 16);
	
	if(c != 0){
		// Check entries for this cycle
		p = &ocycleInfoTable[(c & 0x03) << 4];
		if(((p[0] >> 24) != 0x01) || ((p[1] >> 24) != 0x10) || ((p[2] >> 24) != 0x20) || ((p[3] >> 24) != 0x02)){
			printf("Error in cycle: %d\n", c);
			exit(1);
		}
	}
}

void* cycleThreadFunc(void* arg){
	while(1){
		pupeCycle();
	}
}



void testCycleInfo(){
	int	i;

	for(i = 0; i < 20; i++){
//		pupeSetDiagnostics(TimingSigHChange);
		pupeSetDiagnostics(TimingSigInjection);

		pupeCycle();
		printf("Control: %x Cycle: %u Time: %u\n", (fpga[PU0_REG_BASE + CONTROL] >> 4) & 0x0F, fpga[PU0_REG_BASE + CYCLE], fpga[PU0_REG_BASE + TIME]);
		pupeDumpDiagnostics();
		pupeDisplayInfoTable();
	}
}

void dumpData(DataValue* data, int numValues, int dump){
	FILE*	file;
	int	n;
	char	name[1024];
	
	sprintf(name, "/data/data%d.txt", dump);
	if(! (file = fopen(name, "w"))){
		fprintf(stderr, "Unable to open: %s\n", name);
		exit(1);
	}

	for(n = 0; n < numValues; n++){
		fprintf(file, "%d %d %d\n", n, data[n].sigma, data[n].time);
	}
	
	fclose(file);
}


int dataCheck(DataValue* data, uint32_t numSamples, uint32_t numBunches){
	uint32_t	c = 1;
	uint32_t	s;
	uint32_t	b;
	uint32_t	bunchValues[] = {8651, 7151, 6070, 4937};
	uint32_t 	timeStart = 115;
	uint32_t	o;
	uint32_t	t;
	int		numErrorSigma = 0;
	int		numErrorTime = 0;
	int		errorSample = -1;
	static int	dump = 0;

	numSamples = numSamples / numBunches;

	dprintf("CheckData: NumBunches: %d NumSamples: %d\n", numBunches, numSamples);

	for(s = 0; s < numSamples; s++){
		for(b = 0; b < numBunches; b++){
			o = (s * numBunches) + b;
			t = timeStart + (s / 437);
//			printf("%d,%d,%d: %d %d\n", c, b, s, data[o].sigma, data[o].time);
			if((data[o].sigma > (bunchValues[b] * 1.1)) || data[o].sigma < (bunchValues[b] * 0.9)){
				dprintf("Sigma out of range: (%d.%d.%d) %d should be: %d\n", c, s, b, data[o].sigma, bunchValues[b]);
				numErrorSigma++;
				if(errorSample == -1)
					errorSample = o;
			}
			if((data[o].time > (t * 1.1)) || (data[o].time < (t * 0.9))){
				dprintf("Time out of range: (%d.%d.%d) %u should be: %u\n", c, s, b, data[o].time, t);
				numErrorTime++;
				if(errorSample == -1)
					errorSample = o;
			}
		}
	}

	if(numErrorSigma || numErrorTime){
		printf("DataErrors: Sigma: %u Time: %u Sample: %d\n", numErrorSigma, numErrorTime, errorSample);
		dumpData(data, numSamples*numBunches, dump++);
		return 1;
	}
	
	return 0;
}

int pupeDataReadCheck(){
	int		numSamples = 10240;
	DataValue	data[numSamples];

	pupeGetData(data, numSamples);
	return dataCheck(data, numSamples, 4);
}

void testDataAccess(){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	DataValue	data[102400];
	int		i;
	double		t1,t2;

	printf("View Data\n");	

	for(i = 0; i < 100; i++){
		// Fetch Data (Only page 0)
		fpga[MEM_REG] = SDRAM_DATA_PU0;
		fpga[ADDR_REG] = 0x00;
		fpga[MEM_RAO] = 0;
		fpga[MEM_RAS] = 1;
		t1 = getTime();
		fpgaReadData64(0, data, sizeof(data));
		t2 = getTime();
		fpga[MEM_REG] = SDRAM_BLOCKRAM;
		
		dataCheck(data, 102400/4, 4);
#ifdef ZAP		
		printf("Time: %f\n", t2 - t1);
		usleep(10000);
#endif
	}
}

int main(int argc, char *argv[])
{
//	float		clock_freq0 = 40.0e6f;
	float		clock_freq0 = 50.0e6f;
	float		clock_freq1 = 125.0e6f;
	ADMXRC2_STATUS	status;
	int		data;
	int		c;
	pthread_t	cycleThread;

	recvbuf = (char*)ADMXRC2_Malloc(BufferSize);
	memset(recvbuf, 0x55, BufferSize); 

	status = ADMXRC2_OpenCardByIndex(0, &card);
	if(status != ADMXRC2_SUCCESS) {
		printf("Failed to open driver (%s) Status (%x)\n", driver, status);
		return 1;
	}

	if((status = ADMXRC2_GetCardInfo(card, &ocardInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to get info: %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}

	if((status = ADMXRC2_GetSpaceInfo(card, 0, &spInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to read info %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}
	fpga = (volatile uint*)spInfo.VirtualBase;

	// Set up Clocks
	status = ADMXRC2_SetClockRate(card, 0, clock_freq0, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set LCLK to %.1fMHz\n", clock_freq0 / 1.0e6f);
		return 1;
	}

	status = ADMXRC2_SetClockRate(card, 1, clock_freq1, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set MCLK to %.1fMHz\n", clock_freq1 / 1.0e6f);
		return 1;
	}
	printf("Set Clocks to: Lclk: %.2fMHz MClk: %.2fMHz\n", clock_freq0 / 1.0e6f, clock_freq1 / 1.0e6f);

#ifndef ZAP
        printf("Programming FPGA: %s\n", fpgafile);
        if((status = ADMXRC2_ConfigureFromFile(card, fpgafile)) != ADMXRC2_SUCCESS) {
                printf("Unable to load fpga: status(%x)\n", status);
                return 1;
        }
#endif

	status = ADMXRC2_SetupDMA(card, recvbuf, BufferSize, 0, &dmaDesc);
	if(status != ADMXRC2_SUCCESS){
		printf("Failed to set up receive DMA buffer:\n");
		return 1;
	}

	// Wait a bit
	sleep(1);

#ifdef ZAP
	// Re-sync master clock
	printf("Reseting ADC Clock\n");
	fpga[0x90 >> 2] = 0x001;
	usleep(10000);
	fpga[0x90 >> 2] = 0x000;
	sleep(1);
#endif

	// Reset FPGA
#ifdef ZAP
	printf("Resetting Card\n");
	data = 1;
	ADMXRC2_Write(card, ADMXRC2_IOWIDTH_32, ADMXRC2_IOADAPTER, 0x28, &data, 4);
	usleep(10000);
	data = 0;
	ADMXRC2_Write(card, ADMXRC2_IOWIDTH_32, ADMXRC2_IOADAPTER, 0x28, &data, 4);
	sleep(1);
#endif

#ifndef ZAP
	uint32_t	ocycleInfoTable[FpgaCycleInfoTableSize];

	printf("Clean CycleInfoTable\n");
	memset(ocycleInfoTable, 0, sizeof(ocycleInfoTable));
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = PU0_BRAM_BASE + BRAM_CYCLE_INFO;
	fpgaWriteData64(ocycleInfoTable, 0, sizeof(ocycleInfoTable));
#endif

	// Setup registers
	fpga[ADC] = 0x00;
	fpga[TIMING_IO] = 0xFF00 | (1 << 29);
	fpga[PU0_REG_BASE + CYCLE] = 0;

#ifndef ZAP
	// Initialise PLL
	fpga[PU0_REG_BASE + CONTROL] = CONTROL_INIT;
	usleep(10000);
	fpga[PU0_REG_BASE + CONTROL] = 0x00;
	usleep(10000);
#endif

	// Setup registers
	fpga[ADC] = 0x00;
	fpga[TIMING_IO] = 0xFF00 | (1 << 29);
	fpga[PU0_REG_BASE + CYCLE] = 0;

	printf("FPGA Firmware: %x\n", fpga[FIRMWARE]);
	printf("Control: %x Cycle: %u Time: %u\n", (fpga[PU0_REG_BASE + CONTROL] >> 4) & 0x0F, fpga[PU0_REG_BASE + CYCLE], fpga[PU0_REG_BASE + TIME]);
	printf("TIMINGIO: %x\n", fpga[TIMING_IO]);
	printf("MEM_GNT0: %x\n", fpga[MEM_GNT0]);

	pupeLoadTestData();
	pupeLoadStatePhaseTables();
	sleep(1);

#ifndef ZAP
	// Perform some cycles and fetch data
	pupeCycle();
	pupeCycle();
	pthread_create(&cycleThread, 0, cycleThreadFunc, 0);

	while(1){
		if(pupeDataReadCheck())
			exit(1);
		usleep(100000);
	}
#endif
	
#ifdef ZAP
	// Test Data access
	pthread_create(&cycleThread, 0, cycleThreadFunc, 0);
	testDataAccess();
#endif

#ifdef ZAP
	// Test Cycle Info
	testCycleInfo();
#endif

#ifdef ZAP
	pupeCycle();
	pupeCycle();
	pupeCycle();
	pupeCycle();
	pupeCycle();
	pthread_create(&cycleThread, 0, cycleThreadFunc, 0);
	while(1){
		testDataAccess();
	}
#endif

//	pupeSetDiagnostics();
//	pupeDumpDiagnostics();
//	pupeDataView();
	
	// Close the device
	ADMXRC2_CloseCard(card);

	return 0;
}
