/*******************************************************************************
 *	FpgaAccess.c	Fpga access test program
 *			T.Barnaby,	BEAM Ltd,	2007-05-01
 *******************************************************************************
 */
#include <stdlib.h>
#include <memory.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/time.h>
#include <admxrc2.h>
#include <pthread.h>

// FPGA Control Registers
const int	IMEM_REG	= 0x0000 >> 2;	// Block RAM Index
const int	LOCKED		= 0x0008 >> 2;	// DCM Locked Status
const int	ADDR_REG	= 0x0010 >> 2;	// PAGE Register for SDRAM access
const int	MEM_REG		= 0x0018 >> 2;	// SDRAM Bank Select Register and BlockRam enable
const int	IER		= 0x0020 >> 2;	// Interrupt Enable Register
const int	ISR		= 0x0028 >> 2;	// Interrupt Status Register
const int	MEM_RAS		= 0x0030 >> 2;	// SDRAM Read Address Scaling
const int	MEM_RAO		= 0x0038 >> 2;	// SDRAM Read Address Offset
const int	MEM_GNT0	= 0x0400 >> 2;	// Memory Grant Registers
const int	MEM_GNT1	= 0x0408 >> 2;	// Memory Grant Registers
const int	MEM_GNT2	= 0x0410 >> 2;	// Memory Grant Registers
const int	MEM_GNT3	= 0x0408 >> 2;	// Memory Grant Registers

const int	FIRMWARE	= 0x0800 >> 2;	// Firmware ID code "CN" + version numbers
const int	ADC		= 0x0808 >> 2;	// ADC Control Register
const int	TIMING_IO	= 0x0810 >> 2;	// Timing I/O Register
const int	TESTCTRL	= 0x0818 >> 2;	// Test Data Control Register
const int	TESTLEN		= 0x0820 >> 2;	// Test Data Pattern Length

// Per Pick-Up registers
const int	PU0_REG_BASE	= 0x0880 >> 2;	// PickUp 0 Register Base
const int	PU1_REG_BASE	= 0x0900 >> 2;	// PickUp 1 Register Base
const int	PU2_REG_BASE	= 0x0980 >> 2;	// PickUp 2 Register Base

const int	CONTROL		= 0x0000 >> 2;	// PU General Control and Status register
const int	CYCLE		= 0x0008 >> 2;	// Cycle number
const int	TIME		= 0x0010 >> 2;	// Time in ms from start of cycle
const int	TIME_TBLADDR	= 0x0018 >> 2;	// Last write address in timing table
const int	PLL_FREQUENCY	= 0x0020 >> 2;	// PLL Reference orbit frequency
const int	PLL_FREQDELAY	= 0x0028 >> 2;	// PLL frequency load delay
const int	PLL_PHASEDELAY	= 0x0030 >> 2;	// PLL phase delay
const int	PLL_GAIN	= 0x0038 >> 2;	// PLL gain
const int	DDS_FREQ_MIN	= 0x0040 >> 2;	// PLL DDS minimum frequency
const int	DDS_FREQ_MAX	= 0x0048 >> 2;	// PLL DDS maximum frequency
const int	DIAG_CTRL	= 0x0050 >> 2;	// Diagnostics Control/Status
const int	DIAG_TRIGGER	= 0x0058 >> 2;	// Diagnostics Trigger
const int	TEST		= 0x0060 >> 2;	// Timing Test

// Memory access window
const int	FPGA_MEM	= 0x200000 >> 2;	// The FPGA memory

// SDRAM Data Banks
const int	SDRAM_DATA_PU0		= 0x08;	// PickUp 0 data
const int	SDRAM_DATA_PU1		= 0x09;	// PickUp 0 data
const int	SDRAM_DATA_PU2		= 0x0A;	// PickUp 0 data
const int	SDRAM_DATA_TEST		= 0x0B;	// Test input data
const int	SDRAM_BLOCKRAM		= 0x00;	// Test input data

const int	PU0_BRAM_BASE		= 0;	// PickUp 0 Block Ram ID base
const int	PU1_BRAM_BASE		= 8;	// PickUp 0 Block Ram ID base
const int	PU2_BRAM_BASE		= 16;	// PickUp 0 Block Ram ID base

const int	BRAM_CYCLE_TIMING	= 0;	// Cycle Timing Table
const int	BRAM_CYCLE_INFO		= 1;	// Cycle Information Table
const int	BRAM_PHASE_TABLE	= 2;	// Timing Phase Table
const int	BRAM_SWITCH_TABLE	= 3;	// Timing Switch Table
const int	BRAM_DIAG_TABLE		= 4;	// Diagnostics table
const int	BRAM_BUNCH_MEAN0	= 5;	// Bunch Mean Table #0
const int	BRAM_BUNCH_MEAN1	= 6;	// Bunch Mean Table #1

// Control register bits
const int	CONTROL_INIT				= 0x01;	// Reset
const int	CONTROL_LOOP_CONTROL			= 0x02;	// 
const int	CONTROL_DDS_FREQUENCY_LIMIT_ENABLE	= 0x04;	// 

// Interrupt bits
const int	INT_PU0_CYCLE_START	(1<<0);		// 0 PU #0 CYCLE_START
const int	INT_PU0_CYCLE_STOP	(1<<1);		// 1 PU #0 CYCLE_STOP
const int	INT_PU0_ERROR		(1<<2);		// 2 PU #0 ERROR
const int	INT_PU0_DIAG		(1<<3);		// 3 PU #0 DIAGNOSTIC INFO CAPTURED
const int	INT_PU0_WRITE_FIFO	(1<<4);		// 4 PU #0 SDRAM Write FIFO Half Full
const int	INT_PU0_USER0		(1<<5);		// 5 PU #0 User Interrupts (Switch Table bits 5)
const int	INT_PU0_USER1		(1<<6);		// 6 PU #0 User Interrupts (Switch Table bits 6)
const int	INT_PU0_USER2		(1<<7);		// 7 PU #0 User Interrupts (Switch Table bits 7)

const int	INT_PU1_CYCLE_START	(1<<8);		// 8 PU #1 CYCLE_START
const int	INT_PU1_CYCLE_STOP	(1<<9);		// 9 PU #1 CYCLE_STOP
const int	INT_PU1_ERROR		(1<<10);	// 10 PU #1 ERROR
const int	INT_PU1_DIAG		(1<<11);	// 11 PU #1 DIAGNOSTIC INFO CAPTURED
const int	INT_PU1_WRITE_FIFO	(1<<12);	// 12 PU #1 SDRAM Write FIFO Half Full
const int	INT_PU1_USER0		(1<<13);	// 13 PU #1 User Interrupts (Switch Table bits 5)
const int	INT_PU1_USER1		(1<<14);	// 14 PU #1 User Interrupts (Switch Table bits 6)
const int	INT_PU1_USER2		(1<<15);	// 15 PU #1 User Interrupts (Switch Table bits 7)

const int	INT_PU2_CYCLE_START	(1<<16);	// 16 PU #2 CYCLE_START
const int	INT_PU2_CYCLE_STOP	(1<<17);	// 17 PU #2 CYCLE_STOP
const int	INT_PU2_ERROR		(1<<18);	// 18 PU #2 ERROR
const int	INT_PU2_DIAG		(1<<19);	// 19 PU #2 DIAGNOSTIC INFO CAPTURED
const int	INT_PU2_WRITE_FIFO	(1<<20);	// 20 PU #2 SDRAM Write FIFO Half Full
const int	INT_PU2_USER0		(1<<21);	// 21 PU #2 User Interrupts (Switch Table bits 5)
const int	INT_PU2_USER1		(1<<22);	// 22 PU #2 User Interrupts (Switch Table bits 6)
const int	INT_PU2_USER2		(1<<23);	// 23 PU #2 User Interrupts (Switch Table bits 7)


enum TimingSig		{
			TimingSigClock=0x01, TimingSigCycleStart=0x02, TimingSigCycleStop=0x04,
			TimingSigCalStart=0x08, TimingSigCalStop=0x10, TimingSigInjection=0x20,
			TimingSigHChange=0x40, TimingSigFRef=0x80
			};

#define	DEBUG	0

#if DEBUG
#define	dprintf(fmt, a...)       printf(fmt, ##a);
#else
#define	dprintf(fmt, a...)
#endif

void test2a();

int		BufferSize	= (1024*1024*2);
int		MemoryArea	= SDRAM_DATA_TEST;

char		driver[] = "/dev/admxrc0";
// char		fpgafile[] = "/usr/tms/fpga/tms-fpga.bit";
char		fpgafile[] = "tms-fpga.bit";

ADMXRC2_HANDLE		card;
ADMXRC2_STATUS		status;
ADMXRC2_DMADESC		dmaDesc;
ADMXRC2_CARD_INFO	ocardInfo;
ADMXRC2_SPACE_INFO	spInfo;
volatile uint*		fpga;

char*			recvbuf;

// Get current time in seconds
double getTime()
{
	struct timeval	tp;
	
	gettimeofday(&tp, NULL);
	return ((double) tp.tv_sec + (double) tp.tv_usec * 1e-6);
}

void hd8(void* data, int n){
	unsigned char* d = (unsigned char*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%2.2x ", *d++);
		if((i & 0xF) == 0xF)
			printf("\n");
	}
	printf("\n");
}

void hd32(void* data, int n){
	unsigned int* d = (unsigned int*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%8.8x ", *d++);
		if((i & 0x7) == 0x7)
			printf("\n");
	}
	printf("\n");
}
/// 64bit memory copy function for PUPE board 64bit data access
static inline int memcpy_64(void* to, const void* from, size_t len){
	size_t		i;

	// Check alignment and size
	if(((unsigned int)to & 0x07) || ((unsigned int)from & 0x07) || (len & 0x07)){
		fprintf(stderr, "memcpy_mmx: Memory pointers and length need to be aligned to 64bit boundary\n");
		return 1;
	}
	
	i = len / 8;
	while(i--){
		__asm__ __volatile__ (
			"movq (%0), %%mm0\n"
			"movq %%mm0, (%1)\n"
			:: "r" (from), "r" (to) : "memory");
		from = ((const unsigned char *)from) + 8;
		to = ((unsigned char *)to) + 8;
	}
	__asm__ __volatile__ ("emms":::"memory");
	return 0;
}


int readBlock(int useDma){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[MEM_REG] = MemoryArea;
	fpga[ADDR_REG] = 0;
	fpgaAddress = FPGA_MEM << 2;

	if(useDma){
		dprintf("Perform DMA\n");	
		status = ADMXRC2_DoDMA(card, dmaDesc, 0, BufferSize, fpgaAddress, ADMXRC2_LOCALTOPCI, 0, mode, 0, 0, 0);
		if(status != ADMXRC2_SUCCESS){
			printf("DMA: fail: status: %x\n", status);
		}
		dprintf("DMA complete\n");
	}
	else {
		memcpy_64(recvbuf, (void*)&fpga[FPGA_MEM], BufferSize);
//		memcpy(recvbuf, (void*)&fpga[FPGA_MEM], BufferSize);
	}

	return BufferSize;
}

int writeBlock(int useDma){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[MEM_REG] = MemoryArea;
	fpga[ADDR_REG] = 0;
	
	fpgaAddress = FPGA_MEM << 2;

	if(useDma){
		dprintf("Perform DMA\n");	
		status = ADMXRC2_DoDMA(card, dmaDesc, 0, BufferSize, fpgaAddress, ADMXRC2_PCITOLOCAL, 0, mode, 0, 0, 0);
		if(status != ADMXRC2_SUCCESS){
			printf("DMA: fail: status: %x\n", status);
		}
		dprintf("DMA complete\n");
	}
	else {
		memcpy_64((void*)&fpga[FPGA_MEM], recvbuf, BufferSize);
//		memcpy((void*)&fpga[FPGA_MEM], recvbuf, BufferSize);
	}

	return BufferSize;
}

void writeTestPattern(int useDma){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	for(n = 0; n < BufferSize / sizeof(uint32_t); n++){
		p[n] = n;
	}
	
	writeBlock(useDma);
}

void readTestPattern(int useDma){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	readBlock(useDma);
	
	for(n = 0; n < BufferSize / sizeof(uint32_t); n++){
		if(p[n] != n){
			printf("Error at: %x = %x\n", n, p[n]);
			hd32(recvbuf, 64);
			break;
		}
	}
}

void readMem(int useDma){
	readBlock(useDma);
	hd32(recvbuf, 32);
}

void testm1(){
	printf("BoardFirmware:	%x\n", fpga[FIRMWARE]);
}

void testm2(){
	uint64_t	v;
	
	fpga[MEM_REG] = MemoryArea;
	fpga[ADDR_REG] = 0;

#ifdef ZAP
	printf("Mem: %x\n", fpga[FPGA_MEM]);
	fpga[FPGA_MEM] = 0x11223300;
	fpga[FPGA_MEM + 1] = 0x11223301;
	printf("Mem: %x\n", fpga[FPGA_MEM]);
	printf("Mem: %x\n", fpga[FPGA_MEM+1]);
#endif

#ifdef ZAP
	v = 0x1122334455667788ULL;
	memcpy_64((void*)&fpga[FPGA_MEM], &v, sizeof(v));

	memcpy_64(&v, (void*)&fpga[FPGA_MEM], sizeof(v));
	
	printf("Value: %llx\n", v);
#endif	

#ifdef ZAP
	uint64_t*	pv = (uint64_t*)recvbuf;

	BufferSize = 64*1024;
	*pv = 0x1122334455667788ULL;
	writeBlock(1);
	readBlock(1);
	printf("Value: %llx\n", *pv);
#endif

//	BufferSize = 32 * 1024;
}

void testm3(){
	printf("MemoryTest: %d Bytes\n", BufferSize);

//	BufferSize = 32 * 1024;

//	printf("Read Memory First\n");
//	readMem(1);
	
	printf("DmaWrite\n");
	writeTestPattern(1);

	printf("DmaRead\n");
	readTestPattern(1);

//	readMem(1);
	printf("\n");

#ifdef ZAP	
	printf("CpuRead\n");
	readTestPattern(0);
	hd32(recvbuf, 16);
	printf("DmaRead\n");
	readTestPattern(1);
	printf("MemoryTest: End\n");
	
	{
		int	n;
		
		for(n = 0; n < 16; n++){
			printf("%x = %x\n", n, fpga[FPGA_MEM + n]);
		}
	}
#endif
}

void testm4(){
	int		num = 100;
	int		n;
	double		ts, te, r;

	printf("Test Read data rate: ChunkSize: %d\n", BufferSize);
	ts = getTime();
	for(n = 0; n < num; n++){
		readBlock(1);
	}
	te = getTime();
	
	r = (double(num) * BufferSize) / (te - ts);
	printf("DataRate: DMA: %f MBytes/s\n", r / (1024*1024));

	ts = getTime();
	for(n = 0; n < num; n++){
		readBlock(0);
	}
	te = getTime();
	
	r = (double(num) * BufferSize) / (te - ts);
	printf("DataRate: CPU: %f MBytes/s\n", r / (1024*1024));
}

void testm5(){
	int		num = 100;
	int		n;
	double		ts, te, r;

	printf("Test Write data rate: ChunkSize: %d\n", BufferSize);
	ts = getTime();
	for(n = 0; n < num; n++){
		writeBlock(1);
	}
	te = getTime();
	
	r = (double(num) * BufferSize) / (te - ts);
	printf("WriteDataRate: DMA: %f MBytes/s\n", r / (1024*1024));

	ts = getTime();
	for(n = 0; n < num; n++){
		writeBlock(0);
	}
	te = getTime();
	
	r = (double(num) * BufferSize) / (te - ts);
	printf("WriteDataRate: CPU: %f MBytes/s\n", r / (1024*1024));
}

void testm6(){
	int		num = 100;
	int		n;
	double		ts, te, r;

#ifndef ZAP
	printf("Test Write Latency: ChunkSize: %d\n", BufferSize);
	r = 0;
	for(n = 0; n < num; n++){
		ts = getTime();
		writeBlock(0);
		te = getTime();
		r += (te - ts);
	}
	printf("Latency CPU: %fs\n", r / num);

	r = 0;
	for(n = 0; n < num; n++){
		ts = getTime();
		writeBlock(1);
		te = getTime();
		r += (te - ts);
	}
	printf("Latency DMA: %fs\n", r / num);
#endif
	

#ifndef ZAP
	printf("Test Read Latency: ChunkSize: %d\n", BufferSize);
	r = 0;
	for(n = 0; n < num; n++){
		ts = getTime();
		readBlock(0);
		te = getTime();
		r += (te - ts);
	}
	printf("Latency CPU: %fs\n", r / num);

	r = 0;
	for(n = 0; n < num; n++){
		ts = getTime();
		readBlock(1);
		te = getTime();
		r += (te - ts);
	}
	printf("Latency DMA: %fs\n", r / num);
#endif


}

void testi1(){
	printf("Enable and Trigger an interrupt\n");
	printf("Firmware: %x\n", fpga[FIRMWARE]);
	printf("Interrupt: Mask: %x, Status: %x\n", fpga[IER], fpga[ISR]);

	fpga[IER] = INT_PU0_CYCLE_START | INT_PU0_CYCLE_STOP | INT_PU0_ERROR | INT_PU1_ERROR | INT_PU2_ERROR;
	fpga[IER] = 0xFFFFFFFF;

	fpga[TIMING_IO] = 0xFF00;

	fpga[CONTROL] = 1;
	fpga[CONTROL] = 0;
	printf("Interrupt: Mask: %x, Status: %x\n", fpga[IER], fpga[ISR]);

	sleep(1);
	
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	printf("Test: %x\n", fpga[PU0_REG_BASE + TEST]);

#ifndef ZAP
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	fpga[TIMING_IO] = 0xFF02;
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	usleep(100000);
	fpga[TIMING_IO] = 0xFF00;
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	
#ifdef ZAP
	sleep(2);

	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	fpga[TIMING_IO] = 0xFE02;
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	usleep(100000);
	fpga[TIMING_IO] = 0xFE00;
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
#endif
#else
	printf("Test: %x\n", fpga[PU0_REG_BASE + TEST]);
	fpga[PU0_REG_BASE + TEST] = 0x7E02;
	usleep(100000);
	fpga[PU0_REG_BASE + TEST] = 0x7E00;
	printf("Test: %x\n", fpga[PU0_REG_BASE + TEST]);
#endif

	printf("Interrupt: Mask: %x, Status: %x\n", fpga[IER], fpga[ISR]);
	pause();
}

void testi2(){
	int	ts,te;
	printf("Enable and Trigger an interrupt\n");
	printf("Firmware: %x\n", fpga[FIRMWARE]);
	printf("Interrupt: Mask: %x, Status: %x\n", fpga[IER], fpga[ISR]);

	fpga[IER] = INT_PU0_CYCLE_START | INT_PU0_CYCLE_STOP | INT_PU0_ERROR | INT_PU1_ERROR | INT_PU2_ERROR;
	fpga[IER] = 0xFFFFFFFF;

	fpga[PU0_REG_BASE + CYCLE] = 0;
	fpga[TIMING_IO] = 0x0200;
	fpga[PU0_REG_BASE + TEST] = 0x0000;

	fpga[CONTROL] = 0;
	fpga[CONTROL] = 1;
	fpga[CONTROL] = 0;
	printf("Interrupt: Mask: %x, Status: %x\n", fpga[IER], fpga[ISR]);

	printf("Cycle: %d\n", fpga[PU0_REG_BASE + CYCLE]);
	printf("Ctime: %d\n", fpga[PU0_REG_BASE + TIME]);
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	printf("Test: %x\n", fpga[PU0_REG_BASE + TEST]);

	ts = fpga[PU0_REG_BASE + TIME];
	sleep(1);
	te = fpga[PU0_REG_BASE + TIME];
	printf("CTime for 1s: %d\n\n", te - ts);

	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	fpga[TIMING_IO] = fpga[TIMING_IO] | 0x0002;
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	usleep(100000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & 0xFF00;
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	
	ts = fpga[PU0_REG_BASE + TIME];
	sleep(1);
	te = fpga[PU0_REG_BASE + TIME];
	printf("CTime for 1s: %d\n\n", te - ts);

	sleep(4);
	printf("Cycle: %d\n", fpga[PU0_REG_BASE + CYCLE]);
	printf("Ctime: %d\n", fpga[PU0_REG_BASE + TIME]);

	printf("Interrupt: Mask: %x, Status: %x\n", fpga[IER], fpga[ISR]);
	pause();
}

void testc1(){
	int	cs,ce;
	double	ts,te,r;
	
	printf("Test Clocks\n");
	printf("Firmware: %x\n", fpga[FIRMWARE]);

	fpga[PU0_REG_BASE + CYCLE] = 0;

	fpga[CONTROL] = 0;
	fpga[CONTROL] = 1;
	fpga[CONTROL] = 0;

	fpga[TIMING_IO] = 0x0100;
	fpga[PU0_REG_BASE + TEST] = 0x0000;

	printf("Cycle: %d\n", fpga[PU0_REG_BASE + CYCLE]);
	printf("Ctime: %d\n", fpga[PU0_REG_BASE + TIME]);
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	printf("Test: %x\n", fpga[PU0_REG_BASE + TEST]);

	ts = getTime();
	cs = fpga[PU0_REG_BASE + TIME];
	sleep(1);
	te = getTime();
	ce = fpga[PU0_REG_BASE + TIME];

	r = (ce - cs) / (te - ts);	
	printf("CTime for 1s: Time: %f Count: %d %f\n\n", (te - ts), (ce - cs), r);

	fpga[TIMING_IO] = 0x0000;
	fpga[PU0_REG_BASE + TEST] = 0x0000;

	printf("Cycle: %d\n", fpga[PU0_REG_BASE + CYCLE]);
	printf("Ctime: %d\n", fpga[PU0_REG_BASE + TIME]);
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	printf("Test: %x\n", fpga[PU0_REG_BASE + TEST]);

	ts = getTime();
	cs = fpga[PU0_REG_BASE + TIME];
	sleep(1);
	te = getTime();
	ce = fpga[PU0_REG_BASE + TIME];

	r = (ce - cs) / (te - ts);	
	printf("CTime for 1s: Time: %f Count: %d %f\n\n", (te - ts), (ce - cs), r);

	pause();
}

void testd1(){
	int		ts,te;
	uint32_t	s;
	int		i;
	
	printf("Diagnostics test\n");
	printf("Firmware: %x\n", fpga[FIRMWARE]);

	fpga[PU0_REG_BASE + CYCLE] = 0;
	fpga[TIMING_IO] = 0x0200;
	fpga[PU0_REG_BASE + TEST] = 0x0000;

	fpga[CONTROL] = 0;
	fpga[CONTROL] = 1;
	fpga[CONTROL] = 0;

	printf("Cycle: %d\n", fpga[PU0_REG_BASE + CYCLE]);
	printf("Ctime: %d\n", fpga[PU0_REG_BASE + TIME]);
	printf("TimingIO: %x\n", fpga[TIMING_IO]);
	printf("Test: %x\n", fpga[PU0_REG_BASE + TEST]);

	fpga[PU0_REG_BASE + DIAG_TRIGGER] = 0x02;
	fpga[PU0_REG_BASE + DIAG_CTRL] = 0x11;
	
	for(i = 0; i < 20; i++){
		s = fpga[PU0_REG_BASE + DIAG_CTRL];
		printf("Diag: %x\n", s);
		sleep(1);
		if(i == 2){
			printf("TimingIO: %x\n", fpga[TIMING_IO]);
			fpga[TIMING_IO] = fpga[TIMING_IO] | 0x0002;
			printf("TimingIO: %x\n", fpga[TIMING_IO]);
			usleep(100000);
			fpga[TIMING_IO] = fpga[TIMING_IO] & 0xFF00;
			printf("TimingIO: %x\n", fpga[TIMING_IO]);
		}
	}


	pause();
}

void process1(){
	printf("Firmware: %x\n", fpga[FIRMWARE]);

	// Enable interrupts
	fpga[IER] = INT_PU0_CYCLE_START | INT_PU0_CYCLE_STOP | INT_PU0_ERROR | INT_PU1_ERROR | INT_PU2_ERROR;

	// Initialise
	fpga[CONTROL] = 1;
	fpga[CONTROL] = 0;
	
	
}

void* interruptLoop(void* arg){
	while(1){
		printf("WaitForInterrupt\n");
		ADMXRC2_WaitForInterrupt(card, 0, 0, 0);
		printf("Interrupt: %x !!\n", fpga[ISR]);
		fpga[ISR] = 0xFFFFFFFF;
		fpga[ISR] = 0x0;
		printf("ISR: %x\n", fpga[ISR]);
	}
}

int main(int argc, char *argv[])
{
//	float		clock_freq0 = 50.0e6f;
	float		clock_freq0 = 40.0e6f;
	float		clock_freq1 = 125.0e6f;
//	float		clock_freq1 = 100.0e6f;
	ADMXRC2_STATUS	status;
	pthread_t	ithread;
	
	recvbuf = (char*)ADMXRC2_Malloc(BufferSize);
	memset(recvbuf, 0x55, BufferSize); 

	status = ADMXRC2_OpenCard(0, &card);
	if(status != ADMXRC2_SUCCESS) {
		printf("Failed to open driver (%s) Status (%x)\n", driver, status);
		return 1;
	}

	if((status = ADMXRC2_GetCardInfo(card, &ocardInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to get info: %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}


	if((status = ADMXRC2_GetSpaceInfo(card, 0, &spInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to read info %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}
	fpga = (volatile uint*)spInfo.VirtualBase;

#ifdef ZAP
{
	if((status = ADMXRC2_GetSpaceInfo(card, 1, &spInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to read info %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}

	volatile uint*	p = (volatile uint*)spInfo.VirtualBase;
	printf("Space1: Size: %d %x\n", spInfo.LocalSize, p[3]);
}
#endif

	// Set up Clocks
	status = ADMXRC2_SetClockRate(card, 0, clock_freq0, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set LCLK to %.1fMHz\n", clock_freq0 / 1.0e6f);
		return 1;
	}

	status = ADMXRC2_SetClockRate(card, 1, clock_freq1, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set MCLK to %.1fMHz\n", clock_freq1 / 1.0e6f);
		return 1;
	}

        printf("Programming FPGA: %s\n", fpgafile);
        if((status = ADMXRC2_ConfigureFromFile(card, fpgafile)) != ADMXRC2_SUCCESS) {
                printf("Unable to load fpga: status(%x)\n", status);
                return 1;
        }

	status = ADMXRC2_SetupDMA(card, recvbuf, BufferSize, 0, &dmaDesc);
	if(status != ADMXRC2_SUCCESS){
		printf("Failed to set up receive DMA buffer:\n");
		return 1;
	}

	// Set off interrupt thread
	pthread_create(&ithread, 0, interruptLoop, 0);
	sleep(1);

	BufferSize	= 32 * 1024;
	BufferSize	= 286*4;
	BufferSize	= 2 * 1024;
	MemoryArea	= SDRAM_BLOCKRAM;
	MemoryArea	= SDRAM_DATA_TEST;
	MemoryArea	= SDRAM_DATA_PU0;

	printf("MemoryArea: %x\n", MemoryArea);
#ifdef ZAP
	testc1();
	testm1();
	testm2();
	testm3();
	testm4();
	testm5();
	testm6();

	testi1();
//	testi2();
	
	testd1();
#endif
	testm3();
	testm3();
	testm3();
	testm3();
		
	// Close the device
	ADMXRC2_CloseCard(card);

	return 0;
}
