/*******************************************************************************
 *	BTimeStamp.h	TimeStamp classes
 *			T.Barnaby,	BEAM Ltd,	2008-06-26
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 */
#ifndef BTimeStamp_H
#define BTimeStamp_H	1

#include <stdint.h>
#include <BError.h>

class BTimeStampMs;

class BTimeStamp {
public:
			BTimeStamp();
			BTimeStamp(int year, int month = 1, int day = 1, int hour = 0, int minute = 0, int second = 0, int microsecond = 0);
			BTimeStamp(const BString str);
			~BTimeStamp();
	
	void		clear();				///< Clear the date/time
	void		setFirst();				///< Set the first date available
	void		setLast();				///< Set the last date available

	void		set(time_t time, int microSeconds);	///< Set time using Unix time (seconds from 1970-01-01)
	void		set(int year = 0, int month = 1, int day = 1, int hour = 0, int minute = 0, int second = 0, int microsecond = 0);
	void		set(const BTimeStampMs& timeStamp);		///< Set the timeStamp to given MS time stamp
	void		setYDay(int year = 0, int yday = 0, int hour = 0, int minute = 0, int second = 0, int microsecond = 0);
	void		setTime(int hour = 0, int minute = 0, int second = 0, int microsecond = 0);
	void		setNow();				///< Set the timeStamp to now

	int		year() const;
	int		yday() const;
	int		month() const;
	int		day() const;
	int		hour() const;
	int		minute() const;
	int		second() const;
	int		microSecond() const;
	
	void		getDate(int& year, int& mon, int& day) const;

	BString		getString(BString separator = "T") const;	///< Get the time as an ISO date/time string
	BError		setString(const BString dateTime);		///< Set the time from an ISO date/time
	BString		getStringNoMs(BString separator = "T") const;	///< Get the time as an ISO date/time string without microseconds
	BString		getStringFormatted(BString format) const;	///< Gets the time in a string form as per the format. Format syntax as per strftime()
	
	void		addMilliSeconds(int milliSeconds);	///< Add the given number of milli seconds. This should be less that a year
	void		addMicroSeconds(int64_t microSeconds);	///< Add the given number of micro seconds. This should be less that a year
	void		addSeconds(int seconds);		///< Add the given number of seconds. This should be less that a year
	uint32_t	getYearSeconds() const;			///< Get number of seconds within the year
	uint64_t	getYearMicroSeconds() const;			///< Get number of micro seconds within the year

	int		isSet() const{ return oyear != 0; }
	int		compare(const BTimeStamp& timeStamp) const;	///< Compare two dates

			operator BString() const { return getString(); }
	BTimeStamp&	operator=(const BTimeStampMs& timeStamp){ set(timeStamp); return *this; }

	int		operator==(const BTimeStamp& timeStamp) const { return (compare(timeStamp) == 0); }
	int		operator!=(const BTimeStamp& timeStamp) const { return (compare(timeStamp) != 0); }
	int		operator>(const BTimeStamp& timeStamp) const { return (compare(timeStamp) > 0); }
	int		operator>=(const BTimeStamp& timeStamp) const { return (compare(timeStamp) >= 0); }
	int		operator<(const BTimeStamp& timeStamp) const { return (compare(timeStamp) < 0); }
	int		operator<=(const BTimeStamp& timeStamp) const { return (compare(timeStamp) <= 0); }

	static int	isLeap(int year);
	static BInt64	difference(BTimeStamp t2, BTimeStamp t1);

public:
	uint16_t	oyear;		///< Year (0 .. 65535)
	uint16_t	oyday;		///< Day in year (0 .. 365)
	uint8_t		ohour;		///< Hour (0 .. 23)
	uint8_t		ominute;	///< Minute (0 .. 59)
	uint8_t		osecond;	///< Second (0 .. 59)
	uint8_t		ospare;		///< Padding
	uint32_t	omicroSecond;	///< MicroSecond (0 .. 999999)
};

// String conversion functions
void toBString(BTimeStamp& v, BString& s);
void fromBString(BString& s, BTimeStamp& v);

#endif
