/*******************************************************************************
 *	BSemaphore.cpp	BSemaphore Classes
 *	T.Barnaby,	BEAM Ltd,	2012-11-17
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 */
#include <BSemaphore.h>
#include <sys/time.h>

BSemaphore::BSemaphore(){
	sem_init(&osema, 0, 0);
}

BSemaphore::BSemaphore(const BSemaphore& semaphore){
	sem_init(&osema, 0, semaphore.getValue());
}

BSemaphore& BSemaphore::operator=(const BSemaphore& semaphore){
	sem_destroy(&osema);
	sem_init(&osema, 0, semaphore.getValue());
	return *this;
}

BSemaphore::~BSemaphore(){
	sem_destroy(&osema);
}

void BSemaphore::set(){
	sem_post(&osema);
}

Bool BSemaphore::wait(BTimeout timeoutUs){
	int		ret;
	struct timeval	tv;
	struct timespec	ts;

	if(timeoutUs == BTimeoutForever){
		ret = !sem_wait(&osema);
	}
	else {
		gettimeofday(&tv, 0);
		ts.tv_sec = tv.tv_sec + timeoutUs / 1000000;
		ts.tv_nsec = (tv.tv_usec + timeoutUs % 1000000) * 1000;
		ts.tv_sec += (ts.tv_nsec / 1000000000);
		ts.tv_nsec %=  1000000000;

		ret = !sem_timedwait(&osema, &ts);
	}
	return ret;
}

int BSemaphore::getValue() const {
	int	v;
	
	sem_getvalue((sem_t*)&osema, &v);
	return v;
}


BSemaphoreCount::BSemaphoreCount(){
	ovalue = 0;
}

BSemaphoreCount::BSemaphoreCount(const BSemaphoreCount& semaphore){
	ovalue = semaphore.ovalue;
}

BSemaphoreCount& BSemaphoreCount::operator=(const BSemaphoreCount& semaphore){
	ovalue = semaphore.ovalue;
	return *this;
}

BSemaphoreCount::~BSemaphoreCount(){
}

void BSemaphoreCount::add(int v){
	olock.lock();
	ovalue += v;
	olock.unlock();
	osema.set();
}

Bool BSemaphoreCount::wait(BUInt v, BTimeout timeoutUs){
	while(ovalue < v){
		if(!osema.wait(timeoutUs))
			return 0;
	}
	return 1;
}

Bool BSemaphoreCount::take(BUInt v, BTimeout timeoutUs){
	while(ovalue < v){
		if(!osema.wait(timeoutUs))
			return 0;
	}
	olock.lock();
	ovalue -= v;
	olock.unlock();
	return 1;
}

void BSemaphoreCount::setValue(BUInt v){
	olock.lock();
	ovalue = v;
	olock.unlock();
	osema.set();
}

BUInt BSemaphoreCount::value(){
	return ovalue;
}

