/*******************************************************************************
 *	BFifo.h		BFifo class
 *	T.Barnaby,	Beam Ltd,	2013-05-03
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 *
 * This is the BFifo class. It stores data in a ring buffer.
 * It can store up to (size - 1) items.
 * It has separate read and write pointers and these are loxcked with a BMutex.
 * Note the read and write routines do not bounds check the Fifo. You have to use readAvailable()/writeAvailable()
 * functions to check how much can be read/written before performing the reads or writes.
 *
 */
#ifndef BFifo_h
#define BFifo_h

#include <BTypes.h>
#include <BError.h>
#include <BMutex.h>

template <class Type> class BFifo {
public:
			BFifo(BUInt size);
			~BFifo();

	void		clear();

	BUInt		size();						///< Returns fifo size
	BError		resize(BUInt size);				///< Resize FIFO, clears it as well

	BUInt		writeAvailable();				///< How many items that can be written
	BUInt		writeAvailableChunk();				///< How many items that can be written in a chunk
	BError		write(const Type v);				///< Write a single item
	BError		write(const Type* data, BUInt num);		///< Write a set of items. Can only write a maximum of writeAvailableChunk() to save going beyond end of FIFO buffer

	Type*		writeData();					///< Returns a pointer to the data
	Type*		writeData(BUInt& num);				///< Returns a pointer to the data and how many can be written in a chunk
	void		writeDone(BUInt num);				///< Indicates when write is complete
	void		writeBackup(BUInt num);				///< Backup, remove num items at end of fifo. Careful, make sure read is not already happening

	BUInt		readAvailable();				///< How many items are available to read
	BUInt		readAvailableChunk();				///< How many items are available to read in a chunk
	Type		read();						///< Read one item
	BError		read(Type* data, BUInt num);			///< Read a set of items

	Type		readPos(BUInt pos);				///< Read item at given offset from current read position
	Type*		readData();					///< Returns a pointer to the data
	Type*		readData(BUInt& num);				///< Returns a pointer to the data and how many can be read in a chunk
	void		readDone(BUInt num);

	Type&		operator[](int pos);				///< Direct access to read samples in buffer

protected:
	BMutex		olock;
	BUInt		osize;						///< The size of the FIFO
	Type*		odata;						///< FIFO memory buffer
	BUInt		owritePos;					///< The write pointer
	BUInt		oreadPos;					///< The read pointer
};

#include <BFifo.inc>
#endif
