/*
** File: main.c  
** Project: ADM-XRC monolithic driver
** Purpose: OS-independent entry point into ADM-XRC monolithic driver
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "device.h"
#include <core/main.h>
#include <modules/admxrc2/main.h>

const char g_dfDriverName[] = "admxrc";

static boolean_t
onAdd(
  DfDeviceObject* pDevObj)
{
  DeviceContext* pDevCtx;
  boolean_t bSuccess;
  
  dfDebugPrint(1, ("onAdd: entered, pDevObj=%p\n", pDevObj));
  
  /* Initialize the device context */
  pDevCtx = (DeviceContext*)dfDeviceGetContext(pDevObj);
  pDevCtx->pDevObj = pDevObj;
  
  bSuccess = adb3CoreOnAdd(pDevObj, &pDevCtx->adb3core);
  if (!bSuccess) {
    return FALSE;
  }
  bSuccess = admxrc2OnAdd(pDevObj, &pDevCtx->admxrc2);
  if (!bSuccess) {
    adb3CoreOnRemove(pDevObj, &pDevCtx->adb3core);
    return FALSE;
  }

  return TRUE;
}

static void
onRemove(
  DfDeviceObject* pDevObj)
{
  DeviceContext* pDevCtx;

  dfDebugPrint(1, ("onRemove: entered, pDevObj=%p\n", pDevObj));
  
  pDevCtx = (DeviceContext*)dfDeviceGetContext(pDevObj);

  admxrc2OnRemove(pDevObj, &pDevCtx->admxrc2);
  adb3CoreOnRemove(pDevObj, &pDevCtx->adb3core);
}

static boolean_t
onStart(
  DfDeviceObject* pDevObj,
  DfBusResources* pRawRes,
  DfBusResources* pTranslatedRes)
{
  DeviceContext* pDevCtx;
  boolean_t bSuccess;
  
  dfDebugPrint(1, ("onStart: entered, pDevObj=%p\n", pDevObj));

  pDevCtx = (DeviceContext*)dfDeviceGetContext(pDevObj);
  bSuccess = adb3CoreOnStart(pDevObj, &pDevCtx->adb3core, pRawRes, pTranslatedRes);
  if (!bSuccess) {
    return FALSE;
  }
  bSuccess = admxrc2OnStart(pDevObj, &pDevCtx->admxrc2, pRawRes, pTranslatedRes);
  if (!bSuccess) {
    adb3CoreOnStop(pDevObj, &pDevCtx->adb3core, TRUE);
    return FALSE;
  }
  return TRUE;
}

static void
onStop(
  DfDeviceObject* pDevObj,
  boolean_t bCanTouchHardware)
{
  DeviceContext* pDevCtx;

  dfDebugPrint(1, ("onStop: entered, pDevObj=%p bCanTouchHardware=%s\n",
    pDevObj, bCanTouchHardware ? "TRUE" : "FALSE"));

  pDevCtx = (DeviceContext*)dfDeviceGetContext(pDevObj);
  admxrc2OnStop(pDevObj, &pDevCtx->admxrc2, bCanTouchHardware);
  adb3CoreOnStop(pDevObj, &pDevCtx->adb3core, bCanTouchHardware);
}

static boolean_t
onPowerQuery(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int requestedState,
  boolean_t bIsShutdown)
{
  DeviceContext* pDevCtx;
  boolean_t bVeto = FALSE;

  pDevCtx = (DeviceContext*)dfDeviceGetContext(pDevObj);

  dfDebugPrint(1, ("onPowerQuery: entered, pDevObj=%p pDevCtx=%p currentState=D%u requestedState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, requestedState, bIsShutdown ? "TRUE" : "FALSE"));

  if (currentState < requestedState) {
    /* Powering down */
    bVeto |= admxrc2OnPowerQuery(pDevObj, &pDevCtx->admxrc2, currentState, requestedState, bIsShutdown);
    bVeto |= adb3CoreOnPowerQuery(pDevObj, &pDevCtx->adb3core, currentState, requestedState, bIsShutdown);
  } else {
    /* Powering up */
    bVeto |= adb3CoreOnPowerQuery(pDevObj, &pDevCtx->adb3core, currentState, requestedState, bIsShutdown);
    bVeto |= admxrc2OnPowerQuery(pDevObj, &pDevCtx->admxrc2, currentState, requestedState, bIsShutdown);
  }

  dfDebugPrint(1, ("onPowerQuery: bVeto=%s\n", bVeto ? "TRUE" : "FALSE"));

  return bVeto;
}

static void
onPowerSet(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int newState,
  boolean_t bIsShutdown)
{
  DeviceContext* pDevCtx;

  pDevCtx = (DeviceContext*)dfDeviceGetContext(pDevObj);

  dfDebugPrint(1, ("onPowerSet: entered, pDevObj=%p pDevCtx=%p currentState=D%u newState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, newState, bIsShutdown ? "TRUE" : "FALSE"));

  if (currentState < newState) {
    /* Powering down */
    admxrc2OnPowerSet(pDevObj, &pDevCtx->admxrc2, currentState, newState, bIsShutdown);
    adb3CoreOnPowerSet(pDevObj, &pDevCtx->adb3core, currentState, newState, bIsShutdown);
  } else {
    /* Powering up */
    adb3CoreOnPowerSet(pDevObj, &pDevCtx->adb3core, currentState, newState, bIsShutdown);
    admxrc2OnPowerSet(pDevObj, &pDevCtx->admxrc2, currentState, newState, bIsShutdown);
  }
}

boolean_t
dfDriverEntry(
  DfDriverObject* pDrvObj)
{
  boolean_t bSuccess;

  dfDebugPrint(0, ("dfDriverEntry: ADM-XRC Monolithic Driver, version=%lu.%lu.%lu.%lu\n",
    (unsigned long)ADMXRC_VERSION_0, (unsigned long)ADMXRC_VERSION_1, (unsigned long)ADMXRC_VERSION_2, (unsigned long)ADMXRC_VERSION_3));

  pDrvObj->deviceContextSize = sizeof(DeviceContext);
  pDrvObj->methods.pOnAdd = onAdd;
  pDrvObj->methods.pOnRemove = onRemove;
  pDrvObj->methods.pOnStart = onStart;
  pDrvObj->methods.pOnStop = onStop;
  pDrvObj->methods.pOnPowerQuery = onPowerQuery;
  pDrvObj->methods.pOnPowerSet = onPowerSet;

  bSuccess = adb3CoreDriverEntry(pDrvObj);
  if (!bSuccess) {
    return FALSE;
  }
  bSuccess = admxrc2DriverEntry(pDrvObj);
  if (!bSuccess) {
    adb3CoreDriverExit(pDrvObj);
    return FALSE;
  }
  return TRUE;
}

extern void
dfDriverExit(
  DfDriverObject* pDrvObj)
{
  dfDebugPrint(0, ("dfDriverExit: ADM-XRC Monolithic Driver exiting\n"));
  admxrc2DriverExit(pDrvObj);
  adb3CoreDriverExit(pDrvObj);
}
