/*
** File: sensor.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for sensor functions.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>
#include "device.h"
#include "sensor.h"
#include <admxrc2/sensort.h>

DfIoStatus
ioctlGetSensorInfo(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETSENSORINFO* pIoctl = (IOCTLS_ADMXRC2_GETSENSORINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  uint32_t sensorIndex;
  CoreSensorStatus sensorStatus;
  CoreSensorInfo sensorInfo;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  sensorIndex = pIoctl->in.sensorIndex;
  sensorStatus = pDevCtx->coreInterface.pGetSensorInfo(pDevCtx->pCoreContext, sensorIndex, &sensorInfo);
  if (CoreSensorSuccess != sensorStatus) {
    switch (sensorStatus) {
    case CoreSensorInvalidIndex:
      status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
      break;

    default:
      status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
      break;
    }
  } else {
    dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
    dfStrCpy(pIoctl->out.description, DF_ARRAY_LENGTH(pIoctl->out.description), sensorInfo.description);
    pIoctl->out.capabilities = sensorInfo.capabilities;
    pIoctl->out.error.intPart = sensorInfo.error.intPart;
    pIoctl->out.error.fracPart = sensorInfo.error.fracPart;
    pIoctl->out.datatype = sensorInfo.datatype;
    pIoctl->out.unit = sensorInfo.unit;
    pIoctl->out.exponent = sensorInfo.exponent;
  }

  return status;
}

DfIoStatus
ioctlReadSensor(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_READSENSOR* pIoctl = (IOCTLS_ADMXRC2_READSENSOR*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreSensorInfo sensorInfo;
  CoreSensorValue sensorValue;
  CoreSensorStatus sensorStatus;
  uint32_t sensorIndex;
  uint32_t datatype;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  sensorIndex = pIoctl->in.sensorIndex;

  sensorStatus = pDevCtx->coreInterface.pGetSensorInfo(pDevCtx->pCoreContext, sensorIndex, &sensorInfo);
  if (CoreSensorSuccess != sensorStatus) {
    switch (sensorStatus) {
    case CoreSensorInvalidIndex:
      status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
      break;

    case CoreSensorHardwareError:
    case CoreSensorUnexpectedError:
    default:
      status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
      break;
    }
  }
  datatype = sensorInfo.datatype;

  sensorStatus = pDevCtx->coreInterface.pReadSensor(pDevCtx->pCoreContext, sensorIndex, &sensorValue);
  if (CoreSensorSuccess != sensorStatus) {
    switch (sensorStatus) {
    case CoreSensorInvalidIndex:
      status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
      break;

    case CoreSensorHardwareError:
      status = DfIoStatusError(ADMXRC2_CARD_ERROR);
      break;

    case CoreSensorUnexpectedError:
    default:
      status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
      break;
    }
  }

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  switch (datatype) {
  case ADMXRC2_DATA_BOOL:
    pIoctl->out.value.booleanValue = sensorValue.booleanValue;
    break;

  case ADMXRC2_DATA_DOUBLE:
    pIoctl->out.value.doubleValue.intPart = sensorValue.doubleValue.intPart;
    pIoctl->out.value.doubleValue.fracPart = sensorValue.doubleValue.fracPart;
    break;

  case ADMXRC2_DATA_INT32:
    pIoctl->out.value.int32Value = sensorValue.int32Value;
    break;

  case ADMXRC2_DATA_UINT32:
    pIoctl->out.value.uint32Value = sensorValue.uint32Value;
    break;

  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
  pIoctl->out.datatype = datatype;

  return status;
}
