/*
** File: model_admxrc4.c
** Project: ADMXRC2 module driver
** Purpose: Code, including the function to bootstrap the device context,
**          specific to:
**   o ADM-XRC-4LX
**   o ADM-XRC-4SX
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include <core/pci9xxx.h>
#include <core/admxrc4.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  unsigned int numBank = (pDevCtx->info.model == CoreModelAdmxrc4lx) ? 6U : 4U;
  VpdAdmxrc4Rev0* pVpd = (VpdAdmxrc4Rev0*)pVpdBuffer;
  uint32_t zbtAll, bank;
  unsigned int shift, order;

  if (bankIndex >= numBank) {
    return ADMXRC2_INVALID_INDEX;
  }
  zbtAll = pVpd->banks01 | ((uint32_t)pVpd->banks23 << 8) | ((uint32_t)pVpd->banks45 << 16);
  shift = bankIndex * 4;
  bank = (zbtAll >> shift) & 0xfU;
  order = ((bank >> 2) & 0x3U) + 18;
  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.width = 32;
  pIoctl->out.typeMask = bank & 0x3U;
  pIoctl->out.bFitted = (bank & 0x3U) ? TRUE : FALSE;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  unsigned int numBank = (pDevCtx->info.model == CoreModelAdmxrc4lx) ? 6U : 4U;
  VpdAdmxrc4Rev0* pVpd = (VpdAdmxrc4Rev0*)pVpdBuffer;
  uint32_t zbtAll, bank;
  unsigned int bankIndex, shift;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType;
  pIoctl->out.serialNumber = (uint32_t)pVpd->serialLow | ((uint32_t)pVpd->serialHigh << 16);
  pIoctl->out.numMemoryBank = numBank;
  pIoctl->out.memoryBankFitted = 0;
  zbtAll = pVpd->banks01 | ((uint32_t)pVpd->banks23 << 8) | ((uint32_t)pVpd->banks45 << 16);
  for (bankIndex = 0; bankIndex < 6; bankIndex++) {
    shift = bankIndex * 4;
    bank = (zbtAll >> shift) & 0xfU;
    if (0 == (bank & 0x3)) {
      /* Bank is not fitted */
    } else {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << bankIndex);
    }
  }
  pIoctl->out.logicRevision = pVpd->logicRev;
  pIoctl->out.boardRevision = pVpd->boardRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdmxrc4Rev0* pVpd = (VpdAdmxrc4Rev0*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1148");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  pIoctl->out.tempGrade = (uint32_t)-1;
  pIoctl->out.scd = (uint32_t)-1;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdmxrc4(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdmxrc4: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 2U;
  pDevCtx->configureDmaAddress = 0x820000U; /* TO DO - should make this dynamic, i.e. based on current value of LAS1BA */
  pDevCtx->configureDmaMode =
    PCI9656_DMAMODE_8BIT | PCI9656_DMAMODE_USE_READY | PCI9656_DMAMODE_USE_BTERM |
    PCI9656_DMAMODE_ENABLE_BURST | PCI9656_DMAMODE_FIXED_LOCAL;
}
