/*
** File: model_admxrc.c
** Project: ADMXRC2 module driver
** Purpose: ADM-XRC / ADM-XRC-P specific code, including the function to
**          bootstrap the device context.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include <core/pci9xxx.h>
#include <core/admxrc.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdmxrcRev0* pVpd = (VpdAdmxrcRev0*)pVpdBuffer;
  unsigned int order;
  boolean_t bFitted;

  if (bankIndex >= 4) {
    return ADMXRC2_INVALID_INDEX;
  }
  if (pVpd->zbt0123 & (0x1U << bankIndex)) {
    order = (pVpd->zbt0123 & (0x10U << bankIndex)) ? 19U : 18U;
    bFitted = TRUE;
  } else {
    order = 0;
    bFitted = FALSE;
  }
  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.width = 36U;
  pIoctl->out.typeMask = ADMXRC2_RAM_ZBTFT;
  pIoctl->out.bFitted = bFitted;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdmxrcRev0* pVpd = (VpdAdmxrcRev0*)pVpdBuffer;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType;
  pIoctl->out.serialNumber = (uint32_t)pVpd->serialLow | ((uint32_t)pVpd->serialHigh << 16);
  pIoctl->out.numMemoryBank = 4;
  pIoctl->out.memoryBankFitted = pVpd->zbt0123 & 0xFU;
  pIoctl->out.logicRevision = pVpd->cpldRev;
  pIoctl->out.boardRevision = pVpd->pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdmxrcRev0* pVpd = (VpdAdmxrcRev0*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "BG560");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  pIoctl->out.tempGrade = (uint32_t)-1;
  pIoctl->out.scd = (uint32_t)-1;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdmxrc(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdmxrc: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 2U;
  pDevCtx->configureDmaAddress = 0x800008U; /* TODO - should make this dynamic, i.e. based on current value of LAS1BA */
  pDevCtx->configureDmaMode =
    PCI9656_DMAMODE_8BIT | PCI9656_DMAMODE_USE_READY | PCI9656_DMAMODE_USE_BTERM |
    PCI9656_DMAMODE_ENABLE_BURST | PCI9656_DMAMODE_FIXED_LOCAL;
}
