/*
** File: device.h  
** Project: ADMXRC2 module driver
** Purpose: Data structures and definitions private to ADMXRC2 module driver,
**          including the device context structure.
**
** (C) Copyright Alpha Data 2013
*/

#if !defined(ADATA_ADMXRC2_DEVICE_H)
#define ADATA_ADMXRC2_DEVICE_H

#include <df.h>

#include <core/coreif.h>
#include <core/shared_defs.h>
#include "device_defs.h"

typedef int32_t   _ADMXRC2_INT32;
typedef uint64_t  _ADMXRC2_UINT64;
typedef uint32_t  _ADMXRC2_UINT32;
typedef uint16_t  _ADMXRC2_UINT16;
typedef uint8_t   _ADMXRC2_UINT8;
typedef uint8_t   _ADMXRC2_BYTE;
typedef boolean_t _ADMXRC2_BOOL;

#if DF_HAS_USER_EVENT_HANDLE
# define ADMXRC2_HAS_USER_EVENT_HANDLE (1)
typedef user_event_handle_t admxrc2_user_event_handle_t;
# if DF_NEED_THUNK
typedef user_event_handle_t_32 admxrc2_user_event_handle_t_32;
# endif
#else
# define ADMXRC2_HAS_USER_EVENT_HANDLE (0)
#endif
#if DF_NEED_THUNK
# define ADMXRC2_NEED_THUNK (1)
#endif
#include <ioctl_admxrc2.h>
#include <admxrc2/status.h>

struct _BufferDescription;
typedef struct _BufferDescription BufferDescription;
struct _Admxrc2DeviceContext;
typedef struct _Admxrc2DeviceContext Admxrc2DeviceContext;
struct _DmaRequestContext;
typedef struct _DmaRequestContext DmaRequestContext;
struct _Admxrc2ClientContext;
typedef struct _Admxrc2ClientContext Admxrc2ClientContext;
struct _ClockGeneratorTicket;
typedef struct _ClockGeneratorTicket ClockGeneratorTicket;
struct _ConfigureDmaTicket;
typedef struct _ConfigureDmaTicket ConfigureDmaTicket;
struct _DmaTicket;
typedef struct _DmaTicket DmaTicket;
struct _DmaBusTicket;
typedef struct _DmaBusTicket DmaBusTicket;
union _DmaTicketUnion;
typedef union _DmaTicketUnion DmaTicketUnion;
#if defined(TODO)
struct _FpgaAlertNotification;
typedef struct _FpgaAlertNotification FpgaAlertNotification;
#endif
struct _FpgaInterruptNotification;
typedef struct _FpgaInterruptNotification FpgaInterruptNotification;
struct _GenericRequestContext;
typedef struct _GenericRequestContext GenericRequestContext;
struct _InterruptWaitList;
typedef struct _InterruptWaitList InterruptWaitList;
struct _InterruptWaitTicket;
typedef struct _InterruptWaitTicket InterruptWaitTicket;
struct _SensorPollNotification;
typedef struct _SensorPollNotification SensorPollNotification;
struct _UserBufferMappings;
typedef struct _UserBufferMappings UserBufferMappings;
#if DF_HAS_USER_EVENT_HANDLE
struct _UserEventInfo;
typedef struct _UserEventInfo UserEventInfo;
struct _UserEventList;
typedef struct _UserEventList UserEventList;
#endif
struct _RequestContext;
typedef struct _RequestContext RequestContext;
struct _WindowAccessInfo;
typedef struct _WindowAccessInfo WindowAccessInfo;

#if DF_HAS_USER_EVENT_HANDLE
struct _UserEventInfo {
  Admxrc2ClientContext* pClCtx;
  user_event_handle_t hUserEvent;
  DfUserEvent userEvent;
  unsigned int refCount;
  boolean_t bRemoving;
  UserEventInfo* pPrev;
  UserEventInfo* pNext;
};

struct _UserEventList {
  DfSpinLock lock;
  UserEventInfo* pHead;
  UserEventInfo* pTail;
};
#endif

struct _ClockGeneratorTicket {
  CoreTicket ticket;
  CoreClockStatus status;
  DfEvent event;
};

struct _ConfigureDmaTicket {
  CoreTicketMakeStruct(, 2) ticket;
  DfEvent event;
  DfIoStatus status;
  CoreDmaParameters dmaParameters;
  union {
    BufferDescription* pBuffer;        /* Used by ? */
    DfBufferDescription* pDescription; /* Used by ioctlConfigureDma */
  } variant;
};

struct _InterruptWaitList {
  DfSpinLock lock;
  DfListNode header;
};

struct _DmaTicket {
  CoreTicket ticket;
  DfSpinLock lock;
  DfAtomicInt usage; /* This is a count of the number of "potential" threads accessing the structure */
  DfIoStatus status;
  boolean_t bCancel;
  CoreDmaParameters dmaParameters;
  unsigned int dmaChannel;
  uint32_t flags;
	Admxrc2DeviceContext* pDevCtx;
	Admxrc2ClientContext* pClCtx;
  DfClientRequest* pRequest;
  boolean_t bDmaLocked;
  union {
    IOCTLS_ADMXRC2_DODMA* pLocked;
    IOCTLS_ADMXRC2_DODMAIMMEDIATE* pImmediate;
  } ioctl;
  DfTimer timer;
  DfTime timeoutTicks;
  uint8_t acquireState;   /* State of resource acquisition attempt. */
  uint8_t callbackState;  /* Whether or not client request has a cancel callback. */
  uint8_t dmaState;       /* Whether DMA transfer is not started, started, completed etc. */
  uint8_t timerState;     /* Whether or not timer is running. */
  union {
    BufferDescription* pBuffer;        /* Used by ioctlDoDma */
    DfBufferDescription* pDescription; /* Used by ioctlDoDmaImmediate */
  } variant;
};

struct _DmaBusTicket {
  CoreTicket ticket;
  DfSpinLock lock;
  DfAtomicInt usage; /* This is a count of the number of "potential" threads accessing the structure */
  DfIoStatus status;
  unsigned int state;
  boolean_t bAcquired;
  boolean_t bCancel;
  CoreDmaBusParameters dmaParameters;
  unsigned int dmaChannel;
  uint32_t flags;
	Admxrc2DeviceContext* pDevCtx;
	Admxrc2ClientContext* pClCtx;
  DfClientRequest* pRequest;
};

union _DmaTicketUnion {
  DmaTicket dma;
  DmaBusTicket dmaBus;
};

struct _InterruptWaitTicket {
  DfListNode node;
	Admxrc2DeviceContext* pDevCtx;
  DfClientRequest* pRequest;
  IOCTLS_ADMXRC2_WAITFORINTERRUPT* pIoctl;
  DfTimer timer;
  DfTime timeoutTicks;
  unsigned int targetIndex;
  uint8_t waitState;
  uint8_t cancelState;
  uint8_t timerState;
};

/* Structure that keeps track of mappings between ADMXRC2_BUFFER_HANDLE and user space buffers */
struct _UserBufferMappings {
  DfSpinLock lock;
  /* Maps a buffer handle (uint32_t) to a BufferDescription* */
  BufferDescription* map[MAX_NUM_LOCKED_USER_SPACE_BUFFER];
  /* Stack that contains free handles */
  struct {
    uint32_t free[MAX_NUM_LOCKED_USER_SPACE_BUFFER];
    unsigned int top; /* Stack top pointer */
  } stack;
};

struct _Admxrc2ClientContext {
  DfListNode waitNode;           /* Node in Admxrc2DeviceContext::powerMgmt::waitingOpen list */
  DfClientRequest* pOpenRequest; /* If onOpen was deferred, request will be here */

  void* hDisableSleep; /* Handle obtained from PoRegisterSystemState */

  Admxrc2DeviceContext* pDevCtx;
  boolean_t bPrivileged;

  CoreClient coreClient;
  boolean_t bCoopLevelIsSet;

  boolean_t bNeedFlashSync[MAX_NUM_FLASH_BANK]; /* TRUE if need to do a flash sync for a bank when closing the device */
  boolean_t bNeedVpdSync; /* TRUE if need to do a VPD sync when closing the device */

#if !defined(ADMXRC2_LOCKED_BUFFERS_GLOBAL)
  /* Information about locked user-space buffers */
  UserBufferMappings lockedBuffers;
#endif

#if defined(ADMXRC2_SINGLE_REQUEST_PER_HANDLE)
  /*
  ** If ADMXRC_SINGLE_REQUEST_PER_HANDLE is defined, then a restriction is enforced
  ** whereby at most one I/O request can be ongoing per open device handle.
  */
  struct {
    DfClientRequest* pRequest; /* Current I/O request */
    union {              /* Context for current I/O request */
      ClockGeneratorTicket clockGen;
      DmaTicket dma;
    } context;
  } currentRequest;
#endif
};

#if defined(TODO)
struct _FpgaAlertNotification {
  CoreNotification coreNotification;
  unsigned int targetIndex;
};
#endif

struct _FpgaInterruptNotification {
  CoreNotification coreNotification;
  unsigned int targetIndex;
};

struct _SensorPollNotification {
  CoreNotification coreNotification;
};

struct _WindowAccessInfo {
  void* pKernelBase;
  uint64_t localBase;
  uint64_t size;
};

/* As per PowerManagementScheme driver parameter */
typedef enum _PowerMgmtScheme {
  PowerMgmtNone = 0,       /* Never allow sleeping */
  PowerMgmtNotActive = 1,  /* Allow sleeping if nobody has opened us */
  PowerMgmtInvalidValue
} PowerMgmtScheme;

struct _Admxrc2DeviceContext {
  DfDeviceObject* pDevObj;
  Adb3CoreInterface coreInterface;
  void* pCoreContext;
  boolean_t bCoreInterfaceImpd;
  boolean_t bReverseSelectMapData;
  uint64_t localAddressLimit;
  uint32_t configureDmaMode;
  uint32_t configureDmaAddress;
  uint8_t localWidth;
  uint8_t vpdWidth;

  struct {
    DfInterface iface;
    boolean_t bEnabled;
    boolean_t bRegistered;
  } userMode;

  struct {
    CoreModelType model;
    unsigned int numFlashBank;
    unsigned int numTargetFpga;
    unsigned int numWindow;
    unsigned int numCommonBuffer;
  } info;

  struct {
    ADMXRC2_STATUS (*pGetBankInfo)(Admxrc2DeviceContext* pDevCtx, unsigned int bankIndex, const void* pVpdBuffer, IOCTLS_ADMXRC2_GETBANKINFO* pIoctl);
    void           (*pGetCardInfo)(Admxrc2DeviceContext* pDevCtx, const void* pVpdBuffer, IOCTLS_ADMXRC2_GETCARDINFO* pIoctl);
    ADMXRC2_STATUS (*pGetFpgaInfo)(Admxrc2DeviceContext* pDevCtx, unsigned int targetIndex, const void* pVpdBuffer, IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl);
  } methods;

  struct {
    boolean_t bTicketPoolValid;
    DfPool ticketPool;
  } dma;

  struct {
    boolean_t bTicketPoolValid;
    DfPool ticketPool;
  } clockGenerator;

  struct {
    struct {
      DfSpinLock lock;
      Admxrc2ClientContext* pClCtx;
    } owner;
    struct {
      FpgaInterruptNotification notification;
      boolean_t bNotificationRegistered;
#if DF_HAS_USER_EVENT_HANDLE 
      UserEventList eventList;
#endif
      InterruptWaitList waitList;
      boolean_t bInterruptFlag;
    } interrupt;
#if defined(TODO)
    struct {
      FpgaAlertNotification notification;
      boolean_t bNotificationRegistered;
#if DF_HAS_USER_EVENT_HANDLE 
      UserEventList eventList;
#endif
      NotificationWaitList waitList;
    } alert;
#endif
  } fpga[MAX_NUM_TARGET_FPGA];

  struct {
    uint8_t* pStagingBuffer;
  } flash[MAX_NUM_FLASH_BANK];

  struct {
    boolean_t bTicketPoolValid;
    DfPool ticketPool;
  } interruptWait;

  struct {
    SensorPollNotification notification;
    boolean_t bNotificationRegistered;
  } sensor;

  struct {
    DfSpinLock lock;        /* Lock protecting this substruct */
    PowerMgmtScheme scheme; /* See PowerMgmtScheme enum for allowed values */
    unsigned int openCount; /* Count of handles to device open in active mode */
    boolean_t bDeferOpen;   /* TRUE => onOpen will defer opening */
    DfList waitingOpen;     /* List of clients whose open request is deferred */
    void* hDisableSleep;    /* If scheme is PowerMgmtNone, holds handle obtained from PoRegisterSystemState */
  } powerMgmt;

  /* Information required to validate IOCTL parameters in pio.c */
  struct {
    WindowAccessInfo access[MAX_NUM_LOCAL_BUS_WINDOW];
    struct {
      void* pKernelBase;
      size_t size;
    } adapter;
  } window;
};

struct _BufferDescription {
  uint32_t handle;
  uint32_t refCount;
  Admxrc2ClientContext* pCreator;
  DfBufferDescription* pDfBufferDesc;
};

/* Returns TRUE if OK to proceed with an operation that can change device state, otherwise FALSE. */
DF_DECLARE_INLINE_FUNC(boolean_t, admxrc2CheckAccessRights)(
  Admxrc2ClientContext* pClCtx)
{
#if 0
  return pClCtx->bCoopLevelIsSet;
#else
  return TRUE;
#endif
}

#endif
