/*
** File: debugts.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for Debug Timestamping.
**
** (C) Copyright Alpha Data 2016
*/

#include <df.h>
#include "device.h"
#include "debugts.h"
#include <admxrc2/debugtst.h>

#if defined(BUILD_DEBUGTS)

static DfIoStatus
mapReadResult(
  CoreReadDebugTSResult readResult)
{
  switch (readResult) {
  case CoreReadDebugTSSuccess:
    return DfIoStatusSuccess;

  case CoreReadDebugTSNotEnabled:
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);

  case CoreReadDebugTSInvalidFacility:
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);

  case CoreReadDebugTSInvalidRegion:
    return DfIoStatusError(ADMXRC2_INVALID_REGION);

  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapResult(
  CoreDebugTSResult readResult)
{
  switch (readResult) {
  case CoreDebugTSSuccess:
    return DfIoStatusSuccess;

  case CoreDebugTSNotEnabled:
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);

  case CoreDebugTSInvalidFacility:
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);

  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

#endif

#define READ_CHUNK_SIZE (8)

DfIoStatus
ioctlGetDebugTSInfo(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
#if defined(BUILD_DEBUGTS)
  IOCTLS_ADMXRC2_GETDEBUGTSINFO* pIoctl = (IOCTLS_ADMXRC2_GETDEBUGTSINFO*)pBuffer;
  CoreDebugTSResult getResult;
  CoreDebugTSInfo info;
  uint32_t facility;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  facility = pIoctl->in.facility;
  getResult = pDevCtx->coreInterface.pGetDebugTSInfo(pDevCtx->pCoreContext, facility, &info);
  if (CoreDebugTSSuccess == getResult) {
    pIoctl->out.frequency = info.frequency;
    pIoctl->out.initTimestamp = info.initTimestamp;
    pIoctl->out.initTime = info.initTime;
    pIoctl->out.bufferSize = info.bufferSize;
  }
  return mapResult(getResult);
#else
  return ADMXRC2_NOT_SUPPORTED;
#endif
}

#if DF_NEED_THUNK
DfIoStatus
ioctlGetDebugTSInfoThunk(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS32_ADMXRC2_GETDEBUGTSINFO* pIoctl32 = (IOCTLS32_ADMXRC2_GETDEBUGTSINFO*)pBuffer;
  IOCTLS_ADMXRC2_GETDEBUGTSINFO ioctl;
  DfIoStatus status;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize || sizeof(pIoctl32->out) != outSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.facility = pIoctl32->in.facility;
  status = ioctlGetDebugTSInfo(pDevCtx, &ioctl, sizeof(ioctl.in), sizeof(ioctl.out));
  if (status == DfIoStatusSuccess) {
    pIoctl32->out.frequency = ioctl.out.frequency;
    pIoctl32->out.initTime = ioctl.out.initTime;
    pIoctl32->out.initTimestamp = ioctl.out.initTimestamp;
    pIoctl32->out.bufferSize = (size32_t)ioctl.out.bufferSize; /* Cast safe because buffer size limited to 2^24 entries */
  }
  return status;
}
#endif

DfIoStatus
ioctlGetDebugTSStatus(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
#if defined(BUILD_DEBUGTS)
  IOCTLS_ADMXRC2_GETDEBUGTSSTATUS* pIoctl = (IOCTLS_ADMXRC2_GETDEBUGTSSTATUS*)pBuffer;
  CoreDebugTSResult getResult;
  CoreDebugTSStatus status;
  uint32_t facility;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  facility = pIoctl->in.facility;
  getResult = pDevCtx->coreInterface.pGetDebugTSStatus(pDevCtx->pCoreContext, facility, &status);
  if (CoreDebugTSSuccess == getResult) {
    pIoctl->out.count = status.count;
    pIoctl->out.status = status.status;
  }
  return mapResult(getResult);
#else
  return ADMXRC2_NOT_SUPPORTED;
#endif
}

#if DF_NEED_THUNK
DfIoStatus
ioctlGetDebugTSStatusThunk(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS32_ADMXRC2_GETDEBUGTSSTATUS* pIoctl32 = (IOCTLS32_ADMXRC2_GETDEBUGTSSTATUS*)pBuffer;
  IOCTLS_ADMXRC2_GETDEBUGTSSTATUS ioctl;
  DfIoStatus status;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize || sizeof(pIoctl32->out) != outSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.facility = pIoctl32->in.facility;
  status = ioctlGetDebugTSStatus(pDevCtx, &ioctl, sizeof(ioctl.in), sizeof(ioctl.out));
  if (status == DfIoStatusSuccess) {
    pIoctl32->out.count = (size32_t)ioctl.out.count; /* Cast safe because buffer size limited to 2^24 entries */
    pIoctl32->out.status = ioctl.out.status;
  }
  return status;
}
#endif

DfIoStatus
ioctlReadDebugTS(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(BUILD_DEBUGTS)
  IOCTLS_ADMXRC2_READDEBUGTS* pIoctl = (IOCTLS_ADMXRC2_READDEBUGTS*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  typedef uint64_t DebugTSEntry[2];
  DebugTSEntry entryBuffer[READ_CHUNK_SIZE];
  CoreReadDebugTSResult readResult = CoreReadDebugTSSuccess;
  uint32_t facility, flags;
  size_t offset, remaining;
  uint8_t* pUserBuffer;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  pUserBuffer = (uint8_t*)pIoctl->in.pBuffer;
  facility = pIoctl->in.facility;
  offset = pIoctl->in.offset;
  remaining = pIoctl->in.count;
  flags = pIoctl->in.flags;

  DF_USER_SPACE_BEGIN(TAG1)

  dfUserBufferValidate(pUserBuffer, remaining, TRUE, TAG1);

  while (remaining) {
    uint32_t chunkFlags;
    size_t chunk, chunkBytes;

    if (remaining > READ_CHUNK_SIZE) {
      chunkFlags = flags & ~ADMXRC2_DEBUGTS_FLAG_CONSUME;
      chunk = READ_CHUNK_SIZE;
    } else {
      chunkFlags = flags;
      chunk = remaining;
    }
    chunkBytes = chunk * sizeof(DebugTSEntry);
    readResult = pDevCtx->coreInterface.pReadDebugTS(pDevCtx->pCoreContext, facility, offset, chunk, chunkFlags, entryBuffer);
    if (CoreReadDebugTSSuccess != readResult) {
      break;
    }
    dfCopyKU(pUserBuffer, entryBuffer, chunkBytes, TAG1);
    offset += chunk;
    remaining -= chunk;
    pUserBuffer += chunkBytes;
  }

  DF_USER_SPACE_EXCEPT(TAG1)

  status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER);

  DF_USER_SPACE_END(TAG1)

  if (DfIoStatusIsError(status)) {
    return status;
  }

  return mapReadResult(readResult);
#else
  return ADMXRC2_NOT_SUPPORTED;
#endif
}

#if DF_NEED_THUNK
DfIoStatus
ioctlReadDebugTSThunk(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_READDEBUGTS* pIoctl32 = (IOCTLS32_ADMXRC2_READDEBUGTS*)pBuffer;
  IOCTLS_ADMXRC2_READDEBUGTS ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.facility = pIoctl32->in.facility;
  ioctl.in.offset = pIoctl32->in.offset;
  ioctl.in.count = pIoctl32->in.count;
  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.pBuffer = dfThunkPtr(pIoctl32->in.pBuffer);
  return ioctlReadDebugTS(pDevCtx, pClCtx, &ioctl, sizeof(ioctl.in));
}
#endif

DfIoStatus
ioctlResetDebugTS(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(BUILD_DEBUGTS)
  IOCTLS_ADMXRC2_RESETDEBUGTS* pIoctl = (IOCTLS_ADMXRC2_RESETDEBUGTS*)pBuffer;
  CoreDebugTSResult resetResult = CoreDebugTSSuccess;
  uint32_t facility;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  facility = pIoctl->in.facility;
  resetResult = pDevCtx->coreInterface.pResetDebugTS(pDevCtx->pCoreContext, facility);
  return mapResult(resetResult);
#else
  return ADMXRC2_NOT_SUPPORTED;
#endif
}
