/*
** File: combuf.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers relating to common buffer API
**          functions.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>
#include "device.h"
#include "combuf.h"

DfIoStatus
ioctlGetCommonBufferInfo(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETCOMMONBUFFERINFO* pIoctl = (IOCTLS_ADMXRC2_GETCOMMONBUFFERINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreCommonBufferStatus commonBufferStatus;
  CoreCommonBufferInfo commonBufferInfo;
  unsigned int index;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  index = pIoctl->in.index;
  commonBufferStatus = pDevCtx->coreInterface.pGetCommonBufferInfo(pDevCtx->pCoreContext, index, &commonBufferInfo);
  if (CoreCommonBufferSuccess != commonBufferStatus) {
    switch (commonBufferStatus) {
    case CoreCommonBufferInvalidIndex:
      status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
      break;

    default:
      status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
      break;
    }
  } else {
    dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
    pIoctl->out.busAddress = commonBufferInfo.busBase;
    pIoctl->out.busSize = commonBufferInfo.size;
  }
  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlGetCommonBufferInfoThunk(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS32_ADMXRC2_GETCOMMONBUFFERINFO* pIoctl32 = (IOCTLS32_ADMXRC2_GETCOMMONBUFFERINFO*)pBuffer;
  IOCTLS_ADMXRC2_GETCOMMONBUFFERINFO ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize || sizeof(pIoctl32->out) != outSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.index = pIoctl32->in.index;
  status = ioctlGetCommonBufferInfo(pDevCtx, &ioctl, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    pIoctl32->out.busAddress = ioctl.out.busAddress;
    if (ioctl.out.busSize > (size32_t)-1) {
      pIoctl32->out.busSize = (size32_t)-1;
    } else {
      pIoctl32->out.busSize = (size32_t)ioctl.out.busSize; /* Safe cast due to range check */
    }
  }
  return status;
}
#endif

DfIoStatus
ioctlSyncCommonBuffer(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_SYNCCOMMONBUFFER* pIoctl = (IOCTLS_ADMXRC2_SYNCCOMMONBUFFER*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreCommonBufferStatus commonBufferStatus;
  unsigned int index;
  size_t offset, length;
  ADMXRC2_SYNCMODE syncMode;
  boolean_t bDmaWrite;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  index = pIoctl->in.index;
  syncMode = (ADMXRC2_SYNCMODE)pIoctl->in.syncMode;
  offset = pIoctl->in.offset;
  length = pIoctl->in.offset;
  switch (syncMode) {
  case ADMXRC2_SYNC_CPUTOFPGA:
    bDmaWrite = FALSE;
    break;

  case ADMXRC2_SYNC_FPGATOCPU:
    bDmaWrite = TRUE;
    break;

  default:
    return ADMXRC2_INVALID_PARAMETER;
  }

  commonBufferStatus = pDevCtx->coreInterface.pSyncCommonBuffer(pDevCtx->pCoreContext, index, bDmaWrite, offset, length);
  if (CoreCommonBufferSuccess != commonBufferStatus) {
    switch (commonBufferStatus) {
    case CoreCommonBufferInvalidIndex:
      status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
      break;

    case CoreCommonBufferInvalidRegion:
      status = DfIoStatusError(ADMXRC2_INVALID_REGION);
      break;

    default:
      status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
      break;
    }
  }
  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlSyncCommonBufferThunk(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS32_ADMXRC2_SYNCCOMMONBUFFER* pIoctl32 = (IOCTLS32_ADMXRC2_SYNCCOMMONBUFFER*)pBuffer;
  IOCTLS_ADMXRC2_SYNCCOMMONBUFFER ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.index = pIoctl32->in.index;
  ioctl.in.syncMode = pIoctl32->in.syncMode;
  ioctl.in.offset = pIoctl32->in.offset;
  ioctl.in.length = pIoctl32->in.length;
  return ioctlSyncCommonBuffer(pDevCtx, pClCtx, &ioctl, sizeof(ioctl.in), 0);
}

#endif
