/*
** File: clock.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for setting and getting clock
**          generator frequencies.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "clock.h"
#include <admxrc2/types.h>

static DfIoStatus
mapAcquireStatus(
  CoreAcquireStatus acqStatus)
{
  switch (acqStatus) {
  case CoreAcquireSuccess:
    return DfIoStatusSuccess;

  case CoreAcquireMustWait:
    return DfIoStatusError(ADMXRC2_DEVICE_BUSY);

  case CoreAcquireCancelled:
    return DfIoStatusError(ADMXRC2_CANCELLED);

  case CoreAcquireRemoved:
  case CoreAcquireFailed:
    return DfIoStatusError(ADMXRC2_FAILED);

  case CoreAcquireInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreAcquireInvalid:
  case CoreAcquireUnexpectedError:
  case CoreAcquireWaiting:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

DfIoStatus
ioctlGetClockFrequency(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETCLOCKFREQUENCY* pIoctl = (IOCTLS_ADMXRC2_GETCLOCKFREQUENCY*)pBuffer;
  DfIoStatus status;
  CoreTicket ticket;
  CoreAcquireStatus acqStatus;
  CoreClockStatus clockStatus;
  unsigned int clockIndex;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize ) {
    return DfIoStatusInvalid;
  }

  clockIndex = pIoctl->in.clockIndex;
  if (clockIndex >= pDevCtx->coreInterface.pGetNumClockGen(pDevCtx->pCoreContext)) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_CLOCKGEN;
  ticket.resources[0].index = (uint8_t)clockIndex;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }

  clockStatus = pDevCtx->coreInterface.pClockGetFreq(pDevCtx->pCoreContext, &ticket, clockIndex, &pIoctl->out.frequency);
  switch (clockStatus) {
  case CoreClockSuccess:
    status = DfIoStatusSuccess;
    break;
  case CoreClockInvalidIndex:
    status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
    break;
  case CoreClockHardwareError:
    status = DfIoStatusError(ADMXRC2_FAILED);
    break;
  default:
    status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
    break;
  }

  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);
  return status;
}

static void
clockProgCallback(
  CoreTicket* pCoreTicket,
  void* pCallbackContext /* unused */,  
  CoreClockStatus status)
{
  ClockGeneratorTicket* pTicket = (ClockGeneratorTicket*)pCoreTicket;

  pTicket->status = status;
  dfEventSignal(&pTicket->event);
}

DfIoStatus
ioctlSetClockFrequency(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  const uint32_t legalFlagsMask = 0;
#endif
  IOCTLS_ADMXRC2_SETCLOCKFREQUENCY* pIoctl = (IOCTLS_ADMXRC2_SETCLOCKFREQUENCY*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  ClockGeneratorTicket* pTicket = NULL;
  CoreAcquireStatus acqStatus;
  CoreClockStatus clockStatus;
  unsigned int clockIndex;
  uint64_t frequency;
  uint32_t flags;
  CoreClockWord clockWord;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize ) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  clockIndex = pIoctl->in.clockIndex;
  frequency  = pIoctl->in.frequency;
  flags = pIoctl->in.flags;
  if (clockIndex >= pDevCtx->coreInterface.pGetNumClockGen(pDevCtx->pCoreContext)) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }
#if defined(ADMXRC2_CHECK_FLAGS)
  if (flags & ~legalFlagsMask) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  if (flags & ADMXRC2_SETCLOCK_MINIMUM) {
    flags |= CoreClockMinimumFrequency;
  }
  if (flags & ADMXRC2_SETCLOCK_MAXIMUM) {
    flags |= CoreClockMaximumFrequency;
  }
  clockStatus = pDevCtx->coreInterface.pClockTestFreq(pDevCtx->pCoreContext, clockIndex, flags, frequency, frequency, frequency, &clockWord);
  switch (clockStatus) {
  case CoreClockSuccess:
    break;

  case CoreClockInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreClockOutOfRange:
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);

  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }

  pTicket = dfPoolAlloc(&pDevCtx->clockGenerator.ticketPool, ClockGeneratorTicket);
  if (NULL == pTicket) {
    return DfIoStatusError(ADMXRC2_NO_MEMORY);
  }

  if (!(flags & ADMXRC2_SETCLOCK_TESTONLY)) {
    pTicket->ticket.count = 1;
    pTicket->ticket.resources[0].code = CORE_RESOURCE_CLOCKGEN;
    pTicket->ticket.resources[0].index = (uint8_t)clockIndex;
    acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &pTicket->ticket);
    if (CoreAcquireSuccess != acqStatus) {
      status = mapAcquireStatus(acqStatus);
      goto done;
    }
    acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &pTicket->ticket, 0);
    if (CoreAcquireSuccess != acqStatus) {
      status = mapAcquireStatus(acqStatus);
      goto done;
    }

    dfEventInit(&pTicket->event);
    clockStatus = pDevCtx->coreInterface.pClockProgram(pDevCtx->pCoreContext, &pTicket->ticket, clockIndex, &clockWord, clockProgCallback, NULL);
    if (CoreClockSuccess == clockStatus) {
      dfEventWait(&pTicket->event);
      clockStatus = pTicket->status;
    }

    switch (clockStatus) {
    case CoreClockSuccess:
      break;

    case CoreClockInvalidIndex:
      status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
      break;

    case CoreClockHardwareError:
      status = DfIoStatusError(ADMXRC2_FAILED);
      break;

    case CoreClockOutOfRange:
      status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
      break;

    default:
      status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
      break;
    }

    pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &pTicket->ticket);
    dfEventUninit(&pTicket->event);
  }

done:
  if (NULL != pTicket) {
    dfPoolFree(&pDevCtx->clockGenerator.ticketPool, pTicket);
  }

  if (status == DfIoStatusSuccess) {
    pIoctl->out.actualFrequency = clockWord.frequency;
  }

  return status;
}
