/*
** File: tsbuf.c  
** Project: Linux Driver Framework
** Purpose: Timestamp buffer implementation.
**
** (C) Copyright Alpha Data 2016
*/

#include "df.h"

boolean_t
dfTimestampBufferInit(
  DfTimestampBuffer* pTSBuf,
  unsigned int size)
{
  struct timeval tv;

  pTSBuf->buffer.pBuffer = dfMalloc(size * 2 * sizeof(uint64_t));
  if (NULL == pTSBuf->buffer.pBuffer) {
    return FALSE;
  }
  pTSBuf->buffer.size = size;
  dfSpinLockInit(&pTSBuf->lock);
  pTSBuf->attributes.frequency = dfTimestampGetFrequency();
  do_gettimeofday(&tv);
  pTSBuf->attributes.initTime = ((uint64_t)tv.tv_sec << 32) | (uint64_t)tv.tv_usec;
  pTSBuf->attributes.initTimestamp = dfTimestampGet();
  dfTimestampBufferReset(pTSBuf);
  return TRUE;
}

void
dfTimestampBufferUninit(
  DfTimestampBuffer* pTSBuf)
{
  dfFree(pTSBuf->buffer.pBuffer);
  pTSBuf->buffer.pBuffer = NULL;
}

boolean_t
dfTimestampBufferReset(
  DfTimestampBuffer* pTSBuf)
{
  DfSpinLockFlags f;

  if (!dfTimestampBufferEnabled(pTSBuf)) {
    return FALSE;
  }

  f = dfSpinLockGet(&pTSBuf->lock);
  pTSBuf->state.head = 0;
  pTSBuf->state.tail = 0;
  pTSBuf->state.bOverflow = FALSE;
  dfSpinLockPut(&pTSBuf->lock, f);

  return TRUE;
}

#define ENTRY_SIZE_BYTES (2 * sizeof(uint64_t))

boolean_t
dfTimestampBufferRead(
  DfTimestampBuffer* pTSBuf,
  uint32_t offset,
  uint32_t count,
  void* pBuffer,
  boolean_t bConsume)
{
  uint32_t head, tail, n, size, pos;
  uint32_t firstChunk, secondChunk;
  uint8_t* pSrcBuffer;
  DfSpinLockFlags f;

  pSrcBuffer = (uint8_t*)pTSBuf->buffer.pBuffer;
  size = pTSBuf->buffer.size;

  dfAssert(size > 0);

  f = dfSpinLockGet(&pTSBuf->lock);

  head = pTSBuf->state.head;
  tail = pTSBuf->state.tail;
  n = (tail >= head) ? tail - head : size + tail - head;

  if (offset >= n || offset + count > n) {
    dfSpinLockPut(&pTSBuf->lock, f);
    return FALSE;
  }

  pos = head + (uint32_t)offset; /* Cast safe because of previous validation step */
  if (pos >= size) {
    pos -= size;
  }

  if (pos + count > size) {
    /* Wrap around */
    firstChunk = size - pos;
    memcpy(pBuffer, pSrcBuffer + ENTRY_SIZE_BYTES * pos, ENTRY_SIZE_BYTES * firstChunk);
    secondChunk = (uint32_t)count - firstChunk; /* Cast safe due to prior validation */
    memcpy((uint8_t*)pBuffer + ENTRY_SIZE_BYTES * firstChunk, pSrcBuffer, ENTRY_SIZE_BYTES * secondChunk);
  } else {
    /* No wrap around */
    memcpy(pBuffer, pSrcBuffer + ENTRY_SIZE_BYTES * pos, ENTRY_SIZE_BYTES * count);
  }

  if (bConsume) {
    pos += (uint32_t)count; /* Cast safe because of previous validation step */
    if (pos >= size) {
      pos -= size;
    }
    pTSBuf->state.head = pos;
  }

  dfSpinLockPut(&pTSBuf->lock, f);

  return TRUE;
}
