/*
** File: linux-2.6.c  
** Project: Linux Driver Framework
** Purpose: Entry point for kernel module
**
** (C) Copyright Alpha Data 2010
*/

#include <linux/moduleparam.h>
#include <linux/stat.h>
#include <linux/pci.h>
#include <linux/types.h>

#include "df.h"
#include "dflinux.h"

extern boolean_t
dfDriverEntry(
  DfDriverObject* pDrvObj);

extern void
dfDriverExit(
  DfDriverObject* pDrvObj);

static DfDriverObject g_dfDrvObj;
static boolean_t g_bPciDriverRegistered = FALSE;

/*
** --------------------------------------------------------------------------
** PCI driver
** --------------------------------------------------------------------------
*/

static struct pci_driver driver;

static int
pciProbe(
  struct pci_dev* pPciDev,
  const struct pci_device_id* pPciId)
{
  int res = 0;
  DfDeviceObject* pDevObj = NULL;
  DfBusResources rawResources, xlatResources;
  uint8_t revision, irqPin, irqLine;
  int i;
  unsigned int numMemBar, numIoBar, numIrq;
  boolean_t bAdded = FALSE;
  boolean_t bStarted = FALSE;
  boolean_t bDevEnabled = FALSE;

  dfDebugPrint(1, ("pciProbe: entered, pPciDev=%p\n", pPciDev));

  pDevObj = (DfDeviceObject*)dfMalloc(sizeof(*pDevObj));
  if (pDevObj == NULL) {
    res = -ENOMEM;
    dfDebugPrint(0, ("*** pciProbe: Cannot allocate device object\n"));
    goto done;
  }
  pci_set_drvdata(pPciDev, (void*)pDevObj);
  pDevObj->system.pContext = NULL;
  pDevObj->system.device.pPci = pPciDev;
  pDevObj->system.pExportedInterfaces = NULL;
  pDevObj->system.pImportedInterfaces = NULL;
  pDevObj->system.pMmappableRegions = NULL;
  pDevObj->system.interfaceList.pHead = NULL;
  pDevObj->system.dma.pMapArray = NULL;
  pDevObj->system.request.bPoolValid = FALSE;
  pDevObj->system.request.bPoolNonBlockValid = FALSE;
  pDevObj->hardwareId.busType = DfBusPci;
  pDevObj->hardwareId.variant.pci.vendor = pPciDev->vendor;
  pDevObj->hardwareId.variant.pci.device = pPciDev->device;
  pDevObj->hardwareId.variant.pci.subVendor = pPciDev->subsystem_vendor;
  pDevObj->hardwareId.variant.pci.subDevice = pPciDev->subsystem_device;
  pci_read_config_byte(pPciDev, PCI_REVISION_ID, &revision);
  pDevObj->hardwareId.variant.pci.revision = revision;
  pDevObj->index = atomic_inc_return(&g_dfDrvObj.system.deviceIndex) - 1;

  if (!dfPoolInit(&pDevObj->system.request.pool, pDevObj, "clReq", DfClientRequest)) {
    res = -ENOMEM;
    dfDebugPrint(0, ("*** pciProbe: Cannot init I/O request pool\n"));
    goto done;
  }
  pDevObj->system.request.bPoolValid = TRUE;

  if (!dfPoolInit(&pDevObj->system.request.poolNonBlock, pDevObj, "clReqNB", DfLinuxClientRequestNonBlock)) {
    res = -ENOMEM;
    dfDebugPrint(0, ("*** pciProbe: Cannot init non-blocking I/O request pool\n"));
    goto done;
  }
  pDevObj->system.request.bPoolNonBlockValid = TRUE;

  if (g_dfDrvObj.deviceContextSize != 0) {
    pDevObj->system.pContext = dfMalloc(g_dfDrvObj.deviceContextSize);
    if (NULL == pDevObj->system.pContext) {
      res = -ENOMEM;
      dfDebugPrint(0, ("*** pciProbe: Cannot allocate device context, size=%lu(0x%lx) bytes\n",
		       (unsigned long)g_dfDrvObj.deviceContextSize, (unsigned long)g_dfDrvObj.deviceContextSize));
      goto done;
    }
  }

  if (NULL != g_dfDrvObj.methods.pOnAdd) {
    dfDebugPrint(1, ("pciProbe: calling OnAdd method\n"));
    if (!g_dfDrvObj.methods.pOnAdd(pDevObj)) {
      res = -ENODEV;
      goto done;
    }
    bAdded = TRUE;
  }

  res = pci_enable_device(pPciDev);
  if (res) {
    return res;
  }
  pci_set_master(pPciDev);
  bDevEnabled = TRUE;

  /* Build raw bus resources structure */
  DF_ZERO_OBJECT(&rawResources);
  numMemBar = 0;
  numIoBar = 0;
  for (i = 0; i < 6; i++) {
    resource_size_t start, size;
    unsigned long flags, memFlags;
    
    start = pci_resource_start(pPciDev, i);
    size = pci_resource_len(pPciDev, i);
    flags = pci_resource_flags(pPciDev, i);
    if (flags & IORESOURCE_IO) {
      rawResources.variant.pci.ioBar[numIoBar].base = (uint64_t)start;
      rawResources.variant.pci.ioBar[numIoBar].size = (uint64_t)size;
      rawResources.variant.pci.ioBar[numIoBar].flags = DF_PCIBARF_IO;
      numIoBar++;
    } else {
      rawResources.variant.pci.memBar[numMemBar].base = (uint64_t)start;
      rawResources.variant.pci.memBar[numMemBar].size = (uint64_t)size;
      rawResources.variant.pci.memBar[numMemBar].flags = 0;
      memFlags = flags & IORESOURCE_BITS;
      if (memFlags & PCI_BASE_ADDRESS_MEM_PREFETCH) {
        rawResources.variant.pci.memBar[numMemBar].flags |= DF_PCIBARF_PREFETCH;
      }
      if ((flags & PCI_BASE_ADDRESS_MEM_TYPE_MASK) == PCI_BASE_ADDRESS_MEM_TYPE_64) {
        i++;
      }
      numMemBar++;
    }
  }
  rawResources.variant.pci.flags = 0;
  rawResources.variant.pci.numMemBar = numMemBar;
  rawResources.variant.pci.numIoBar = numIoBar;
  numIrq = 0;
#if STRUCT_PCI_DEV_HAS_PIN
  irqPin = pPciDev->pin;
#else
  pci_read_config_byte(pPciDev, PCI_INTERRUPT_PIN, &irqPin);
#endif
  if (irqPin != 0) {
    pci_read_config_byte(pPciDev, PCI_INTERRUPT_LINE, &irqLine);
    rawResources.variant.pci.irq[0].variant.raw.intPin = irqPin;
    rawResources.variant.pci.irq[0].variant.raw.intLine = irqLine;
    numIrq++;
  }
  rawResources.variant.pci.numIrq = numIrq;

  /* Build translated bus resources structure */
  DF_ZERO_OBJECT(&xlatResources);
  xlatResources.variant.pci.flags = 0;
  for (i = 0; i < numMemBar; i++) {
    xlatResources.variant.pci.memBar[i] = rawResources.variant.pci.memBar[i];
  }
  xlatResources.variant.pci.numMemBar = numMemBar;
  for (i = 0; i < numIoBar; i++) {
    xlatResources.variant.pci.ioBar[i] = rawResources.variant.pci.ioBar[i];
  }
  xlatResources.variant.pci.numIoBar = numIoBar;
#if 0
  if (numIrq) {
    xlatResources.variant.pci.irq[0].variant.translated.system.vector = pPciDev->irq;
  }
#endif
  xlatResources.variant.pci.numIrq = numIrq;

  if (NULL != g_dfDrvObj.methods.pOnStart) {
    dfDebugPrint(1, ("pciProbe: calling OnStart method\n"));
    if (!g_dfDrvObj.methods.pOnStart(pDevObj, &rawResources, &xlatResources)) {
      res = -ENODEV;
      goto done;
    }
    bStarted = TRUE;
  }

done:
  if (res) {
    if (bDevEnabled) {
      pci_disable_device(pPciDev);
      bDevEnabled = FALSE;
    }
    if (NULL != pDevObj) {
      if (bStarted) {
        g_dfDrvObj.methods.pOnStop(pDevObj, TRUE);
        bStarted = FALSE;
      }
      if (bAdded) {
        g_dfDrvObj.methods.pOnRemove(pDevObj);
        bAdded = FALSE;
      }
      if (pDevObj->system.request.bPoolNonBlockValid) {
        dfPoolUninit(&pDevObj->system.request.poolNonBlock);	
        pDevObj->system.request.bPoolNonBlockValid = FALSE;
      }
      if (pDevObj->system.request.bPoolValid) {
        dfPoolUninit(&pDevObj->system.request.pool);	
        pDevObj->system.request.bPoolValid = FALSE;
      }
      if (NULL != pDevObj->system.pContext && g_dfDrvObj.deviceContextSize != 0) {
        dfFree(pDevObj->system.pContext);
        pDevObj->system.pContext = NULL;
      }
      dfFree(pDevObj);
      pDevObj = NULL;
    }
  }
  return res;
}

static void
pciRemove(
 struct pci_dev* pPciDev)
{
  DfDeviceObject* pDevObj;

  dfDebugPrint(1, ("pciRemove: entered\n"));

  pDevObj = (DfDeviceObject*)pci_get_drvdata(pPciDev);
  if (NULL == pDevObj) {
    dfDebugPrint(0, ("+++ pciRemove: pDevObj is NULL\n"));
    return;
  }

  if (NULL != g_dfDrvObj.methods.pOnStop) {
    dfDebugPrint(1, ("pciRemove: calling OnStop method\n"));
    g_dfDrvObj.methods.pOnStop(pDevObj, TRUE);
  }
  
  if (NULL != g_dfDrvObj.methods.pOnRemove) {
    dfDebugPrint(1, ("pciRemove: calling OnRemove method\n"));
    g_dfDrvObj.methods.pOnRemove(pDevObj);
  }

  pci_disable_device(pPciDev);

  if (pDevObj->system.request.bPoolNonBlockValid) {
    dfPoolUninit(&pDevObj->system.request.poolNonBlock);	
    pDevObj->system.request.bPoolNonBlockValid = FALSE;
  }
  if (pDevObj->system.request.bPoolValid) {
    dfPoolUninit(&pDevObj->system.request.pool);	
    pDevObj->system.request.bPoolValid = FALSE;
  }

  if (NULL != pDevObj->system.pContext && g_dfDrvObj.deviceContextSize != 0) {
    dfFree(pDevObj->system.pContext);
  }
  dfFree(pDevObj);
}

void
dfLinuxModuleExitPci(
  void)
{
  dfDebugPrint(1, ("dfLinuxModuleExitPci: entered\n"));

  if (g_bPciDriverRegistered) {
    pci_unregister_driver(&driver);
  }
  dfDriverExit(&g_dfDrvObj);
}

int
dfLinuxModuleInitPci(
  const char* pciDriverName,
  struct pci_device_id* pPciIdTable,
  boolean_t bSupportLegacyHardware)
{
  int result;
  struct pci_device_id* pId;
  unsigned int i, j;

  dfDebugPrint(1, ("dfLinuxModuleInitPci: entered\n"));

  dfLinuxCheckDataTypes();

  dfLinuxUserIfInit();

  DF_ZERO_OBJECT(&g_dfDrvObj);
  atomic_set(&g_dfDrvObj.system.deviceIndex, 0);
  g_dfDrvObj.methods.pOnAdd = NULL;
  g_dfDrvObj.methods.pOnRemove = NULL;
  g_dfDrvObj.methods.pOnStart = NULL;
  g_dfDrvObj.methods.pOnStop = NULL;
  g_dfDrvObj.methods.pOnPowerQuery = NULL;
  g_dfDrvObj.methods.pOnPowerSet = NULL;
  atomic_set(&g_dfDrvObj.system.deviceIndex, 0);

  dfDebugPrint(1, ("dfLinuxModuleInitPci: calling dfDriverEntry\n"));
  if (!dfDriverEntry(&g_dfDrvObj)) {
    return -ENODEV;
  }

  /* Find legacy hardware in the PCI ID table */
  if (bSupportLegacyHardware) {
    for (i = 0;; i++) {
      pId = &pPciIdTable[i];
      if (pId->vendor == 0 && pId->device == 0) {
        /* Found end of non-legacy hardware */
        break;
      }
    }
    for (j = i + 1;; j++) {
      pId = &pPciIdTable[j];
      pPciIdTable[i] = *pId;
      i++;
      if (pId->vendor == 0 && pId->device == 0) {
        /* Found end of legacy hardware */
        break;
      }
    }
  }

  DF_ZERO_OBJECT(&driver);
  driver.name = (char*)pciDriverName;
  driver.id_table = pPciIdTable;
  driver.probe = pciProbe;
  driver.remove = pciRemove;
  dfDebugPrint(1, ("dfLinuxModuleInitPci: registering PCI driver\n"));
  result = pci_register_driver(&driver);
  if (result != 0) {
    dfDebugPrint(0, ("*** dfLinuxModuleInitPci: failed to register PCI driver\n"));
    dfDriverExit(&g_dfDrvObj);
    return result;
  }
  g_bPciDriverRegistered = TRUE;
  return result;
}
