/*
** File: interrupt.c  
** Project: Linux Driver Framework
** Purpose: Interrupt hooking functions
**
** (C) Copyright Alpha Data 2010, 2012
*/

#include "df.h"
#include "dflinux.h"

static uint PciUseMsi = 0; /* Disallow use of Message-Signalled Interrupts (MSI) by default */
module_param(PciUseMsi, uint, S_IRUGO);
MODULE_PARM_DESC(
  PciUseMsi,
  "If nonzero and hardware is capable, driver will attempt to enable use of Message-Signalled Interrupts (MSI). Default is 0.");

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 19))
irqreturn_t
_dfLinuxIsrStub(
  int irq,
  void* pContext)
#else
irqreturn_t
_dfLinuxIsrStub(
  int irq,
  void* pContext,
  struct pt_regs* pPtRegs /* ignored */)
#endif
{
  DfInterruptObject* pInterruptObject = (DfInterruptObject*)pContext;
  boolean_t bHandled;

  dfAssert(pInterruptObject->bConnected);

  bHandled = pInterruptObject->pIsr(pInterruptObject, pInterruptObject->pContext);

  return bHandled ? IRQ_HANDLED : IRQ_NONE;
}

boolean_t
dfInterruptConnect(
  DfDeviceObject* pDevObj,
  DfInterruptObject* pInterruptObject,
  DfBusResources* pBusResources,
  DfBusType busType,
  unsigned int interruptIndex,
  DfIsr* pIsr,
  void* pContext)
{
  int res, vector;
  unsigned long flags;

  dfAssert(busType == DfBusPci);
  dfAssert(!pInterruptObject->bConnected);
  dfAssert(interruptIndex < pBusResources->variant.pci.numIrq);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18))
  flags = IRQF_SHARED;
#else
  /* SA_INTERRUPT => fast handler, so interrupts are disabled in ISR */
  flags = SA_SHIRQ | SA_INTERRUPT;
#endif

  spin_lock_init(&pInterruptObject->lock);
  pInterruptObject->pIsr = pIsr;
  pInterruptObject->pContext = pContext;
  pInterruptObject->bConnected = TRUE;
  pInterruptObject->bUsingMsi = FALSE; /* Will set true later if necessary */
  if (pDevObj->hardwareId.busType == DfBusPci && PciUseMsi) {
    /* Attempt to use MSI if a PCI device (includes PCI-X & PCI-E). */
    res = pci_enable_msi(pDevObj->system.device.pPci);
    if (!res) {
      pInterruptObject->bUsingMsi = TRUE;
    }
  }
  /* Need to get vector from pci_dev struct AFTER call to pci_enable_msi, as it may have changed */
  vector = pDevObj->system.device.pPci->irq;
  pInterruptObject->vector = vector;
  res = request_irq(vector, _dfLinuxIsrStub, flags, "adb3", pInterruptObject);
  if (res) {
    if (pDevObj->hardwareId.busType == DfBusPci && pInterruptObject->bUsingMsi) {
      pci_disable_msi(pDevObj->system.device.pPci); /* Remember to disable MSI on failure! */
      pInterruptObject->bUsingMsi = FALSE;
    }
    pInterruptObject->bConnected = FALSE;
    return FALSE;
  }
  return TRUE;
}

void
dfInterruptDisconnect(
  DfDeviceObject* pDevObj,
  DfInterruptObject* pInterruptObject)
{
  dfAssert(pInterruptObject->bConnected);
  free_irq(pInterruptObject->vector, pInterruptObject);
  if (pDevObj->hardwareId.busType == DfBusPci && pInterruptObject->bUsingMsi) {
    /* Disable use of MSI *after* calling free_irq(). */
    pci_disable_msi(pDevObj->system.device.pPci);
    pInterruptObject->bUsingMsi = FALSE;
  }
  pInterruptObject->bConnected = FALSE;
}

boolean_t
dfInterruptUsingMsi(
  DfDeviceObject* pDevObj,
  DfInterruptObject* pInterruptObject)
{
  dfAssert(pInterruptObject->bConnected);
  return pInterruptObject->bUsingMsi;
}
