/*
** File: df.h  
** Project: Linux Driver Framework
** Purpose: Defines API used by OS-independent driver code
**
** (C) Copyright Alpha Data 2010
*/

#if !defined(ADATA_DF_DF_H)
#define ADATA_DF_DF_H

#if !defined(DF_DBG_EXECUTION_CONTEXT)
# define DF_DBG_EXECUTION_CONTEXT (0)
#endif
#if !defined(DF_DBG_ALLOCATION)
# define DF_DBG_ALLOCATION (0)
#endif

#define DF_DECLARE_INLINE_FUNC(type, ident) static inline type ident

#include <asm/uaccess.h>
#include <asm/signal.h>
#include <asm/div64.h>
#include <linux/capability.h>
#include <linux/ctype.h> /* for isprint() */
#include <linux/fs.h>
#include <linux/kobject.h>
#include <linux/cdev.h>
#include <linux/interrupt.h>
#include <linux/ioctl.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/pci.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/stat.h>
#include <linux/timer.h>
#include <linux/version.h>
#include <linux/delay.h>
#include <linux/vmalloc.h>

#if defined(CONFIG_COMPAT)

# define DF_NEED_THUNK (1)
# include <linux/compat.h>
typedef compat_size_t size32_t;
typedef compat_uptr_t ptr32_t;
typedef const ptr32_t const_ptr32_t;

# define dfThunkPtr(x) compat_ptr(x)

static inline const void*
dfThunkConstPtr(
  const_ptr32_t p)
{
  return (const void*)compat_ptr((compat_uptr_t)p);
}

#else
# define DF_NEED_THUNK (0)
#endif

typedef int boolean_t;
#ifndef FALSE
# define FALSE (0)
#endif
#ifndef TRUE
# define TRUE (1)
#endif

#if !HAS_UINTPTR_T
typedef unsigned long uintptr_t;
#endif

#if !HAS_RESOURCE_SIZE_T
typedef unsigned long resource_size_t;
#endif

#define DF_HAS_USER_EVENT_HANDLE (0)

typedef atomic_t DfAtomicInt;

#include <../common/dfcommon.h>
#include <../common/dflist.h>

#define DF_FMT_I64 "%lld"
#define DF_FMT_U64 "%llu"
#define DF_FMT_H64 "%llx"
#define DF_FMT_16H64 "%016llx"

#define dfLe8ToCpu(x) x
#define dfLe16ToCpu(x) le16_to_cpu(x)
#define dfLe32ToCpu(x) le32_to_cpu(x)
#define dfLe64ToCpu(x) le64_to_cpu(x)
#define dfCpuToLe8(x) x
#define dfCpuToLe16(x) cpu_to_le16(x)
#define dfCpuToLe32(x) cpu_to_le32(x)
#define dfCpuToLe64(x) cpu_to_le64(x)

DF_DECLARE_INLINE_FUNC(uint64_t, dfDivide64By32)(
  uint64_t dividend,
  uint32_t divisor)
{
  do_div(dividend, divisor);
  return dividend;
}

DF_DECLARE_INLINE_FUNC(uint64_t, dfDivide64By32Rem)(
  uint64_t dividend,
  uint32_t divisor,
  uint32_t* pRemainder)
{
  *pRemainder = do_div(dividend, divisor);
  return dividend;
}

/* This is nasty :( but convenient */
#define dfSplitUint64(x) (unsigned long)(uint32_t)((x) >> 32U), (unsigned long)(uint32_t)(x)

static inline uint16_t
dfReadUnaligned16(
	uint16_t* p)
{
	uint8_t* p8 = (uint8_t*)p;
	uint16_t val16;

	val16  = ((uint16_t)*p8++) << 0U;
	val16 |= ((uint16_t)*p8++) << 8U;

	return dfLe16ToCpu(val16);
}

static inline uint32_t
dfReadUnaligned32(
	uint32_t* p)
{
	uint8_t* p8 = (uint8_t*)p;
	uint32_t val32;

	val32  = ((uint32_t)*p8++) << 0U;
	val32 |= ((uint32_t)*p8++) << 8U;
	val32 |= ((uint32_t)*p8++) << 16U;
	val32 |= ((uint32_t)*p8++) << 24U;

	return dfLe32ToCpu(val32);
}

static inline uint64_t
dfReadUnaligned64(
	uint64_t* p)
{
	uint8_t* p8 = (uint8_t*)p;
	uint64_t val64;

	val64  = ((uint64_t)*p8++) << 0U;
	val64 |= ((uint64_t)*p8++) << 8U;
	val64 |= ((uint64_t)*p8++) << 16U;
	val64 |= ((uint64_t)*p8++) << 24U;
	val64 |= ((uint64_t)*p8++) << 32U;
	val64 |= ((uint64_t)*p8++) << 40U;
	val64 |= ((uint64_t)*p8++) << 48U;
	val64 |= ((uint64_t)*p8++) << 56U;

	return dfLe64ToCpu(val64);
}

/*
** Attributes for functions. It would be nice if we had some kind of
** tool able parse the code and use these attributes to look for errors.
*/
#define DF_ATTR_PASSIVE_CONTEXT /* Must be called in a thread context */

/* Return value from DfOnIoctlMethod functions */
typedef int DfIoStatus;
#define DfIoStatusSuccess      0                             /* Completed successfully */
#define DfIoStatusNoMemory     (-ENOMEM)                     /* Not enough memory / resources */
#define DfIoStatusAccessDenied (-EACCES)                     /* Insufficient privileges */
#define DfIoStatusInvalid      (-EFAULT)                     /* pIoctl, inSize and/or outSize were invalid */
#define DfIoStatusInvalidCode  (-ENOTTY)                     /* IOCTL code was invald */
#define DfIoStatusCancelled    (-ECANCELED)                  /* Cancelled */
#define DfIoStatusDeferred     (-EINPROGRESS)                /* Will be completed later */
#define DfIoStatusError(x)     x                             /* Defines an API-specific error */
#define DfIoStatusIsError(x)   ((x) != DfIoStatusSuccess && (x) != DfIoStatusDeferred)

/*
** Forward declarations of structures and types exported by framework
*/

struct _DfBufferDescription;
typedef struct _DfBufferDescription DfBufferDescription;

struct _DfBusResources;
typedef struct _DfBusResources DfBusResources;

struct _DfClientObject;
typedef struct _DfClientObject DfClientObject;

struct _DfClientRequest;
typedef struct _DfClientRequest DfClientRequest;

typedef void DfCancelCallback(DfClientRequest* pRequest, DfIoStatus reason);

struct _DfDeviceObject;
typedef struct _DfDeviceObject DfDeviceObject;

struct _DfDmaBuffer;
typedef struct _DfDmaBuffer DfDmaBuffer;

struct _DfDmaMapperNode;
typedef struct _DfDmaMapperNode DfDmaMapperNode;

struct _DfDmaMapperPosition;
typedef struct _DfDmaMapperPosition DfDmaMapperPosition;

struct _DfDmaMapperSlice;
typedef struct _DfDmaMapperSlice DfDmaMapperSlice;

typedef void DfDmaMapperCallback(DfDeviceObject* pDevObj, unsigned int mapIndex, DfDmaMapperNode* pTable, unsigned int tableLength, DfDmaMapperPosition* pProgress, void* pContext);

struct _DfDpc;
typedef struct _DfDpc DfDpc;

typedef void DfDpcFunction(DfDpc* pDpc, void* pContext, void* pArg);

struct _DfDriverObject;
typedef struct _DfDriverObject DfDriverObject;

struct _DfEvent;
typedef struct _DfEvent DfEvent;

struct _DfMutex;
typedef struct _DfMutex DfMutex;

struct _DfExportedInterface;
typedef struct _DfExportedInterface DfExportedInterface;

struct _DfImportedInterface;
typedef struct _DfImportedInterface DfImportedInterface;

struct _DfInterface;
typedef struct _DfInterface DfInterface;

struct _DfInterfaceClass;
typedef struct _DfInterfaceClass DfInterfaceClass;

struct _DfInterfaceMethods;
typedef struct _DfInterfaceMethods DfInterfaceMethods;

struct _DfInterruptObject;
typedef struct _DfInterruptObject DfInterruptObject;

typedef boolean_t DfIsr(DfInterruptObject* pInterruptObj, void* pContext);

struct _DfMappedIoRegion;
typedef struct _DfMappedIoRegion DfMappedIoRegion;

typedef void* DfMemoryHandle;

struct _DfTimestampBuffer;
typedef struct _DfTimestampBuffer DfTimestampBuffer;

/*
** Encapsulates the notion of an address space and a pointer in that address space.
** In Linux there is only memory space.
*/
typedef struct {
  uint8_t* p;
} DfRegPtr;

struct _DfMmappableRegion;
typedef struct _DfMmappableRegion DfMmappableRegion;

struct _DfPool;
typedef struct _DfPool DfPool;

typedef spinlock_t DfSpinLock;

typedef unsigned long DfSpinLockFlags;

struct _DfTimer;
typedef struct _DfTimer DfTimer;

#include <../common/dfentry.h>

/*
** Device structures
*/

#define DFPOOL_NAME_MAX_LENGTH (32)
struct _DfPool {
  struct {
    char name[DFPOOL_NAME_MAX_LENGTH];
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18))
    struct kmem_cache* pCache;
#else
    kmem_cache_t* pCache;
#endif
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
    size_t itemSize;
#endif
  } system;
};

#define DF_PCIBARF_IO       (0x1U)
#define DF_PCIBARF_PREFETCH (0x8U)
struct _DfPciBar {
	uint64_t base;
	uint64_t size;
	uint32_t flags;
};
typedef struct _DfPciBar DfPciBar;

/* This datatype must be treated as being opaque */
struct _DfPciIrqLine {
  boolean_t bMessageSignalled;
  boolean_t bLevelSensitive;
  union {
    struct {
      uint8_t intPin;
      uint8_t intLine;
    } raw;
    struct {
      struct { /* This struct must be consider opaque */
        int dummy; /* Dummy member to avoid empty struct */
      } system;
    } translated;
  } variant;
};
typedef struct _DfPciIrqLine DfPciIrqLine;

#define DF_PCIBUSRESOURCESF_EXPROMVALID (0x2U)
struct _DfPciBusResources {
  uint32_t flags;
  unsigned int numMemBar;
  DfPciBar memBar[6];
  DfPciBar expRomBar;
  unsigned int numIoBar;
  DfPciBar ioBar[6];
  unsigned int numIrq;
  DfPciIrqLine irq[4];
};
typedef struct _DfPciBusResources DfPciBusResources;

struct _DfBusResources {
  union {
    DfPciBusResources pci;
  } variant;
};

struct _DfPciHardwareId {
  uint16_t vendor, device;
  uint16_t subVendor, subDevice;
  uint8_t revision;
};
typedef struct _DfPciHardwareId DfPciHardwareId;

struct _DfUnknownHardwareId {
  char id[48];
};
typedef struct _DfUnknownHardwareId DfUnknownHardwareId;

typedef enum _DfBusType {
  DfBusInvalid = 0,
  DfBusUnknown = 1,
  DfBusPci = 2
} DfBusType;

struct _DfHardwareId {
  DfBusType busType;
  union {
    DfUnknownHardwareId unknown;
    DfPciHardwareId pci;
  } variant;
};
typedef struct _DfHardwareId DfHardwareId;

struct _DfClientRequest {
  struct {
    DfClientObject* pClObj;
    DfCancelCallback* pCancelCallback;
    boolean_t bCancel;
    boolean_t bCompleted;
    DfIoStatus status;
    wait_queue_head_t* pWaitQueue;
    DfListNode node;
    void* pContext; /* Client code can get/set value via dfRequest{Get|Set}Context */
  } system;
};

struct _DfLinuxClientRequestNonBlock;
typedef struct _DfLinuxClientRequestNonBlock DfLinuxClientRequestNonBlock;

struct _DfClientObject {
  struct { /* Client code must not touch this struct */
    DfSpinLock lock;
    struct file* pFile;
    DfDeviceObject* pDevObj;
    boolean_t bPassive; /* TRUE if opened in passive mode */
    DfInterface* pInterface;
    struct {
      /* List of client requests that have a cancel callback */
      unsigned int count;
      DfList list;
    } requests;
    struct {
      /* Information about non-blocking client requests */
      int state;
      DfLinuxClientRequestNonBlock* pHead; /* Currently, at most one allowed at any moment */
      wait_queue_head_t pollWaitQueue;     /* Used to implement poll() file_operations method */
    } nonblock;
    void* pContext; /* Client code can get/set value via dfClient{Get|Set}Context */
  } system;
};

static inline void*
dfClientGetContext(
 DfClientObject* pClObj)
{
  return pClObj->system.pContext;
}

static inline void
dfClientSetContext(
 DfClientObject* pClObj,
 void* pContext)
{
  pClObj->system.pContext = pContext;
}

struct _DfLinuxDmaMapperContext;
typedef struct _DfLinuxDmaMapperContext DfLinuxDmaMapperContext;

struct _DfDeviceObject {
  struct { /* Client code must not touch this struct */
    union { /* hardwareId.busType qualifies members */
      struct pci_dev* pPci;
    } device;
    struct {
      unsigned int maxMapRegister;
      size_t maxMapSize;
      unsigned int numMap;
      DfLinuxDmaMapperContext* pMapArray;
    } dma;
    DfExportedInterface* pExportedInterfaces;
    DfImportedInterface* pImportedInterfaces;
    DfMmappableRegion* pMmappableRegions;
    struct {
      DfInterface* pHead;
    } interfaceList;
    struct {
      DfPool pool;
      DfPool poolNonBlock;
      boolean_t bPoolValid;
      boolean_t bPoolNonBlockValid;
    } request;
    void* pContext;/* Client code can get/set any value here via dfDevice{Get|Set}Context */
  } system;

  DfDriverObject* pDrvObj;
  DfHardwareId hardwareId;
  unsigned int index;
};

static inline void
dfDeviceSetContext(
  DfDeviceObject* pDevObj,
  void* pContext)
{
  pDevObj->system.pContext = pContext;
}

static inline void*
dfDeviceGetContext(
  DfDeviceObject* pDevObj)
{
  return pDevObj->system.pContext;
}

/*
** Driver structures
*/

typedef boolean_t
DfOnAddMethod(
  DfDeviceObject* pDevObj);

typedef void
DfOnRemoveMethod(
  DfDeviceObject* pDevObj);

typedef boolean_t
DfOnStartMethod(
  DfDeviceObject* pDevObj,
  DfBusResources* pRawResources,
  DfBusResources* pTranslatedResources);

typedef void
DfOnStopMethod(
	DfDeviceObject* pDevObj,
  boolean_t bCanTouchHardware);

typedef boolean_t /* bVeto */
DfOnPowerQueryMethod(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int requestedState,
  boolean_t bIsShutdown);

typedef void
DfOnPowerSetMethod(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int newState,
  boolean_t bIsShutdown);

typedef void
DfOnCleanupMethod(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj);

typedef DfIoStatus
DfOnOpenMethod(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  DfInterface* pInterface,
  boolean_t bPrivileged);

typedef void
DfOnCloseMethod(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj);

typedef DfIoStatus
DfOnIoctlMethod(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  unsigned int ioctlCode,
  void* pIoctl,
  unsigned int inSize,
  unsigned int outSize);

typedef enum _DfQueryMapStatus {
  DfQueryMapSuccess = 0,
  DfQueryMapInvalidIndex,
  DfQueryMapInvalidRegion,
  DfQueryMapGeneralFailure 
} DfQueryMapStatus;

struct _DfDriverObject {
  unsigned int deviceContextSize;
  
  struct {
    DfOnAddMethod* pOnAdd;
    DfOnRemoveMethod* pOnRemove;
    DfOnStartMethod* pOnStart;
    DfOnStopMethod* pOnStop;
		DfOnPowerQueryMethod* pOnPowerQuery;
		DfOnPowerSetMethod* pOnPowerSet;
  } methods;
  
  /* Private */
  struct {
    struct pci_driver* pPciDrv;
    atomic_t deviceIndex;
  } system;
};

/*
** Debug messages
*/

#if DF_DBG_BUILD
extern int g_dfDebugLevel;

extern void
_dfDebugPrint(
  const char* fmt, ...);

# define dfDebugLevel(LEVEL) ((LEVEL) <= g_dfDebugLevel)

# define dfDebugPrint(LEVEL, ARGS) \
do { if (dfDebugLevel(LEVEL)) _dfDebugPrint ARGS; } while (0)

# define dfDebugTrace(LEVEL) \
do { if (dfDebugLevel(LEVEL)) _dfDebugPrint("reached function '%s', %s:%u\n", __FUNCTION__, __FILE__, __LINE__); } while (0)

extern void
dfDebugDumpMemory(
  int level,
  unsigned int indent,
  unsigned int nColumn,
  unsigned int radix,
  boolean_t bAlign,
  const void* pBuffer,
  size_t n);

# define dfAssert(expr) do { if (!(expr)) _dfDebugPrint("*** function '%s' %s:%u assertion failed: %s\n", __FUNCTION__, __FILE__, __LINE__, #expr); } while (0)

# define dfBugCheck(ARGS) _dfDebugPrint ARGS;

#else

# define dfAssert(expr)
# define dfBugCheck(ARGS)
# define dfDebugDumpMemory(LEVEL, IDENT, NCOLUMN, RADIX, BALIGN, PBUFFER, N);
# define dfDebugPrint(LEVEL, ARGS)
# define dfDebugLevel(LEVEL) FALSE

#endif

#if DF_DBG_BUILD && DF_DBG_EXECUTION_CONTEXT

# define dfAssertPassiveContext()     /* TO DO */
# define dfAssertNotHoldingSpinLock() /* TO DO */

#else

# define dfAssertPassiveContext()
# define dfAssertNotHoldingSpinLock()

#endif

/*
** Import and export of direct-call interfaces
*/

#define DF_MAX_INTERFACE_SIZE (1024)
extern boolean_t
dfImportInterface(
  DfDeviceObject* pDevObj,
  const char* pName,
  unsigned int size,
  void* pInterfaceStruct,
  void** ppInterfaceContext);

extern boolean_t
dfExportInterface(
  DfDeviceObject* pDevObj,
  const char* pName,
  unsigned int size,
  void* pInterfaceStruct,
  void* pInterfaceContext);

extern boolean_t
dfReleaseInterface(
  DfDeviceObject* pDevObj,
  const char* pName);

extern boolean_t
dfUnexportInterface(
  DfDeviceObject* pDevObj,
  const char* pName);

/*
** Mapping and unmapping I/O space for access by driver (NOT user space)
*/

extern void*
dfIoSpaceMap(
  DfDeviceObject* pDevObj,
  DfBusResources* pResources,
  unsigned int barIndex,
  uint64_t offset,
  uint64_t length,
  DfMemoryHandle* pHandle);

extern boolean_t
dfIoSpaceUnmap(
  DfDeviceObject* pDevObj,
  void* pKernelAddress);

/*
** Memory space register access
*/

/*
** PCI Memory register access functions; the user can assume that they perform
** conversion to little-endian endian format if necessary.
*/

DF_DECLARE_INLINE_FUNC(void, dfPciMemWrite32)(
  DfMemoryHandle h,
  uint32_t* p,
  uint32_t d32)
{
  iowrite32(d32, p);
}

DF_DECLARE_INLINE_FUNC(uint32_t, dfPciMemRead32)(
  DfMemoryHandle h,
  const uint32_t* p)
{
  return (uint32_t)ioread32((uint32_t*)p);
}

DF_DECLARE_INLINE_FUNC(void, dfPciMemWrite16)(
  DfMemoryHandle h,
  uint16_t* p,
  uint16_t d16)
{
  iowrite16(d16, p);
}

DF_DECLARE_INLINE_FUNC(uint16_t, dfPciMemRead16)(
  DfMemoryHandle h,
  const uint16_t* p)
{
  return (uint16_t)ioread16((uint16_t*)p);
}

DF_DECLARE_INLINE_FUNC(void, dfPciMemWrite8)(
  DfMemoryHandle h,
  uint8_t* p,
  uint8_t d8)
{
  iowrite8(d8, p);
}

DF_DECLARE_INLINE_FUNC(uint8_t, dfPciMemRead8)(
  DfMemoryHandle h,
  const uint8_t* p)
{
  return (uint8_t)ioread8((uint8_t*)p);
}

/*
** Delaying execution
*/

#define DFLINUX_USE_JIFFIES_64 (0)

#if DFLINUX_USE_JIFFIES_64
typedef u64 DfTime;
#else
typedef unsigned long DfTime;
#endif

/* Busy delays CPU for specified number of microseconds */
#define dfDelayMicroseconds(microseconds) udelay(microseconds)

static inline DfTime
dfMicrosecondsToTime(
  unsigned long microseconds)
{
  return usecs_to_jiffies(microseconds);
}

static inline unsigned long
dfTimeToMicroseconds(
  unsigned long ticks)
{
  return jiffies_to_usecs(ticks);
}

extern boolean_t /* TRUE if wait was interrupted */
dfDelayThreadUntil(
  DfTime when);

extern boolean_t /* TRUE if wait was interrupted */
dfDelayThreadFor(
  DfTime period);

#if DFLINUX_USE_JIFFIES_64
static inline DfTime
dfTimeGet(
  void)
{
  return get_jiffies_64();
}
#else
static inline DfTime
dfTimeGet(
  void)
{
  return jiffies;
}
#endif

#if DFLINUX_USE_JIFFIES_64
# define dfTimeGreaterEqual(a, b) ((a) >= (b))
# define dfTimeGreater(a, b) ((a) > (b))
# define dfTimeLesserEqual(a, b) ((a) <= (b))
# define dfTimeLesser(a, b) ((a) < (b))
#else
# define dfTimeGreaterEqual(a, b) time_after_eq((a), (b))
# define dfTimeGreater(a, b) time_after((a), (b))
# define dfTimeLesserEqual(a, b) !time_after((a), (b))
# define dfTimeLesser(a, b) !time_after_eq((a), (b))
#endif

/*
** Critical sections and mutual exclusion
*/

#define dfSpinLockInit(pLock) spin_lock_init(pLock)

static inline DfSpinLockFlags
dfSpinLockGet(
  DfSpinLock* pLock)
{
  DfSpinLockFlags f;

  spin_lock_irqsave(pLock, f);

  return f;
}

#define dfSpinLockPut(pLock, flags) spin_unlock_irqrestore(pLock, flags)

/*
** Kernel event waiting and signalling
*/

struct _DfEvent {
  struct { /* Client code must not touch */
    spinlock_t lock;
    wait_queue_head_t wq;
    int state; /* 0 => idle, 1 => waiting, 2 => signalled */
  } system;
};

static inline void
dfEventInit(
  DfEvent* pEvent)
{
  spin_lock_init(&pEvent->system.lock);
  init_waitqueue_head(&pEvent->system.wq);
  pEvent->system.state = 0;
}

static inline void
dfEventUninit(
  DfEvent* pEvent)
{
  /* Nothing to do */
}

static inline void
dfEventWait(
  DfEvent* pEvent)
{
  DfSpinLockFlags f;
  wait_queue_t wait;

  init_wait(&wait);
  add_wait_queue(&pEvent->system.wq, &wait);
  set_current_state(TASK_UNINTERRUPTIBLE);
  f = dfSpinLockGet(&pEvent->system.lock);
  if (pEvent->system.state == 2) {
    dfSpinLockPut(&pEvent->system.lock, f);
  } else {
    pEvent->system.state = 1;
    dfSpinLockPut(&pEvent->system.lock, f);
    schedule();
  }
  set_current_state(TASK_RUNNING);
  remove_wait_queue(&pEvent->system.wq, &wait);
}

/* Returns TRUE if wait was interrupted by a signal, otherwise FALSE */
static inline boolean_t
dfEventWaitInterruptible(
  DfEvent* pEvent)
{
  DfSpinLockFlags f;
  wait_queue_t wait;

  init_wait(&wait);
  add_wait_queue(&pEvent->system.wq, &wait);
  set_current_state(TASK_INTERRUPTIBLE);
  f = dfSpinLockGet(&pEvent->system.lock);
  if (pEvent->system.state == 2) {
    dfSpinLockPut(&pEvent->system.lock, f);
  } else {
    pEvent->system.state = 1;
    dfSpinLockPut(&pEvent->system.lock, f);
    schedule();
  }
  set_current_state(TASK_RUNNING);
  remove_wait_queue(&pEvent->system.wq, &wait);
  return signal_pending(current);
}

static inline void
dfEventSignal(
  DfEvent* pEvent)
{
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pEvent->system.lock);
  if (pEvent->system.state == 1) {
    wake_up(&pEvent->system.wq);
  }
  pEvent->system.state = 2;
  dfSpinLockPut(&pEvent->system.lock, f);
}

/*
** Kernel mutex
*/

struct _DfMutex {
  struct semaphore sem;
};

static inline void
dfMutexInit(
  DfMutex* pMutex)
{
  sema_init(&pMutex->sem, 1);
}

static inline void
dfMutexUninit(
  DfMutex* pMutex)
{
  /* Nothing to do */
}

static inline void
dfMutexAcquire(
  DfMutex* pMutex)
{
  down(&pMutex->sem);
}

static inline boolean_t
dfMutexAcquireInterruptible(
  DfMutex* pMutex)
{
  return (down_interruptible(&pMutex->sem) != 0) ? TRUE : FALSE;
}

static inline void
dfMutexRelease(
  DfMutex* pMutex)
{
  up(&pMutex->sem);
}

/*
** Timers
*/

typedef void DfTimerCallback(DfTimer* pTimer, void* pTimerContext, void* pCallbackContext);

extern void _dfLinuxTimerStub(unsigned long context);

struct _DfTimer {
  /* Client code must not touch */
  struct {
    struct timer_list timer;
    DfTimerCallback* pCallback;
    void* pTimerContext;
    void* pCallbackContext;
  } system;
};

static inline void
dfTimerInit(
  DfTimer* pDfTimer,
  void* pTimerContext)
{
  pDfTimer->system.pTimerContext = pTimerContext;
  init_timer(&pDfTimer->system.timer);
  pDfTimer->system.timer.function = _dfLinuxTimerStub;
  pDfTimer->system.timer.data = (unsigned long)pDfTimer;
}

static inline void
dfTimerUninit(
  DfTimer* pDfTimer)
{
  /* Nothing to do */
}

static inline void
dfTimerSchedule(
  DfTimer* pDfTimer,
  DfTime expires,
  DfTimerCallback* pCallback,
  void* pCallbackContext)
{
  pDfTimer->system.timer.expires = (unsigned long)expires;
  pDfTimer->system.pCallback = pCallback;
  pDfTimer->system.pCallbackContext = pCallbackContext;
  add_timer(&pDfTimer->system.timer);
}

static inline boolean_t
dfTimerCancel(
  DfTimer* pDfTimer)
{
  return del_timer(&pDfTimer->system.timer) ? TRUE : FALSE;
}

/*
** Registering interfaces to user-mode applications
*/

struct _DfInterfaceClass {
  struct {
    char name[32];
    dev_t devNum;
    int numInterface;
#if USE_CLASS_SIMPLE
    struct class_simple* pClass;
#else
    struct class* pClass;
#endif
  } system;
};

struct _DfInterfaceMethods {
  DfOnCleanupMethod* pOnCleanup;
  DfOnCloseMethod* pOnClose;
  DfOnIoctlMethod* pOnIoctl;
#if DF_NEED_THUNK
  DfOnIoctlMethod* pOnIoctlThunk;
#endif
  DfOnOpenMethod* pOnOpen;
};

struct _DfInterface {
  struct {
    DfInterface* pNext;
    DfDeviceObject* pDevObj;
    DfInterfaceMethods methods;
    boolean_t bEnabled;
    DfInterfaceClass* pClass;
    struct cdev cdev;
    int index;
#if USE_CLASS_SIMPLE
    struct class_device* pDevice;
#else
    struct device* pDevice;
#endif
    /* Client code can get/set this value via DfInterface{Get|Set}Context */
    void* pContext;
  } system;
};

extern boolean_t
dfInterfaceClassRegister(
  DfDriverObject* pDrvObj,
  const char* pClassName,
  DfInterfaceClass** ppClass);

extern void
dfInterfaceClassUnregister(
  DfInterfaceClass* pClass);

extern boolean_t
dfInterfaceRegister(
  DfDeviceObject* pDevObj,
  DfInterface* pInterface,
  DfInterfaceClass* pClass,
  const DfInterfaceMethods* pMethods);

extern boolean_t
dfInterfaceUnregister(
  DfDeviceObject* pDevObj,
  DfInterface* pInterface);

extern boolean_t
dfInterfaceEnable(
  DfDeviceObject* pDevObj,
  DfInterface* pInterface,
  boolean_t bEnable);

static inline void*
dfInterfaceGetContext(
  DfInterface* pInterface)
{
  return pInterface->system.pContext;
}

static inline void
dfInterfaceSetContext(
  DfInterface* pInterface,
  void* pContext)
{
  pInterface->system.pContext = pContext;
}

/*
** Handling of I/O requests
*/

/* Returns TRUE if cancel routine was cleared OK, otherwise FALSE */
extern boolean_t
dfRequestClearCallback(
  DfClientRequest* pClientRequest);

/* Indicate that a client request has been completed */
extern void
dfRequestComplete(
  DfClientRequest* pClientRequest,
  DfIoStatus status);

/* Returns TRUE if cancel routine was set OK, otherwise FALSE */
extern boolean_t
dfRequestSetCallback(
  DfClientRequest* pClientRequest,
  DfCancelCallback* pCallback);

static inline void*
dfRequestGetContext(
  DfClientRequest* pClientRequest)
{
  return pClientRequest->system.pContext;
}

static inline void
dfRequestSetContext(
  DfClientRequest* pClientRequest,
  void* pContext)
{
  pClientRequest->system.pContext = pContext;
}

#define dfRequestPending(pClientRequest) /* Do nothing */

/*
** Accessing driver parameters
*/

typedef enum _DfParameterStatus {
    DfParameterSuccess = 0,     /* Success */
    DfParameterNotFound,        /* A parameter named 'pKeyName' was not found */
    DfParameterNoMemory,        /* Couldn't allocate memory complete operation */
    DfParameterInvalid,         /* Arguments were invalid */
    DfParameterGeneralFailure   /* All other (unexpected) failures */
} DfParameterStatus;

extern DfParameterStatus
dfParameterGetUint32(
    DfDriverObject* pDrvObj,
    const char* pKeyName,
    uint32_t* pValue);

extern DfParameterStatus
dfParameterGetBoolean(
    DfDriverObject* pDrvObj,
    const char* pKeyName,
    boolean_t* pValue);

/*
** pBufferNeeded   bufferSize    pBufferNeeded          Purpose
** -------------------------------------------------------------------------------------------------------------------------------
** NULL            0             NULL                   tests if parameter exists
** NULL            0             non-NULL               tests if parameter exists & returns buffer size required
** NULL            >0            (any)                  (illegal, status always DfParameterInvalid)
** non-NULL        0             (any)                  (illegal, status always DfParameterInvalid)
** non-NULL        >0            NULL                   tests if parameter exists & returns parameter value
** non-NULL        >0            non-NULL               tests if parameter exists & returns parameter value + buffer size required
*/
extern DfParameterStatus
dfParameterGetString(
    DfDriverObject* pDrvObj,
    const char* pKeyName,
    size_t bufferSize,
    size_t* pBufferNeeded,
    char* pBuffer);

/* Allocates buffer to hold value; caller's responsibility to free using dfFree */
extern DfParameterStatus
dfParameterGetStringAlloc(
    DfDriverObject* pDrvObj,
    const char* pKeyName,
    char** ppBuffer);

/*
** Allocating and freeing memory
*/

#if DF_DBG_BUILD && DF_DBG_ALLOCATION

extern void
dfLinuxRecordKMalloc(
  void* p,
  size_t n,
  int flags,
  const char* pFile,
  int line);

extern void
dfLinuxRecordKFree(
  void* p);

extern void
dfLinuxRecordVMalloc(
  void* p,
  size_t n,
  const char* pFile,
  int line);

extern void
dfLinuxRecordVFree(
  void* p);

extern void
dfLinuxReportLeaks(
  void);

#endif

static inline void*
dfKMalloc(
  size_t n,
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  int flags,
  const char* pFile,
  int line)
#else
  int flags)
#endif
{
  void* p;

  p = kmalloc(n, flags);
  if (NULL != p) {
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
    dfLinuxRecordKMalloc(p, n, flags, pFile, line);
#endif
    memset(p, 0, n);
  }
  return p;
}

static inline void
dfKFree(
  void* p)
{
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  dfLinuxRecordKFree(p);
#endif
  kfree(p);
}

static inline void*
dfVMalloc(
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  size_t n,
  const char* pFile,
  int line)
#else
  size_t n)
#endif
{
  void* p;

  p = vmalloc(n);
  if (NULL != p) {
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
    dfLinuxRecordVMalloc(p, n, pFile, line);
#endif
  }
  return p;
}

static inline void
dfVFree(
  void* p)
{
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  dfLinuxRecordVFree(p);
#endif
  vfree(p);
}

#if DF_DBG_BUILD && DF_DBG_ALLOCATION
# define dfMalloc(n) dfKMalloc(n, GFP_KERNEL, __FILE__, __LINE__)
# define dfMallocThread(n) dfVMalloc(n, __FILE__, __LINE__)
#else
# define dfMalloc(n) dfKMalloc(n, GFP_KERNEL)
# define dfMallocThread(n) dfVMalloc(n)
#endif

#define dfFree(p) dfKFree(p)
#define dfFreeThread(p) dfVFree(p)

/*
** String handling
*/

static inline void
dfStrCpy(
  char* pDst,
  size_t charSize,
  const char* pSrc)
{
  dfAssert(charSize > 1);
  charSize--;
  while (charSize) {
    *pDst++ = *pSrc++;
    charSize--;
  }
  *pDst = '\0';
}

static inline void
dfStrCat(
  char* pDst,
  size_t charSize,
  const char* pSrc)
{
  size_t n;

  n = strlen(pDst);
  pDst += n;
  dfAssert(n < charSize);
  charSize -= n;
  dfAssert(charSize > 1);
  charSize--;
  while (charSize) {
    *pDst++ = *pSrc++;
    charSize--;
  }
  *pDst = '\0';
}

#define dfSPrintF snprintf

/* 
** Data copying
*/

static inline void
dfZeroMemory(
  void* pDst,
  size_t nBytes)
{
  dfAssert(nBytes == 0 || NULL != (pDst)); \
  memset(pDst, 0, nBytes);
}

static inline void
dfSetMemory8(
  void* pDst,
  size_t nBytes,
  uint8_t val8)
{
  dfAssert(nBytes == 0 || NULL != (pDst)); \
  memset(pDst, val8, nBytes);
}

#define DF_ZERO_OBJECT(pDst) dfZeroMemory(pDst, sizeof(*(pDst)))

#define DF_SET_OBJECT8(pDst, val8) dfSetMemory8(pDst, sizeof(*(pDst)), val8)


#define dfCopyKK(pDst, pSrc, nBytes) memmove(pDst, pSrc, nBytes)

/* This is nasty :( */
#define DF_USER_SPACE_BEGIN(unique)
#define DF_USER_SPACE_EXCEPT(unique) goto _dfBufferAccessSkip##unique; _dfBufferAccessError##unique:
#define DF_USER_SPACE_END(unique) _dfBufferAccessSkip##unique:

/* Must be enclosed by DF_USER_SPACE_BEGIN ... DF_USER_SPACE_EXCEPT */
/* User-space to I/O memory copy */
#define dfCopyUI(pDst, pSrc, n, unique) dfCopyUK(pDst, pSrc, n, unique)

/* Must be enclosed by DF_USER_SPACE_BEGIN ... DF_USER_SPACE_EXCEPT */
/* I/O memory to user-space copy */
#define dfCopyIU(pDst, pSrc, n, unique) dfCopyKU(pDst, pSrc, n, unique)

/* Must be enclosed by DF_USER_SPACE_BEGIN ... DF_USER_SPACE_EXCEPT */
#define dfCopyUK(pDst, pSrc, n, unique)			\
do {						\
  if (copy_from_user(pDst, pSrc, n)) {		\
    goto _dfBufferAccessError##unique;	\
  }						\
} while (0)

/* Must be enclosed by DF_USER_SPACE_BEGIN ... DF_USER_SPACE_EXCEPT */
#define dfCopyKU(pDst, pSrc, n, unique)			\
do {						\
  if (copy_to_user(pDst, pSrc, n)) {		\
    goto _dfBufferAccessError##unique;	\
  }						\
} while(0)

/* Must be enclosed by DF_USER_SPACE_BEGIN ... DF_USER_SPACE_EXCEPT */
#define dfUserBufferValidate(pBuffer, length, bWrite, unique) \
do {									\
  if (!access_ok(bWrite ? VERIFY_WRITE : VERIFY_READ, pBuffer, length)) { \
    goto _dfBufferAccessError##unique;				\
  }									\
} while(0)

/*
** Buffer descriptions
*/

typedef enum _DfDescriptionResult {
  DfDescriptionSuccess = 0,
  DfDescriptionUnexpectedError = 1,
  DfDescriptionNullPointer = 2,
  DfDescriptionInvalidBuffer = 3,
  DfDescriptionNoMemory = 4
} DfDescriptionResult;

struct _DfBufferDescription {
  const void* pAddress;
  size_t length;

  /* The members of the following array are private */
  struct {
    boolean_t bUserMode; /* TRUE => pAddress is a user-mode virtual address */
    boolean_t bLocked;   /* TRUE => buffer is locked-down in memory */
    size_t numRegion;    /* Number of regions */
    size_t numPage;      /* Total number of pages in all regions */
    struct page** region[1]; /* Variable length array of regions */
  } system;
};

extern DfDescriptionResult
dfBufferDescriptionCreateUser(
  const void* pUserAddress,
  size_t length,
  DfBufferDescription** ppDescription);

extern void
dfBufferDescriptionDelete(
  DfBufferDescription* pDescription);

extern boolean_t
dfBufferDescriptionLock(
  DfBufferDescription* pDescription);

extern void
dfBufferDescriptionUnlock(
  DfBufferDescription* pDescription);

/*
** Registering and unregistering user-space mappable regions
*/

typedef enum _DfMmappableStatus {
  DfMmappableSuccess = 0,     /* Success */
  DfMmappableNoMemory,        /* Couldn't allocate memory complete operation */
  DfMmappableInvalidRegion,   /* Arguments were invalid */
  DfMmappableRegionTooLarge,  /* Region is too large */
  DfMmappableGeneralFailure   /* All other (unexpected) failures */
} DfMmappableStatus;

extern DfMmappableStatus
dfMmappableRegionRegister(
  DfDeviceObject* pDevObj,
  uint32_t regionTag,
  boolean_t bIsMemory,
  void* pKernelBase,
  size_t kernelSize,
  uint64_t translatedBase,  /* ignored if bIsMemory is TRUE */
  uint64_t translatedSize); /* ignored if bIsMemory is TRUE */

extern DfMmappableStatus
dfMmappableRegionUnregister(
  DfDeviceObject* pDevObj,
  uint32_t regionTag);

extern DfMmappableRegion*
dfMmappableRegionFind(
	DfDeviceObject* pDevObj,
  uint32_t regionTag);

/*
** Pools
*/

#define dfPoolInit(pPool, pDevObj, name, type) _dfPoolInit(pPool, pDevObj, name, sizeof(type))

#if DF_DBG_BUILD && DF_DBG_ALLOCATION

extern void
dfLinuxRecordPoolAlloc(
  void* p,
  DfPool* pPool,
  const char* pFile,
  int line);

extern void
dfLinuxRecordPoolFree(
  void* p);

#endif

static inline void*
_dfPoolAlloc(
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  size_t itemSize,
  DfPool* pPool,
  const char* pFile,
  int line)
#else
  DfPool* pPool)
#endif
{
  void* p;

#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  dfAssert(pPool->system.itemSize >= itemSize);
#endif
  p = kmem_cache_alloc(pPool->system.pCache, GFP_KERNEL);
  if (NULL == p) {
    return NULL;
  }
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  dfLinuxRecordPoolAlloc(p, pPool, pFile, line);
#endif
  return p;
}

#if DF_DBG_BUILD && DF_DBG_ALLOCATION
# define dfPoolAlloc(pPool, type) ((type*)_dfPoolAlloc(sizeof(type), pPool, __FILE__, __LINE__))
#else
# define dfPoolAlloc(pPool, type) ((type*)_dfPoolAlloc(pPool))
#endif

static inline void
dfPoolFree(
  DfPool* pPool,
  void* pObject)
{
#if DF_DBG_BUILD && DF_DBG_ALLOCATION
  dfAssert(NULL != pObject);
  dfLinuxRecordPoolFree(pObject);
#endif
  kmem_cache_free(pPool->system.pCache, pObject);
}

extern boolean_t
_dfPoolInit(
  DfPool* pPool,
  DfDeviceObject* pDevObj,
  const char* pName,
  size_t objectSize);

extern void
dfPoolUninit(
  DfPool* pPool);

/*
** Interrupt handling
*/

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 19))
extern irqreturn_t _dfLinuxIsrStub(int irq, void* pContext);
#else
extern irqreturn_t _dfLinuxIsrStub(int irq, void* pContext, struct pt_regs* pPtRegs);
#endif

struct _DfInterruptObject {
  DfSpinLock lock;
  boolean_t bConnected;
  boolean_t bUsingMsi;
  int vector;
  DfIsr* pIsr;
  void* pContext;
};

extern boolean_t
dfInterruptConnect(
  DfDeviceObject* pDevObj,
  DfInterruptObject* pInterruptObject,
  DfBusResources* pBusResources,
  DfBusType busType,
  unsigned int interruptIndex,
  DfIsr* pIsr,
  void* pContext);

extern void
dfInterruptDisconnect(
  DfDeviceObject* pDevObj,
  DfInterruptObject* pInterruptObject);

extern boolean_t
dfInterruptUsingMsi(
  DfDeviceObject* pDevObj,
  DfInterruptObject* pInterruptObject);

static inline DfSpinLockFlags
dfInterruptSpinLockGet(
  DfInterruptObject* pInterruptObject)
{
  DfSpinLockFlags f;

  spin_lock_irqsave(&pInterruptObject->lock, f);

  return f;
}

static inline void
dfInterruptSpinLockPut(
  DfInterruptObject* pInterruptObject,
  DfSpinLockFlags flags)
{
  spin_unlock_irqrestore(&pInterruptObject->lock, flags);
}

#define dfIsrSpinLockGet(pInterruptObject) dfInterruptSpinLockGet(pInterruptObject)
#define dfIsrSpinLockPut(pInterruptObject, flags) dfInterruptSpinLockPut(pInterruptObject, flags)

/*
** User-mode event handling
*/

#if DF_HAS_USER_EVENT_HANDLE
extern boolean_t
dfUserEventGet(
  user_event_handle_t hUserEvent,
  DfUserEvent** ppUserEvent);

extern void
dfUserEventPut(
  DfUserEvent* pUserEvent);

extern void
dfUserEventSignal(
  DfUserEvent* pUserEvent);
#endif

/*
** Deferred procedure calls
*/

struct _DfDpc {
  /* Client code must not touch */
  struct {
    struct tasklet_struct tasklet;
    DfDpcFunction* pCallback;
    void* pContext;
    void* pArg;
  } system;
};

extern void _dfLinuxDpcStub(unsigned long context);

static inline void
dfDpcInit(
  DfDpc* pDpc,
  DfInterruptObject* pInterruptObject, /* ignored */
  DfDpcFunction* pFunction,
  void* pContext)
{
  pDpc->system.pCallback = pFunction;
  pDpc->system.pContext = pContext;
  tasklet_init(&pDpc->system.tasklet, _dfLinuxDpcStub, (unsigned long)pDpc);
}

static inline void
dfDpcSchedule(
  DfDpc* pDpc,
  void* pArg)
{
  pDpc->system.pArg = pArg;
  tasklet_schedule(&pDpc->system.tasklet);
}

static inline void
dfDpcUninit(
  DfDpc* pDpc)
{
  /*
  ** NOTE: If this happens to be called just as a tasklet is scheduled, it may sleep
  ** until the tasklet function has run.
  */
  tasklet_kill(&pDpc->system.tasklet);
}

/*
** DMA support
**
** dfDmaMapperAllocate
**   - Declares to the system that this driver requires a DMA mapper.
**   - Must not be called twice for the same device object without an intervening call to
**     dfDmaMapperFree.
**
** dfDmaMapperFree
**   - Inverse of dfDmaMapperAllocate; undoes a call to dfDmaMapperAllocate.
**   - Must be called after dfDmaMapperAllocate.
**
** dfDmaBufferAllocate
**   - Allocates a small persistent buffer for holding data structures such as a DMA engine linked list.
**   - Must be called after dfDmaMapperAllocate & before dfDmaMapperFree.
**   - Requested block size must be <= g_dfDmaBufferMaxLength.
**
** dfDmaBufferFree
**   - Inverse of dfDmaBufferAllocate; undoes a call to dfDmaBufferAllocate.
**   - Must be called after dfDmaBufferAllocate & before dfDmaMapperFree.
**
** dfDmaBufferSync
**   - For a bus master reading memory, flushes data out of CPU write buffers & caches for specified persistent buffer.
**   - For a bus master writing memory, invalidates CPU caches for specified persistent buffer.
**   - Must be called after dfDmaBufferAllocate & before dfDmaMapperFree.
**
** dfDmaMapperSetup
** dfDmaMapperSetupThread
**   - Gets a scatter-gather table for a specified buffer description.
**   - Must be called after dfDmaMapperAllocate & before dfDmaMapperFree.
**   - Calls a client-supplied callback function when the scatter-gather tables is ready; may cause
**     resources such as bounce-buffers or hardware scatter-gather table entries to be allocated.
**   - dfDmaMapperSetupThread is the version that must be called when executing in the context of a
**     thread, as opposed to DPC context.
**
** dfDmaMapperTeardown
**   - Inverse of dfDmaMapperSetup; undoes a call to dfDmaMapperSetup.
**   - Must be called after dfDmaMapperSetup & before dfDmaMapperFree.
**   - Frees any resources such as bounce-buffers or hardware scatter-gather table entries.
*/

#define g_dfDmaBufferMaxLength PAGE_SIZE

struct _DfDmaMapperPosition {
  size_t sliceIndex;
  size_t slicePosition;
};

struct _DfDmaMapperSlice {
  size_t offset;
  size_t length;
  uint64_t localAddress;
};

struct _DfDmaMapperNode {
  uint64_t busAddress;
  uint64_t localAddress;
  size_t length;
};

struct _DfDmaBuffer {
  uint64_t busAddress;
  void* pKernelAddress;
  size_t size;

  struct { /* Private */
    DfDeviceObject* pDevObj;
    struct {
      uint64_t busAddress;
      void* pKernelAddress;
      size_t size;
    } unaligned;
  } system;
};

/* Must be called after dfDmaMapperAllocate */
extern DfDmaBuffer*
dfDmaBufferAllocate(
  DfDeviceObject* pDevObj,
  size_t size,
  size_t alignment);

/* Must be called after dfDmaBufferAllocate */
extern void
dfDmaBufferFree(
  DfDmaBuffer* pDmaBuffer);

extern boolean_t
dfDmaMapperAllocate(
  DfDeviceObject* pDevObj,
  unsigned int numberOfMaps,
  unsigned int mapLength,          /* In blocks */
  boolean_t b64BitAddressing,
  size_t maxMapSize,
  unsigned int* pActualMapLength); /* In blocks */

/* Must be called after dfDmaMapperAllocate */
extern void
dfDmaMapperFree(
  DfDeviceObject* pDevObj);

/* Must be called after dfDmaBufferAllocate */
DF_DECLARE_INLINE_FUNC(boolean_t, dfDmaBufferSync)(
  DfDmaBuffer* pDmaBuffer,
  boolean_t bDmaWrite, /* TRUE => buffer was written by bus master; FALSE => buffer will be read by bus master */
  size_t offset,
  size_t count)
{
  if (offset + count < offset ||         /* Catch wrapping */
      offset + count > pDmaBuffer->size) /* Overruns buffer */
  {
    return FALSE; /* Invalid region */
  }
  smp_wmb();
  return TRUE; /* OK */
}

/* Must be called after dfDmaMapperAllocate */
extern boolean_t
dfDmaMapperSetup(
  DfDeviceObject* pDevObj,
  unsigned int mapIndex,
  DfBufferDescription* pBufferDescription,
  DfDmaMapperCallback* pCallback,
  void* pCallbackContext,
  boolean_t bWriteToDevice,
  boolean_t bFixedLocalAddress,
  DfDmaMapperSlice* pSlices,
  size_t numSlice,
  DfDmaMapperPosition* pStartPosition);

/* Must be called after dfDmaMapperAllocate */
#define dfDmaMapperSetupThread dfDmaMapperSetup

/* Must be called after dfDmaMapperSetup */
extern void
dfDmaMapperTeardown(
  DfDeviceObject* pDevObj,
  unsigned int mapIndex);

/*
** Atomic integer operations
*/

DF_DECLARE_INLINE_FUNC(void, dfAtomicSet)(
  DfAtomicInt* pValue,
  int n)
{
  atomic_set(pValue, n);
  smp_mb();
}

DF_DECLARE_INLINE_FUNC(int, dfAtomicGet)(
  DfAtomicInt* pValue)
{
  smp_mb();
  return atomic_read(pValue);
}

#define dfAtomicDec(pValue)    atomic_dec_return(pValue)
#define dfAtomicInc(pValue)    atomic_inc_return(pValue)

/*
** System sleeping disable / enable
**
** dfSystemSleepEnable
**   - Hints to the system that we don't want the system to sleep when idle.
**
** dfSystemSleepReenable - allows system to sleep again
**   - Hints to the system that it's OK again for system to sleep when idle.
*/

__inline static void*
dfSystemSleepDisable(
  DfDeviceObject* pDevObj /* ignored */)
{
  /* Currently does nothing in Linux */
  return (void*)(uintptr_t)1U;
}

__inline static void
dfSystemSleepReenable(
  DfDeviceObject* pDevObj, /* ignored */
  void* hDisableSleep)
{
  /* Currently does nothing in Linux */
  dfAssert(NULL != hDisableSleep);
}

/*
** Timestamping
*/

static inline uint64_t
dfTimestampGet(
  void)
{
  struct timespec ts;
  ktime_get_ts(&ts);
  return (uint64_t)ts.tv_sec * 1000000000 + (uint64_t)ts.tv_nsec;
}

static inline uint64_t
dfTimestampGetFrequency(
  void)
{
  return 1000000000;
}

struct _DfTimestampBuffer {
  DfSpinLock lock;
  struct {
    uint64_t frequency;
    uint64_t initTimestamp;
    DfTime initTime;
  } attributes;
  struct {
    void* pBuffer;
    uint32_t size;
  } buffer;
  struct {
    uint32_t tail;
    uint32_t head;
    boolean_t bOverflow;
  } state;
};

extern boolean_t
dfTimestampBufferInit(
  DfTimestampBuffer* pTSBuf,
  unsigned int size);

extern void
dfTimestampBufferUninit(
  DfTimestampBuffer* pTSBuf);

extern boolean_t
dfTimestampBufferReset(
  DfTimestampBuffer* pTSBuf);

extern boolean_t
dfTimestampBufferRead(
  DfTimestampBuffer* pTSBuf,
  uint32_t offset,
  uint32_t count,
  void* pBuffer,
  boolean_t bConsume);

static inline boolean_t
dfTimestampBufferEnabled(
  DfTimestampBuffer* pTSBuf)
{
  return (NULL == pTSBuf->buffer.pBuffer) ? FALSE : TRUE;
}

static inline boolean_t
dfTimestampBufferLog(
  DfTimestampBuffer* pTSBuf,
  uint64_t metadata,
  uint64_t timestamp)
{
  DfSpinLockFlags f;
  uint32_t size, head, tail, count;
  uint64_t* pBuffer;

  if (!dfTimestampBufferEnabled(pTSBuf)) {
    return FALSE;
  }

  size = pTSBuf->buffer.size;
  pBuffer = pTSBuf->buffer.pBuffer;

  f = dfSpinLockGet(&pTSBuf->lock);
  head = pTSBuf->state.head;
  tail = pTSBuf->state.tail;
  count = (tail >= head) ? tail - head : size + tail - head;
  if (count + 1 >= size) {
    pTSBuf->state.bOverflow = TRUE;
  } else {
    pBuffer += (tail << 1);
    tail++;
    if (tail == size) {
      tail = 0;
    }
    pBuffer[0] = metadata;
    pBuffer[1] = timestamp;
    pTSBuf->state.tail = tail;
  }
  dfSpinLockPut(&pTSBuf->lock, f);

  return TRUE;
}

#endif
