/*
** File: dfcommon.h  
** Project: Driver Framework
** Purpose: Platform-independent definitions
**
** (C) Copyright Alpha Data 2009-2010
*/

#if !defined(ADATA_DF_DFCOMMON_H)
#define ADATA_DF_DFCOMMON_H

#define DF_OFFSET_OF(STRUCT, MEMBER) \
  ((size_t)&((STRUCT*)0)->MEMBER)

#define DF_CONTAINER_OF(VARIABLE, STRUCT, MEMBER) \
  ((STRUCT*)((uint8_t*)(VARIABLE) - DF_OFFSET_OF(STRUCT, MEMBER)))

/* Aligns a pointer to specified alignment */
#define DF_ALIGN_POINTER(x, align) \
	(void*)(((uintptr_t)(x) + (align - 1U)) & ~(uintptr_t)(align - 1U))

/* Returns offset by which a pointer is misaligned to specified alignment */
#define DF_POINTER_ALIGNMENT(x, align) \
	(((uintptr_t)(x)) & (uintptr_t)((align) - 1U))

#define DF_ARRAY_LENGTH(x) (sizeof(x) / sizeof((x)[0]))

/*
** Bit field conventions to be used in drivers:
**
** xxx_L_yyy_zzz is a field length, in bits
** xxx_S_yyy_zzz is a field shift, in bits
** xxx_M_yyy_zzz is a field mask, shifted by field shift
** xxx_V_yyy_zzz is a field value, mask by unshifted field mask and then shifted by field shift
** xxx_V_yyy_zzz(x) is a field value macro; takes given value, masks it, and shifts it left by field shift
** xxx_E_yyy_zzz(x) is a field extraction macro field value; takes given value, shifts shifted by field shift, can be a macro
**
** where xxx identifies a set of registers
**       yyy identifies a register
**       zzz identifies a field in a register
**
** DF_MASKxU creates an unshifted field mask from field length
** DF_MASKxS creates a shifted field mask from field shift and field length
** DF_VALxS  creates a shifted value from field value, field shift and field length
** DF_EXxS   extracts a field value given register value, field shift and field length
*/

#define DF_MASK8U(length)   (((uint8_t)1U << (length)) - 1U)
#define DF_MASK16U(length) (((uint16_t)1U << (length)) - 1U)
#define DF_MASK32U(length) (((uint32_t)1U << (length)) - 1U)
#define DF_MASK64U(length) (((uint64_t)1U << (length)) - 1U)

#define DF_MASK8S(shift, length)   (DF_MASK8U(length) << (shift))
#define DF_MASK16S(shift, length) (DF_MASK16U(length) << (shift))
#define DF_MASK32S(shift, length) (DF_MASK32U(length) << (shift))
#define DF_MASK64S(shift, length) (DF_MASK64U(length) << (shift))

#define DF_VAL8S(val, shift, length)   (((uint8_t)(val) &  DF_MASK8U(length)) << (shift))
#define DF_VAL16S(val, shift, length) (((uint16_t)(val) & DF_MASK16U(length)) << (shift))
#define DF_VAL32S(val, shift, length) (((uint32_t)(val) & DF_MASK32U(length)) << (shift))
#define DF_VAL64S(val, shift, length) (((uint64_t)(val) & DF_MASK64U(length)) << (shift))

#define DF_EX8S(val, shift, length)   ((uint8_t)((val) >> (shift)) &  DF_MASK8U(length))
#define DF_EX16S(val, shift, length) ((uint16_t)((val) >> (shift)) & DF_MASK16U(length))
#define DF_EX32S(val, shift, length) ((uint32_t)((val) >> (shift)) & DF_MASK32U(length))
#define DF_EX64S(val, shift, length) ((uint64_t)((val) >> (shift)) & DF_MASK32U(length))

/* 64-bit long division (slow); not to be used in situations where execution time matters */
DF_DECLARE_INLINE_FUNC(uint64_t, dfDivide64By64)(
  uint64_t dividend,
  uint64_t divisor)
{
  uint64_t quotient = 0U, tmp = 0U;
  unsigned int i, shift;

  for (i = 0U; i < 64U; i++) {
    tmp <<= 1;
    shift = 63U - i;
    tmp |= (dividend >> shift) & 0x1U;
    if (tmp >= divisor) {
      quotient |= (uint64_t)0x1U << shift;
      tmp -= divisor;
    }
  }

  return quotient;
}

/* 64-bit long division with remainder (slow); not to be used in situations where execution time matters */
DF_DECLARE_INLINE_FUNC(uint64_t, dfDivide64By64Rem)(
  uint64_t dividend,
  uint64_t divisor,
  uint64_t* pRemainder)
{
  uint64_t quotient = 0U, tmp = 0U;
  unsigned int i, shift;

  for (i = 0U; i < 64U; i++) {
    tmp <<= 1;
    shift = 63U - i;
    tmp |= (dividend >> shift) & 0x1U;
    if (tmp >= divisor) {
      quotient |= (uint64_t)0x1U << shift;
      tmp -= divisor;
    }
  }

  *pRemainder = tmp;
  return quotient;
}

#endif
