/*
** File: v6clksynth.c  
** Project: ADB3 core driver
** Purpose: Common functions for Virtex-6 MMCM clock synthesizer
**
** (C) Copyright Alpha Data 2011
**
** NOTES
**
** 1. Clock word layout is:
**
**   d0[7:0]    D divider value (1..80)
**   d0[15:8]   M multiplier value (5..64)
**   d0[23:16]  O divider value (1..128)
**   d0[24]     Filter bandwidth: 0 => LOW, 1 => OPTIMIZED/HIGH
**   d0[31:25]  (unused)
**   d1[31:0]   LKTABLE[31:0]
**   d2[7:0]    LKTABLE[39:32]
**   d2[17:8]   TABLE[9:0]
**   d2[31:18]  (unused)
*/

#include <df.h>

#include "v6clksynth.h"
#include "v6clksynth_common.h"

/* Constants below taken from MMCM switching parameters in datasheet for V6LXT / V6SXT -1 speed grade */
static const uint8_t g_dMinMmcm = 1U;             /* Min D divider allowed for MMCM */
static const uint8_t g_dMaxMmcm = 80U;            /* Max D divider allowed for MMCM */
static const uint8_t g_mMinMmcm = 5U;             /* Min M multiplier allowed for MMCM */
static const uint8_t g_mMaxMmcm = 63U;            /* Max M multiplier allowed for MMCM */
static const uint8_t g_oMinMmcm = 1U;             /* Min O divider allowed for MMCM */
static const uint8_t g_oMaxMmcm = 128U;           /* Max O divider allowed for MMCM */
static const uint32_t g_fInMinMmcm = 10000000U;       /* fInMin = 10 MHz */
static const uint32_t g_fInMaxMmcm = 700000000U;      /* fInMax = 700 MHz */
static const uint32_t g_fVcoMinMmcm = 600000000U;     /* fVcoMin = 600 MHz */
static const uint32_t g_fVcoMaxMmcm = 1200000000U;    /* fVcoMax = 1.2 GHz */
static const uint32_t g_fPfdOptMinMmcm = 135000000U;  /* fPfdMin = 135 MHz (HIGH / OPTIMIZED bandwidth) */
static const uint32_t g_fPfdOptMaxMmcm = 450000000U;  /* fPfdMax = 450 MHz (HIGH / OPTIMIZED bandwidth) */
static const uint32_t g_fPfdLowMinMmcm = 10000000U;   /* fPfdMin = 10 MHz (LOW bandwidth) */
static const uint32_t g_fPfdLowMaxMmcm = 300000000U; /* fPfdMax = 300 MHz (LOW bandwidth) */
static const uint32_t g_fOutMinMmcm = 4999000U;   /* FoutMin = 5 MHz */
static const uint32_t g_fOutMaxMmcm = 700001000U; /* FoutMax = 700 MHz */

/* MMCM filter table for LOW bandwidth */
static const uint16_t g_filterLowTable[64] = {
  0x05F, 0x057, 0x07B, 0x05B, 0x06B, 0x073, 0x073, 0x073, /* 0 */
  0x073, 0x04B, 0x04B, 0x04B, 0x0B3, 0x053, 0x053, 0x053, /* 8 */
  0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x063, /* 16 */
  0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, /* 24 */
  0x063, 0x063, 0x063, 0x063, 0x063, 0x093, 0x093, 0x093, /* 32 */
  0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x0A3, /* 40 */
  0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, /* 48 */
  0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3  /* 56 */
};

/* MMCM filter table for HIGH/OPTIMIZED bandwidth */
static const uint16_t g_filterOptTable[64] = {
  0x17C, 0x3FC, 0x3F4, 0x3E4, 0x3F8, 0x3C4, 0x3C4, 0x3D8, /* 0 */
  0x3E8, 0x3E8, 0x3E8, 0x3B0, 0x3F0, 0x3F0, 0x3F0, 0x3F0, /* 8 */
  0x3F0, 0x3F0, 0x3F0, 0x3F0, 0x3B0, 0x3B0, 0x3B0, 0x3E8, /* 16 */
  0x370, 0x308, 0x370, 0x370, 0x3E8, 0x3E8, 0x3E8, 0x1C8, /* 24 */
  0x330, 0x330, 0x3A8, 0x188, 0x188, 0x188, 0x1F0, 0x188, /* 32 */
  0x110, 0x110, 0x110, 0x110, 0x110, 0x110, 0x0E0, 0x0E0, /* 40 */
  0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, /* 48 */
  0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0  /* 56 */
};

/* Table containing MMCM lock bits */
static const struct {
  uint32_t bits32To0;
  uint8_t  bits39To32;
} g_lockTable[64] = {
  { 0x868FA401, 0x31 }, { 0xBE8FA401, 0x31 }, { 0x3E8FA401, 0x42 }, { 0xFE8FA401, 0x5A }, /* 0 */
  { 0xBE8FA401, 0x73 }, { 0x7E8FA401, 0x8C }, { 0xFE8FA401, 0x9C }, { 0xBE8FA401, 0xB5 }, /* 4 */
  { 0x7E8FA401, 0xCE }, { 0x3E8FA401, 0xE7 }, { 0xF84FA401, 0xFF }, { 0xF39FA401, 0xFF }, /* 8 */
  { 0xEEEFA401, 0xFF }, { 0xEBCFA401, 0xFF }, { 0xE8AFA401, 0xFF }, { 0xE71FA401, 0xFF }, /* 12 */
  { 0xE3FFA401, 0xFF }, { 0xE26FA401, 0xFF }, { 0xE0DFA401, 0xFF }, { 0xDF4FA401, 0xFF }, /* 16 */
  { 0xDDBFA401, 0xFF }, { 0xDC2FA401, 0xFF }, { 0xDA9FA401, 0xFF }, { 0xD90FA401, 0xFF }, /* 20 */
  { 0xD90FA401, 0xFF }, { 0xD77FA401, 0xFF }, { 0xD5EFA401, 0xFF }, { 0xD5EFA401, 0xFF }, /* 24 */
  { 0xD45FA401, 0xFF }, { 0xD45FA401, 0xFF }, { 0xD2CFA401, 0xFF }, { 0xD2CFA401, 0xFF }, /* 28 */
  { 0xD2CFA401, 0xFF }, { 0xD13FA401, 0xFF }, { 0xD13FA401, 0xFF }, { 0xD13FA401, 0xFF }, /* 32 */
  { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, /* 36 */
  { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, /* 40 */
  { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, /* 44 */
  { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, /* 48 */
  { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, /* 52 */
  { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, /* 56 */
  { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }, { 0xCFAFA401, 0xFF }  /* 60 */
};

V6ClkSynthClockWordStatus
v6ClkSynthClockWord(
  uint32_t flags,
	uint32_t fRef,
	uint64_t fOutReq,
	uint64_t fOutMin,
	uint64_t fOutMax,
	CoreClockWord* pClockWord)
{
  uint32_t fPfdMin, fPfdMax;
  uint32_t dMin, dMax;
  uint32_t mMin, mMax;
  uint32_t oMin, oMax;
  uint32_t o, m, d, bandwidth;
  uint32_t oBest = 0, mBest = 0, dBest = 0, bandwidthBest = 0, fOutBest = 0; /* init. to avoid compiler whine */
  uint32_t mBestMod64;
	uint32_t fPfd, fVco, fOut;
	uint32_t fError, fErrorBest = (uint32_t)-1;
  boolean_t bOk;

	dfDebugPrint(5, ("v6ClkSynthClockWord: fRef=%lu fOutReq=" DF_FMT_U64 "\n",
    (unsigned long)fRef, (unsigned long long)fOutReq));

  if (fRef < g_fInMinMmcm || fRef > g_fInMaxMmcm) {
		return V6ClkSynthClockWordStatusGeneralFailure;
	}
  if (fOutReq < g_fOutMinMmcm || fOutReq > g_fOutMaxMmcm) {
		return V6ClkSynthClockWordStatusOutOfRange;
	}

  /* Iterate over low bandwidth (0) and high/optimized bandwidth (1) */
  for (bandwidth = 0; bandwidth <= 1; bandwidth++) {
    /* fPfd constraints depend upon bandwidth */
    fPfdMin = bandwidth ? g_fPfdOptMinMmcm : g_fPfdLowMinMmcm;
    fPfdMax = bandwidth ? g_fPfdOptMaxMmcm : g_fPfdLowMaxMmcm;
    /* min/max D divider values depend on fPfd constraints */
    dMin = (fRef + fPfdMax - 1) / fPfdMax;  /* Round up minimum D */
    dMax = fRef / fPfdMin;                  /* Round down maximum D */
    /* Ensure that dMin and dMax stay within D divider constraints */
    if (dMin < g_dMinMmcm) {
      dMin = g_dMinMmcm;
    }
    if (dMax > g_dMaxMmcm) {
      dMax = g_dMaxMmcm;
    }
    /* Iterate over allowed D divider values */
    for (d = dMin; d <= dMax; d++) {
      fPfd = fRef / d;
      /* Find allowed M multiplier values for fPfd based on fVco constraints */
      mMin = (g_fVcoMinMmcm + fPfd - 1) / fPfd; /* Round up for minimum M */
      mMax = g_fVcoMaxMmcm / fPfd;              /* Round down for maximum M */
      /* Ensure that mMin and mMax stay within M multiplier constraints */
      if (mMin < g_mMinMmcm) {
        mMin = g_mMinMmcm;
      }
      if (mMax > g_mMaxMmcm) {
        mMax = g_mMaxMmcm;
      }
      /* Iterate over allowed M multiplier values */
      for (m = mMin; m <= mMax; m++) {
        fVco = fPfd * m;
        /* Find allowed O divider values based on fOut and O divider constraints */
        oMin = (fVco + g_fOutMaxMmcm - 1) / g_fOutMaxMmcm; /* Round up for minimum O */
        /* Ensure that oMin and oMax stay within O divider constraints */
        if (oMin < g_oMinMmcm) {
          oMin = g_oMinMmcm;
        }
        oMax = g_oMaxMmcm;
        /* Find O divider value for output frequency closest to frequencyReq, rounding down */
        o = fVco / (uint32_t)fOutReq;
        if (o >= oMin && o <= oMax) {
          /* Find output frequency */
          fOut = fVco / o;
          bOk = TRUE;
          if ((flags & CoreClockMinimumFrequency) && fOut < fOutMin) {
            /* Reject this set of parameters due to frequency being below minimum requested */
            bOk = FALSE;
          }
          if ((flags & CoreClockMaximumFrequency) && fOut > fOutMax) {
            /* Reject this set of parameters due to frequency being above maximum requested */
            bOk = FALSE;
          }
          if (bOk) {
            /* Compare output frequency with best so far */
            fError = ((uint32_t)fOutReq > fOut) ? ((uint32_t)fOutReq - fOut) : (fOut - (uint32_t)fOutReq);
            if (fErrorBest == (uint32_t)-1 || fError < fErrorBest) {
              /* This is the best set of parameters found so far */
              bandwidthBest = bandwidth;
              dBest = d;
              mBest = m;
              oBest = o;
              fOutBest = fOut;
              fErrorBest = fError;
            }
          }
        }
        /* Find O divider value for output frequency closest to frequencyReq, rounding up */
        o = (fVco + (uint32_t)fOutReq - 1) / (uint32_t)fOutReq;
        if (o >= oMin && o <= oMax) {
          /* Find output frequency */
          fOut = fVco / o;
          bOk = TRUE;
          if ((flags & CoreClockMinimumFrequency) && fOut < fOutMin) {
            /* Reject this set of parameters due to frequency being below minimum requested */
            bOk = FALSE;
          }
          if ((flags & CoreClockMaximumFrequency) && fOut > fOutMax) {
            /* Reject this set of parameters due to frequency being above maximum requested */
            bOk = FALSE;
          }
          if (bOk) {
            /* Compare output frequency with best so far */
            fError = ((uint32_t)fOutReq > fOut) ? ((uint32_t)fOutReq - fOut) : (fOut - (uint32_t)fOutReq);
            if (fErrorBest == (uint32_t)-1 || fError < fErrorBest) {
              /* This is the best set of parameters found so far */
              bandwidthBest = bandwidth;
              dBest = d;
              mBest = m;
              oBest = o;
              fOutBest = fOut;
              fErrorBest = fError;
            }
          }
        }
      }
    }
  }

	if (fErrorBest == (uint32_t)-1) {
		/* Failed to find a suitable set of parameters */
		return V6ClkSynthClockWordStatusOutOfRange;
	}

  pClockWord->frequency = fOutBest;
  pClockWord->d0 = (dBest << 0) | (mBest << 8) | (oBest << 16) | (bandwidthBest << 24);
  mBestMod64 = mBest & 0x3FU;
  pClockWord->d1 = g_lockTable[mBestMod64].bits32To0;
  pClockWord->d2 = g_lockTable[mBestMod64].bits39To32;
  pClockWord->d2 |= (bandwidthBest ? g_filterOptTable[mBestMod64] : g_filterLowTable[mBestMod64]) << 8;

	dfDebugPrint(5, ("v6ClkSynthClockWord: fOut=%lu bw=%s d=%u m=%u o=%u lock=0x%02lx_%08lx filter=0x%03lx\n",
    (unsigned long)fOutBest, bandwidthBest ? "optimized" : "low", (unsigned int)dBest, (unsigned int)mBest, (unsigned int)oBest,
    (unsigned long)(pClockWord->d2 & 0xFFU), (unsigned long)pClockWord->d1, (unsigned long)((pClockWord->d2 >> 8) & 0x3FFU)));

	return V6ClkSynthClockWordStatusSuccess;
}

static void
v6mmcmGetDivRegValue(
  uint8_t d,
  uint16_t* pDivRegVal)
{
  uint16_t val16;

  if (d == 1) {
    /* D = 1 => set NO COUNT bit */
    val16 = 0x1000U;
  } else {
    val16 = (uint16_t)(((d + 1) >> 1) & 0x3FU);                   /* LOW TIME */
    val16 = (uint16_t)(val16 | (((d >> 1) << 6) & 0xFC0U));       /* HIGH TIME */
    val16 = (uint16_t)(val16 | ((d & 0x1U) ? (0x1U << 13) : 0U)); /* Set EDGE bit for odd division factor */
  }
  *pDivRegVal = val16;
}

static void
v6mmcmGetClkRegValues(
  uint8_t mo,
  uint16_t* pClkReg1Val,
  uint16_t* pClkReg2Val)
{
  uint16_t v1, v2;

  if (mo == 1) {
    /* M or O = 1 => set NO COUNT bit */
    v1 = 0U;
    v2 = 0x0040U;
  } else {
    v1 = (uint16_t)(((mo + 1) >> 1) & 0x3FU);          /* LOW TIME */
    v1 = (uint16_t)(v1 | (((mo >> 1) << 6) & 0xFC0U)); /* HIGH TIME */
    v2 = (uint16_t)((mo & 0x1U) ? 0x0080U : 0U);       /* Set EDGE bit for odd division factor */
  }
  *pClkReg1Val = v1;
  *pClkReg2Val = v2;
}

static void
v6mmcmGetLockRegValues(
  uint32_t lktableLo,
  uint32_t lktableHi,
  uint16_t* pLockReg1Val,
  uint16_t* pLockReg2Val,
  uint16_t* pLockReg3Val)
{
  *pLockReg1Val = (uint16_t)((lktableLo >> 20) & 0x3FFU);
  *pLockReg2Val = (uint16_t)(((lktableLo >> 0) & 0x3FFU) | (((lktableLo >> 30) & 0x3U) << 10) | ((lktableHi & 0x7U) << 12));
  *pLockReg3Val = (uint16_t)(((lktableLo >> 10) & 0x3FFU) | (((lktableHi >> 3) & 0x1FU) << 10));
}

static void
v6mmcmGetFiltRegValues(
  uint16_t table,
  uint16_t* pFiltReg1Val,
  uint16_t* pFiltReg2Val)
{
  uint32_t tmp;

  tmp = (((table >> 6) & 0x1U) << 8) | (((table >> 7) & 0x3U) << 11) | (((table >> 9) & 0x1U) << 15);
  *pFiltReg1Val = (uint16_t)tmp;
  tmp = (((table >> 0) & 0x1U) << 4) | (((table >> 1) & 0x3U) << 7) | (((table >> 3) & 0x3U) << 11) | (((table >> 5) & 0x1U) << 15);
  *pFiltReg2Val = (uint16_t)tmp; 
}

static boolean_t /* bOk */
v6mmcmDrpRead(
  Adb3CoreDeviceContext* pDevCtx,
  uint8_t lclkSel,
  uint32_t lclkCtl,
  uint8_t drpAddress,
  uint16_t* pVal16)
{
  static const unsigned int countMax = 2;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int count;
  uint32_t val32 = 0U;

  hModel = pDevCtx->hardware.hModel;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;

  /* Poll a couple of times for MMCM DRP port not busy */
  for (count = 0; count < countMax; count++) {
    if (lclkSel) {
      val32 = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkMmcmStat1);
    } else {
      val32 = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkMmcmStat0);
    }
    if (!(val32 & 0x10000U)) {
      /* MMCM not busy */
      break;
    }
  }
  if (count == countMax) {
    /* Timed out */
    dfDebugPrint(0, ("*** v6mmcmDrpWrite: MMCM %u not ready, lclkMmcmStat=0x%08lx\n", lclkSel, (unsigned long)val32));
    return FALSE;
  }

  /* Send read command to MMCM DRP interface */
  val32 = lclkCtl & 0xF8000000U;
  val32 |= ((uint32_t)drpAddress << 16) | 0x01000000U | ((uint32_t)lclkSel << 26);
  dfPciMemWrite32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl, val32);

  /* Poll a couple of times for MMCM DRP port not busy */
  for (count = 0; count < countMax; count++) {
    if (lclkSel) {
      val32 = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkMmcmStat1);
    } else {
      val32 = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkMmcmStat0);
    }
    if (!(val32 & 0x10000U)) {
      /* MMCM not busy */
      break;
    }
  }
  if (count == countMax) {
    /* Timed out */
    dfDebugPrint(0, ("*** v6mmcmDrpWrite: MMCM %u not ready, lclkMmcmStat=0x%08lx\n", lclkSel, (unsigned long)val32));
    return FALSE;
  }

  dfDebugPrint(6, ("v6mmcmDrpRead: lclkSel=0x%x lclkCtl=0x%lx drpAddress=0x%02x data=0x%04x\n",
    lclkSel, lclkCtl, drpAddress, (val32 & 0xFFFFU)));

  *pVal16 = (uint16_t)val32;
  return TRUE; /* Success */
}

static boolean_t /* bOk */
v6mmcmDrpWrite(
  Adb3CoreDeviceContext* pDevCtx,
  uint8_t lclkSel,
  uint32_t lclkCtl,
  uint8_t drpAddress,
  uint16_t mask,
  uint16_t value)
{
  static const unsigned int countMax = 2;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int count;
  uint16_t tmp16;
  uint32_t val32 = 0U;

  dfDebugPrint(6, ("v6mmcmDrpWrite: lclkSel=0x%x lclkCtl=0x%lx drpAddress=0x%02x mask=0x%04x data=0x%04x\n",
    lclkSel, lclkCtl, drpAddress, mask, value));

  hModel = pDevCtx->hardware.hModel;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;

  /* Do we need a read-modify-write cycle? */
  if (mask != 0xFFFFU) {
    /* Yes, do read part of read-modify write cycle */
    if (!v6mmcmDrpRead(pDevCtx, lclkSel, lclkCtl, drpAddress, &tmp16)) {
      return FALSE;
    }
    value = (uint16_t)(value & mask);
    value = (uint16_t)(value | (tmp16 & ~mask));
  }

  /* Poll a couple of times for MMCM DRP port not busy */
  for (count = 0; count < countMax; count++) {
    if (lclkSel) {
      val32 = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkMmcmStat1);
    } else {
      val32 = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkMmcmStat0);
    }
    if (!(val32 & 0x10000U)) {
      /* MMCM not busy */
      break;
    }
  }
  if (count == countMax) {
    /* Timed out */
    dfDebugPrint(0, ("*** v6mmcmDrpWrite: MMCM %u not ready, lclkMmcmStat=0x%08lx\n", lclkSel, (unsigned long)val32));
    return FALSE;
  }

  /* Send write command to MMCM DRP interface */
  val32 = lclkCtl & 0xF8000000U;
  val32 |= (uint32_t)value | ((uint32_t)drpAddress << 16) | 0x03000000U | ((uint32_t)lclkSel << 26);
  dfPciMemWrite32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl, val32);
  dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);

  return TRUE; /* Success */
}

boolean_t
v6ClkSynthInit(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx,
  unsigned int refFreq,
  uint32_t* pFOut)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  boolean_t bOk = TRUE;
  uint32_t lclkCtl;
  uint8_t lclkSel;
  uint16_t val16a, val16b;
  unsigned int d, m, mMax, o;
  uint32_t fPfd, fVco, fOut;

  hModel = pDevCtx->hardware.hModel;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;

  dfTimerInit(&pV6ClkSynthCtx->timer, pDevCtx);
  pV6ClkSynthCtx->lclkCtlShadow = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);
  pV6ClkSynthCtx->save.bLastValid = FALSE;

  /* Figure out the current frequency output by the synthesizer by reading the MMCM registers via DRP */
  lclkCtl = pV6ClkSynthCtx->lclkCtlShadow;
  lclkSel = (uint8_t)((lclkCtl >> 27) & 0x1U);
  /* Get DIVCLK value */
  bOk &= v6mmcmDrpRead(pDevCtx, lclkSel, lclkCtl, 0x16U, &val16a); /* D register, DRP address 0x16 */
  if (val16a & 0x1000U) {
    d = 1; /* NO COUNT bit is set => D = 1 */
  } else {
    d = (val16a & 0x3FU) + ((val16a >> 6) & 0x3FU);
  }
  bOk &= v6mmcmDrpRead(pDevCtx, lclkSel, lclkCtl, 0x14U, &val16a); /* M register, DRP address 0x14 */
  bOk &= v6mmcmDrpRead(pDevCtx, lclkSel, lclkCtl, 0x15U, &val16b); /* M register, DRP address 0x15 */
  if (val16b & 0x0040U) {
    m = 1; /* NO COUNT bit is set => M = 1 */
  } else {
    m = (val16a & 0x3FU) + ((val16a >> 6) & 0x3FU);
  }
  bOk &= v6mmcmDrpRead(pDevCtx, lclkSel, lclkCtl, 0x8U, &val16a); /* O register, DRP address 0x8 */
  bOk &= v6mmcmDrpRead(pDevCtx, lclkSel, lclkCtl, 0x9U, &val16b); /* O register, DRP address 0x9 */
  if (val16b & 0x0040U) {
    o = 1; /* NO COUNT bit is set => M = 1 */
  } else {
    o = (val16a & 0x3FU) + ((val16a >> 6) & 0x3FU);
  }
  if (!bOk) {
    dfDebugPrint(0, ("*** v6ClkSynthInit: failed to read parameters from MMCM\n"));
    return FALSE;
  }
  if (o == 0) {
    dfDebugPrint(0, ("*** v6ClkSynthInit: O value from MMCM is invalid (%u)\n", o));
    return FALSE;
  }
  if (d == 0) {
    dfDebugPrint(0, ("*** v6ClkSynthInit: D value from MMCM is invalid (%u)\n", d));
    return FALSE;
  }
  dfDebugPrint(2, ("v6ClkSynthInit: refFreq=%u d=%u m=%u o=%u\n", refFreq, d, m, o));
  fPfd = refFreq / d;
  if ((fPfd < (g_fPfdLowMinMmcm - 1000U) || fPfd > (g_fPfdLowMaxMmcm + 1000U)) &&
      (fPfd < (g_fPfdOptMinMmcm - 1000U) || fPfd > (g_fPfdOptMaxMmcm + 1000U))) {
    dfDebugPrint(0, ("*** v6ClkSynthInit: fPfd of %lu Hz is out of range\n", (unsigned long)fPfd));
    return FALSE;
  }
  dfDebugPrint(2, ("v6ClkSynthInit: fPfd=%lu Hz \n", (unsigned long)fPfd));
  mMax = (unsigned int)-1 / fPfd;
  if (m > mMax) {
    dfDebugPrint(0, ("*** v6ClkSynthInit: M value from MMCM is invalid (%u)\n", m));
    return FALSE;
  }
  fVco = fPfd * m;
  if (fVco < (g_fVcoMinMmcm - 1000U) || fVco > (g_fVcoMaxMmcm + 1000U)) {
    dfDebugPrint(0, ("*** v6ClkSynthInit: fVco of %lu Hz is out of range\n", (unsigned long)fVco));
    return TRUE;
  }
  dfDebugPrint(2, ("v6ClkSynthInit: fVco=%lu Hz\n", (unsigned long)fVco));
  fOut = fVco / o;
  if (fOut < (g_fOutMinMmcm - 1000U) || fOut > (g_fOutMaxMmcm + 1000U)) {
    dfDebugPrint(0, ("*** v6ClkSynthInit: fOut of %lu Hz is out of range\n", (unsigned long)fOut));
    return FALSE;
  }
  dfDebugPrint(1, ("v6ClkSynthInit: fOut=%lu Hz\n", (unsigned long)fOut));
  *pFOut = fOut;

  return TRUE; /* Success */
}

void
v6ClkSynthUninit(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx)
{
  dfTimerUninit(&pV6ClkSynthCtx->timer);
}

static void
onCheckV6ClkSynth(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  V6ClkSynthProgramRequest* pRequest = (V6ClkSynthProgramRequest*)pCallbackContext;
  V6ClkSynthDeviceContext* pV6ClkSynthCtx = pRequest->pV6ClkSynthCtx;
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t lclkCtl = pV6ClkSynthCtx->lclkCtlShadow;
  uint8_t lclkSel = (uint8_t)((lclkCtl >> 27) & 0x1U); /* indicates which MMCM's clock is output by the synthesizer */
	uint32_t lclkStat;
  V6ClkSynthProgramStatus status = V6ClkSynthProgramStatusSuccess;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Check that the unselected MMCM is now locked */
  lclkStat = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkStat);
  lclkStat &= (lclkSel ? 0x1U : 0x2U);
  if (!lclkStat) {
    dfDebugPrint(0, ("*** onCheckV6ClkSynth: MMCM not locked, lclkStat=0x%08lx\n", (unsigned long)lclkStat));
    status = V6ClkSynthProgramStatusHardwareError;
  } else {
    /* Swap over the unselected and selected MMCMs */
    lclkCtl ^= 0x1U << 27;
    lclkSel ^= 0x1U;
    dfPciMemWrite32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl, lclkCtl);
    dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);
    /* Power down the now unselected MMCM */
    lclkCtl |= (lclkSel ? 0x1U << 30 : 0x1U << 31);
    dfPciMemWrite32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl, lclkCtl);
    dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);
    /* Update the shadow of the lclkCtl register */
    pV6ClkSynthCtx->lclkCtlShadow = lclkCtl;
    /* Remember what has just been programmed, so we can save & restore store for power management */
    pV6ClkSynthCtx->save.bLastValid = TRUE;
    pV6ClkSynthCtx->save.last = pRequest->clockWord;
  }

  pRequest->pCallback(pDevCtx, pV6ClkSynthCtx, pRequest, status);
}

static void
onMmcmPoweredUp(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  V6ClkSynthProgramRequest* pRequest = (V6ClkSynthProgramRequest*)pCallbackContext;
  V6ClkSynthDeviceContext* pV6ClkSynthCtx = pRequest->pV6ClkSynthCtx;
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t lclkCtl = pV6ClkSynthCtx->lclkCtlShadow;
  uint8_t lclkSel = (uint8_t)((lclkCtl >> 27) & 0x1U); /* indicates which MMCM's clock is output by the synthesizer */
  uint16_t val16a, val16b, val16c;
  boolean_t bOk = TRUE;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* We're going to do everything below to the currently unselected MMCM */
  lclkSel ^= 0x1U;

  /* Power-up the unselected MMCM as per XAPP878, DRP address 0x28 */
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x28U, 0xFFFFU, 0xFFFFU);

  /* Put the unselected MMCM into reset */
  lclkCtl |= lclkSel ? (0x1U << 29) : (0x1U << 28);
  dfPciMemWrite32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl, lclkCtl);
  dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);

  /* Update the unselected MMCM's parameters via its DRP interface */
  v6mmcmGetDivRegValue((uint8_t)pRequest->clockWord.d0, &val16a);
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x16U, 0x3FFFU, val16a); /* D register, DRP address 0x16 */
  v6mmcmGetClkRegValues((uint8_t)(pRequest->clockWord.d0 >> 8), &val16a, &val16b);
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x14U, 0xEFFFU, val16a); /* M register (CLKFBOUT), DRP address 0x14 */
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x15U, 0x03FFU, val16b); /* M register (CLKFBOUT), DRP address 0x15 */
  v6mmcmGetClkRegValues((uint8_t)(pRequest->clockWord.d0 >> 16), &val16a, &val16b);
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x8U, 0xEFFFU, val16a); /* O register (CLKOUT0), DRP address 0x8 */
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x9U, 0x03FFU, val16b); /* O register (CLKOUT0), DRP address 0x9 */
  v6mmcmGetLockRegValues(pRequest->clockWord.d1, pRequest->clockWord.d2 & 0xFFU, &val16a, &val16b, &val16c);
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x18U, 0x03FFU, val16a); /* LockReg1, DRP address 0x18 */
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x19U, 0x7FFFU, val16b); /* LockReg2, DRP address 0x19 */
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x1AU, 0x7FFFU, val16c); /* LockReg3, DRP address 0x1A */
  v6mmcmGetFiltRegValues((uint16_t)((pRequest->clockWord.d2 >> 8) & 0x3FFU), &val16a, &val16b);
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x4EU, 0x9900U, val16a); /* FiltReg1, DRP address 0x4E */
  bOk &= v6mmcmDrpWrite(pDevCtx, lclkSel, lclkCtl, 0x4FU, 0x9981U, val16b); /* FiltReg2, DRP address 0x4F */

  /* Remove the reset from the unselected MMCM, which allows it to begin locking */
  lclkCtl &= ~(lclkSel ? (0x1U << 29) : (0x1U << 28));
  dfPciMemWrite32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl, lclkCtl);
  dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);

  if (bOk) {
    dfTimerSchedule(&pV6ClkSynthCtx->timer, dfTimeGet() + dfMicrosecondsToTime(V6CLKSYNTH_MMCM_LOCK_DELAY), onCheckV6ClkSynth, (void*)pRequest);
  } else {
    pRequest->pCallback(pDevCtx, pV6ClkSynthCtx, pRequest, V6ClkSynthProgramStatusHardwareError);
  }
}

V6ClkSynthProgramStatus
v6ClkSynthProgram(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx,
  V6ClkSynthProgramRequest* pRequest,
  const CoreClockWord* pClockWord,
  V6ClkSynthProgramCallback* pCallback)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t lclkCtl = pV6ClkSynthCtx->lclkCtlShadow;
  uint8_t lclkSel = (uint8_t)((lclkCtl >> 27) & 0x1U); /* indicates which MMCM's clock is output by the synthesizer */

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  pRequest->pV6ClkSynthCtx = pV6ClkSynthCtx;
  pRequest->pCallback = pCallback;
  pRequest->clockWord = *pClockWord;

  /* We're going to do everything below to the currently unselected MMCM */
  lclkSel ^= 0x1U;

  /* Ensure the unselected MMCM's PWRDN input is deasserted */
  lclkCtl &= ~(lclkSel ? (0x1U << 31) : (0x1U << 30));
  dfPciMemWrite32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl, lclkCtl);
  dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);
  pV6ClkSynthCtx->lclkCtlShadow = lclkCtl;
  dfTimerSchedule(&pV6ClkSynthCtx->timer, dfTimeGet() + dfMicrosecondsToTime(V6CLKSYNTH_POWER_UP_MMCM_DELAY), onMmcmPoweredUp, (void*)pRequest);
  return V6ClkSynthProgramStatusSuccess;
}

static void
onProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx,
  V6ClkSynthProgramRequest* pRequest,
  V6ClkSynthProgramStatus status)
{
  V6ClkSynthProgramRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, V6ClkSynthProgramRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

V6ClkSynthProgramStatus
v6ClkSynthProgramSync(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx,
  const CoreClockWord* pClockWord)
{
  V6ClkSynthProgramRequestSync req;
  V6ClkSynthProgramStatus status;

  dfEventInit(&req.ev);
  status = v6ClkSynthProgram(pDevCtx, pV6ClkSynthCtx, &req.request, pClockWord, onProgramDone);
  if (V6ClkSynthProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

void
v6ClkSynthSaveState(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx)
{
  /* Nothing to do yet */
}

V6ClkSynthProgramStatus
v6ClkSynthRestoreState(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx)
{
  V6ClkSynthProgramStatus status = V6ClkSynthProgramStatusSuccess;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Reestablish LCLOCK register shadow */
  pV6ClkSynthCtx->lclkCtlShadow = dfPciMemRead32(hModel, &pModelRegs->model.v6ClkSynth.lclkCtl);

  /* If the clock synthesizer had been programmed before power-down, reprogram it with the same thing */
  if (pV6ClkSynthCtx->save.bLastValid) {
    status = v6ClkSynthProgramSync(pDevCtx, pV6ClkSynthCtx, &pV6ClkSynthCtx->save.last);
  }

  return status;
}

ClockProgramStatus
v6ClkSynthMapProgramStatus(
  V6ClkSynthProgramStatus status)
{
  switch (status) {
  case V6ClkSynthProgramStatusSuccess:
    return ClockProgramSuccess;

  case V6ClkSynthProgramStatusHardwareError:
    return ClockProgramHardwareError;

  case V6ClkSynthProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
v6ClkSynthMapClockWordStatus(
  V6ClkSynthClockWordStatus status)
{
  switch (status) {
  case V6ClkSynthClockWordStatusSuccess:
    return ClockWordSuccess;

  case V6ClkSynthClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case V6ClkSynthClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
