/*
** File: pci9xxx_93x6.c  
** Project: ADB3 core driver
** Purpose: Functions common to models that use the PCI9080 or PCI9656 PCI to
**          Local Bus Bridge.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "pci9xxx.h"
#include "pci9xxx_common.h"
#include "pci9xxx_93x6.h"

void
pci9xxxDumpDmaDescriptors(
  Adb3CoreDeviceContext* pDevCtx,
  Pci9xxxDmaDescriptor* pDescriptors,
  uint64_t descBus,
  unsigned int count)
{
  Pci9xxxDmaDescriptor* pCurrent;
  unsigned int i;

  if (sizeof(void*) == 4) {
    dfDebugPrint(0, ("  ### DescVirt   DescBus             [ PA       LA       SIZ      NEXT     ]\n"));
    for (i = 0; i < count; i++) {
      pCurrent = &pDescriptors[i];
      dfDebugPrint(0, ("  %03lu %p 0x%08lx_%08lx   %08lx %08lx %08lx %08lx\n",
        (unsigned long)i, (void*)pCurrent, dfSplitUint64(descBus),
        (unsigned long)dfLe32ToCpu(pCurrent->pci), (unsigned long)dfLe32ToCpu(pCurrent->local),
        (unsigned long)dfLe32ToCpu(pCurrent->count), (unsigned long)dfLe32ToCpu(pCurrent->link)));
      descBus += sizeof(Pci9xxxDmaDescriptor);
    }
  } else {
    dfDebugPrint(0, ("  ### DescVirt           DescBus             [ PA       LA       SIZ      NEXT     ]\n"));
    for (i = 0; i < count; i++) {
      pCurrent = &pDescriptors[i];
      dfDebugPrint(0, ("  %03lu %p 0x%08lx_%08lx   %08lx %08lx %08lx %08lx\n",
        (unsigned long)i, (void*)pCurrent, dfSplitUint64(descBus),
        (unsigned long)dfLe32ToCpu(pCurrent->pci), (unsigned long)dfLe32ToCpu(pCurrent->local),
        (unsigned long)dfLe32ToCpu(pCurrent->count), (unsigned long)dfLe32ToCpu(pCurrent->link)));
      descBus += sizeof(Pci9xxxDmaDescriptor);
    }
  }
}

void
pci9xxxDumpDmaRegisters(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channel)
{
  Pci9xxxDmaRegisters* pDmaRegs = &pDevCtx->hardware.bridge.pPci9656->DMA[channel];
  uint32_t mode, padr, ladr, size, dpr;
  uint16_t dmacsr;

  mode = pDmaRegs->MODE;
  padr = pDmaRegs->PADR;
  ladr = pDmaRegs->LADR;
  size = pDmaRegs->SIZE;
  dpr = pDmaRegs->DPR;
  dmacsr = pDevCtx->hardware.bridge.pPci9656->DMACSR;
  dfDebugPrint(0, ("  channel=%u MODE=%08lx PADR=%08lx LADR=%08lx SIZE=%08lx DPR=%08lx DMACSR=0x%04lx\n",
    channel, (unsigned long)mode, (unsigned long)padr, (unsigned long)ladr, (unsigned long)size, (unsigned long)dpr, (unsigned long)dmacsr));
}

void
pci9xxxDmaList(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWriteToDevice,
  boolean_t bFixedLocal,
  DfDmaMapperNode* pTable,
  unsigned int tableLength,
  DmaNodeTablePosition* pTablePosition,
  void* pDescriptorBuffer,
  unsigned int maxDescriptor,
  uint64_t descriptorBusAddress)
{
  Pci9xxxDmaDescriptor* pDescriptor = (Pci9xxxDmaDescriptor*)pDescriptorBuffer;
  DfDmaMapperNode* pTableTmp;
  uint64_t busAddress, nextBusAddress, tableBusAddress, localAddress = 0U;
  unsigned int i;
  uint32_t flags;
  unsigned int numDesc = 0;
  unsigned int index;
  size_t offset, remaining;

  dfDebugPrint(7, ("pci9xxxDmaList: entered, pDevCtx=%p bWriteToDevice=%s bFixedLocal=%s pTable=%p tableLength=%u *pTablePosition={%u, 0x%08lx_%08lx} pDescriptorBuffer=%p maxDescriptor=%u descriptorBusAddrss=0x%08lx_%08lx\n",
    (void*)pDevCtx, bWriteToDevice ? "TRUE" : "FALSE", bFixedLocal ? "TRUE" : "FALSE", (void*)pTable, tableLength, pTablePosition->index, dfSplitUint64((uint64_t)pTablePosition->offset), pDescriptorBuffer, maxDescriptor, dfSplitUint64(descriptorBusAddress)));

  flags = PCI9xxxDESCF_FETCH_PCI | (bWriteToDevice ? PCI9xxxDESCF_PCI_TO_LOCAL : PCI9xxxDESCF_LOCAL_TO_PCI);
  busAddress = descriptorBusAddress;
  index = pTablePosition->index;
  offset = pTablePosition->offset;
  pTableTmp = pTable + pTablePosition->index;
  for (i = index; i < tableLength && numDesc < maxDescriptor; i++, numDesc++) {
    nextBusAddress = busAddress + sizeof(Pci9xxxDmaDescriptor);
    tableBusAddress = pTableTmp->busAddress + offset;
    localAddress = pTableTmp->localAddress;
    if (!bFixedLocal) {
      localAddress += offset;
    }
    remaining = pTableTmp->length - offset;
    pDescriptor->pci = dfCpuToLe32((uint32_t)tableBusAddress);
    pDescriptor->local = dfCpuToLe32((uint32_t)localAddress);
    pDescriptor->count = dfCpuToLe32((uint32_t)remaining);
    pDescriptor->link = dfCpuToLe32((uint32_t)nextBusAddress | flags);
    pTableTmp++;
    busAddress = nextBusAddress;
    offset = 0;
    pDescriptor++;
  }
  pDescriptor--;
  pDescriptor->link = dfCpuToLe32(flags | PCI9xxxDESCF_ENDCHAIN);

  /* Cast is safe as pTableTmp cannot be much greater than pTable */
  pTablePosition->index = (unsigned int)(pTableTmp - pTable);
  pTablePosition->offset = offset;

#if DF_DBG_BUILD
  if (g_dfDebugLevel >= 7) {
    dfDebugPrint(0, ("pci9xxxDmaList: dump of DMA engine linked list\n"));
    pci9xxxDumpDmaDescriptors(pDevCtx, (Pci9xxxDmaDescriptor*)pDescriptorBuffer, descriptorBusAddress, numDesc);
  }
#endif
}

void
pci9xxxDmaTransfer(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channel,
  boolean_t bStart,
  void* pDescriptorBuffer,       /* Valid iff 'bStart' is TRUE */
  uint64_t descriptorBusAddress) /* Valid iff 'bStart' is TRUE */
{
  static const uint32_t specifiableModeBits =
    PCI9656_DMAMODE_USE_READY | PCI9656_DMAMODE_USE_BTERM | PCI9656_DMAMODE_ENABLE_BURST |
    PCI9656_DMAMODE_FIXED_LOCAL | PCI9656_DMAMODE_DEMAND | PCI9656_DMAMODE_WIDTHMASK;
  static const uint32_t alwaysModeBits =
    PCI9656_DMAMODE_CHAINED | PCI9656_DMAMODE_WRITE_INVALIDATE |
    PCI9656_DMAMODE_INTERRUPT_PCI | PCI9656_DMAMODE_DONE_INTERRUPT;
  const unsigned int maxAbortPoll = 20;
  Pci9656Registers* pRegs;
  Pci9xxxDmaRegisters* pDmaRegs;
  DfMemoryHandle hBridge;
  uint32_t mode;
  uint16_t dmacsr, mask, val16;
  unsigned int i;

  dfDebugPrint(7, ("pci9xxxDmaTransfer: entered, pDevCtx=%p channel=%lu bStart=%s pDescriptorBuffer=%p busAddress=0x%08lx_%08lx\n",
    (void*)pDevCtx, (unsigned long)channel, bStart ? "TRUE" : "FALSE", pDescriptorBuffer, dfSplitUint64(descriptorBusAddress)));

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;
  pDmaRegs = &pRegs->DMA[channel];

  if (bStart) {
    mode = (pDevCtx->dma.channel[channel].curr.parameters.user.modeFlags & specifiableModeBits) | alwaysModeBits;
    /* Set current transfer mode */
    dfPciMemWrite32(hBridge, &pDmaRegs->MODE, mode);
    /* Set current transfer size to 0 */
    dfPciMemWrite32(hBridge, &pDmaRegs->SIZE, 0);
    /* Set the descriptor pointer & flags */
    dfPciMemWrite32(hBridge, &pDmaRegs->DPR, (uint32_t)descriptorBusAddress | PCI9xxxDESCF_FETCH_PCI);
    dfPciMemRead32(hBridge, &pDmaRegs->DPR);
    
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("pci9xxxDmaTransfer: DMA channel %lu registers before starting on hardware:\n", (unsigned long)channel));
      pci9xxxDumpDmaRegisters(pDevCtx, channel);
    }
#endif

    /* Start the DMA transfer */
    mask = (uint16_t)(channel ? 0x0200U : 0002U);
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, (uint16_t)(mask | pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr));
    (void)dfPciMemRead16(hBridge, &pRegs->DMACSR); 

#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("pci9xxxDmaTransfer: DMA channel %lu registers after starting on hardware:\n", (unsigned long)channel));
      pci9xxxDumpDmaRegisters(pDevCtx, channel);
    }
#endif
  } else {
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("pci9xxxDmaTransfer: DMA channel %lu registers before attempt to abort:\n", (unsigned long)channel));
      pci9xxxDumpDmaRegisters(pDevCtx, channel);
    }
#endif

    mask = (uint16_t)(channel ? 0x0100U : 0001U);
    /* First, disable the DMA channel */
    dmacsr = (uint16_t)(pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr & ~mask);
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, dmacsr);
    /* Now check if DMA channel is active or not */
    val16 = dfPciMemRead16(hBridge, &pRegs->DMACSR);
    if (val16 & (mask << 4)) {
      /* DMA is not active so must have already finished */
      dfDebugPrint(1, ("pci9xxxDmaTransfer: nothing to abort, channel=%lu\n", (unsigned long)channel));
    } else {
      /* DMA is active - abort it */
      dfPciMemWrite16(hBridge, &pRegs->DMACSR, (uint16_t)(dmacsr | (mask << 2)));
      /* Poll for PCI9080/PCI9656 aborting the DMA */
      for (i = 0; i < maxAbortPoll; i++) {
        val16 = dfPciMemRead16(hBridge, &pRegs->DMACSR);
        if (val16 & (mask << 4)) {
          dfDebugPrint(2, ("pci9xxxDmaTransfer: aborted, channel=%lu\n", (unsigned long)channel));
          break;
        }
      }
      if (maxAbortPoll == i) {
        /* No! Not much we can do, though */
        dfDebugPrint(0, ("*** pci9xxxDmaTransfer: abort failed, channel=%lu\n", (unsigned long)channel));
      }
    }
    /* Reenable the DMA channel */
    dmacsr = (uint16_t)(dmacsr | mask);
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, dmacsr);
    
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 7) {
      dfDebugPrint(0, ("pci9xxxDmaTransfer: DMA channel %lu registers after attempt to abort:\n", (unsigned long)channel));
      pci9xxxDumpDmaRegisters(pDevCtx, channel);
    }
#endif
  }
}

void
pci9xxxEnableInterrupts(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bEnable)
{
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  uint32_t intcsr;
  uint16_t dmacsr;
  
  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;

  dfDebugPrint(1, ("pci9xxxEnableInterrupts: entered bEnable=%s\n", bEnable ? "TRUE" : "FALSE"));
  
  if (bEnable) {
    pDevCtx->hardware.shadow.bridge.pci9xxx.intcsr = pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaster;
    pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMask = pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaskMaster;
    dfPciMemWrite32(hBridge, &pRegs->INTCSR, pDevCtx->hardware.shadow.bridge.pci9xxx.intcsr);
    intcsr = dfPciMemRead32(hBridge, &pRegs->INTCSR);
    dfDebugPrint(2, ("pci9xxxEnableInterrupts: interrupts enabled, INTCSR=0x%08lx\n", (unsigned long)intcsr));
    
    pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr = 0x0101U;
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr);
    dmacsr = dfPciMemRead16(hBridge, &pRegs->DMACSR);
    dfDebugPrint(2, ("pci9xxxEnableInterrupts: DMA engines enabled, DMACSR=0x%04lx\n", (unsigned long)dmacsr));
  } else {
    pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr = 0U;
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr);
    dmacsr = dfPciMemRead16(hBridge, &pRegs->DMACSR);
    dfDebugPrint(2, ("pci9xxxEnableInterrupts: DMA engines disabled, DMACSR=0x%04lx\n", (unsigned long)dmacsr));

    pDevCtx->hardware.shadow.bridge.pci9xxx.intcsr = 0U;
    pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMask = 0U;
    dfPciMemWrite32(hBridge, &pRegs->INTCSR, pDevCtx->hardware.shadow.bridge.pci9xxx.intcsr);
    intcsr = dfPciMemRead32(hBridge, &pRegs->INTCSR);
    dfDebugPrint(2, ("pci9xxxEnableInterrupts: interrupts disabled, INTCSR=0x%08lx\n", (unsigned long)intcsr));
  }
}

CoreWindowConfigStatus
pci9xxxGetWindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  CoreWindowConfig* pConfig)
{
  const uint32_t prefDisBit = (windowIndex == 0) ? (0x1U << 8) : (0x1U << 9);
  const uint32_t burstEnBit = (windowIndex == 0) ? (0x1U << 24) : (0x1U << 8);
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  DfSpinLockFlags f;
  uint32_t lasxrr, lasxba, lbrdx, mask;

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;

  switch (windowIndex) {
  case 0:
  case 1:
    f = dfSpinLockGet(&pDevCtx->info.window[windowIndex].lock);
    lasxrr = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->LAS0RR : &pRegs->LAS1RR);
    lasxba = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->LAS0BA : &pRegs->LAS1BA);
    lbrdx = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->LBRD0 : &pRegs->LBRD1);
    dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);

    mask = lasxrr & 0xFFFFFFF0U;
    pConfig->base = lasxba & mask;

    switch (lbrdx & 0x3U) {
    case 0:
      pConfig->width = COREWINDOWCONFIG_WIDTH_8BIT;
      break;

    case 1:
      pConfig->width = COREWINDOWCONFIG_WIDTH_16BIT;
      break;

    case 3:
    case 2:
      pConfig->width = COREWINDOWCONFIG_WIDTH_32BIT;
      break;
    }

    if (lbrdx & prefDisBit) {
      /* Prefetch disabled */
      pConfig->prefetch = COREWINDOWCONFIG_PREFETCH_MIN;
    } else {
      if (lbrdx & (0x1U << 10)) {
        /* Prefetch limited by counter */
        pConfig->prefetch = COREWINDOWCONFIG_PREFETCH_NORMAL;
      } else {
        /* Prefetch not limited by counter (unlimited) */
        pConfig->prefetch = COREWINDOWCONFIG_PREFETCH_MAX;
      }
    }

    if (lbrdx & burstEnBit) {
      pConfig->burst = COREWINDOWCONFIG_BURST_ENABLE;
    } else {
      pConfig->burst = COREWINDOWCONFIG_BURST_DISABLE;
    }
    break;

  case 2:
    return CoreWindowConfigNotSupported;

  default:
    return CoreWindowConfigInvalidIndex;
  }

  return CoreWindowConfigSuccess;
}

CoreWindowConfigStatus
pci9xxxSetWindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  uint32_t flags,
  const CoreWindowConfig* pConfig)
{
  const uint32_t prefDisBit = (windowIndex == 0) ? (0x1U << 8) : (0x1U << 9);
  const uint32_t burstEnBit = (windowIndex == 0) ? (0x1U << 24) : (0x1U << 8);
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  DfSpinLockFlags f;
  uint32_t lasxrr, lasxba, lbrdx, mask;

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;

  switch (windowIndex) {
  case 0:
  case 1:
    f = dfSpinLockGet(&pDevCtx->info.window[windowIndex].lock);
    lasxrr = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->LAS0RR : &pRegs->LAS1RR);
    lasxba = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->LAS0BA : &pRegs->LAS1BA);
    lbrdx = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->LBRD0 : &pRegs->LBRD1);
    if (flags & COREWINDOWCONFIG_SET_BASE) {
      mask = lasxrr & 0xFFFFFFF0U;
      if (pConfig->base & ~(uint64_t)mask) {
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
      lasxba = (lasxba & ~mask) | ((uint32_t)pConfig->base & mask);
    }
    if (flags & COREWINDOWCONFIG_SET_WIDTH) {
      switch (pConfig->width) {
      case COREWINDOWCONFIG_WIDTH_8BIT:
        lbrdx = (lbrdx & ~0x3U) | (0x0U << 0);
        break;

      case COREWINDOWCONFIG_WIDTH_16BIT:
        lbrdx = (lbrdx & ~0x3U) | (0x1U << 0);
        break;

      case COREWINDOWCONFIG_WIDTH_DEFAULT:
      case COREWINDOWCONFIG_WIDTH_32BIT:
        lbrdx = (lbrdx & ~0x3U) | (0x2U << 0);
        break;

      default:
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
    }
    if (flags & COREWINDOWCONFIG_SET_PREFETCH) {
      switch (pConfig->prefetch) {
      case COREWINDOWCONFIG_PREFETCH_MIN:
        lbrdx = lbrdx | prefDisBit; /* Disable prefetch */
        break;

      case COREWINDOWCONFIG_PREFETCH_DEFAULT:
      case COREWINDOWCONFIG_PREFETCH_NORMAL:
        lbrdx = (lbrdx & ~((0x1U << 10) | prefDisBit)) | (0xFU << 11); /* 64-byte prefetch */
        break;

      case COREWINDOWCONFIG_PREFETCH_MAX:
        lbrdx = (lbrdx & ~prefDisBit) | (0x1U << 10); /* Unlimited prefetch */
        break;

      default:
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
    }
    if (flags & COREWINDOWCONFIG_SET_BURST) {
      switch (pConfig->prefetch) {
      case COREWINDOWCONFIG_BURST_DEFAULT:
      case COREWINDOWCONFIG_BURST_ENABLE:
        lbrdx = lbrdx | burstEnBit; /* Enable bursting */
        break;

      case COREWINDOWCONFIG_BURST_DISABLE:
        lbrdx = lbrdx & ~burstEnBit; /* Disable bursting */
        break;

      default:
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
    }
    /* Update registers only after we know requested values are valid */
    if (flags & COREWINDOWCONFIG_SET_BASE) {
      dfPciMemWrite32(hBridge, (windowIndex == 0) ? &pRegs->LAS0BA : &pRegs->LAS1BA, lasxba);
      pDevCtx->info.window[windowIndex].localBase = pConfig->base;
    }
    if (flags & (COREWINDOWCONFIG_SET_WIDTH | COREWINDOWCONFIG_SET_PREFETCH | COREWINDOWCONFIG_SET_BURST)) {
      dfPciMemWrite32(hBridge, (windowIndex == 0) ? &pRegs->LBRD0 : &pRegs->LBRD1, lbrdx);
    }
    dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
    break;

  case 2:
    return CoreWindowConfigNotSupported;

  default:
    return CoreWindowConfigInvalidIndex;
  }

  return CoreWindowConfigSuccess;
}

CoreWindowConfigStatus
pci9xxxWindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  boolean_t bConfigure,
  uint32_t flags,
  CoreWindowConfig* pConfig)
{
  if (bConfigure) {
    return pci9xxxSetWindowConfig(pDevCtx, windowIndex, flags, pConfig);
  } else {
    return pci9xxxGetWindowConfig(pDevCtx, windowIndex, pConfig);
  }
}

void
pci9xxxInitRegs(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bEnableDirectMaster)
{
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  uint32_t val32;

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;

  if (bEnableDirectMaster) {
    /* Enable Direct Master access */
    dfPciMemWrite32(hBridge, &pRegs->DMRR, 0xFF000000U);
    dfPciMemWrite32(hBridge, &pRegs->DMLBAM, 0xFF000000U);
    dfPciMemWrite32(hBridge, &pRegs->DMLBAI, 0x0U);
    dfPciMemWrite32(hBridge, &pRegs->DMPBAM, 0x1U);
  }

  /* Ensure use of PCI Read Multiple commands when a bus master, for better performance */
  val32 = dfPciMemRead32(hBridge, &pRegs->CNTRL);
  val32 &= ~0x0f0fU;
  val32 |= 0x0c0cU;
  dfPciMemWrite32(hBridge, &pRegs->CNTRL, val32);
}
